﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/os/os_ReaderWriterLockCommon.h>
#include <nn/os/os_ReaderWriterLockTypes.h>
#include <nn/os/os_ReaderWriterLockApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"


#define NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, name)                    \
        NN_SDK_REQUIRES((rwlock)->_state == (rwlock)->State_Initialized,    \
            NN_TEXT_OS("nn::os::%s(): 指定された rwlock オブジェクトが初期化されていません。"), name);

namespace nn { namespace os {

//---------------------------------------------------------------------------
//  ReaderWriterLockType オブジェクトの初期化
void InitializeReaderWriterLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    // 排他リソースを初期化
    new( &rwlock->_cs )                 detail::InternalCriticalSection;
    new( &rwlock->_cvReadLockWaiter  )  detail::InternalConditionVariable;
    new( &rwlock->_cvWriteLockWaiter )  detail::InternalConditionVariable;

    // メンバの初期化
    rwlock->_ReadLockCount          = 0;
    rwlock->_ReadLockWaiterCount    = 0;

    rwlock->_WriteLockCount         = 0;
    rwlock->_WriteLockWaiterCount   = 0;
    rwlock->_LockOwner              = NULL;

    // ReaderWriterLockType を Initialized 状態へ（最後に行なう）
    rwlock->_state        = ReaderWriterLockType::State_Initialized;
}


//---------------------------------------------------------------------------
//  ReaderWriterLockType オブジェクトの破棄
void FinalizeReaderWriterLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    // 待機スレッドがいないことを確認
    NN_SDK_REQUIRES( rwlock->_ReadLockCount == 0, NN_TEXT_OS("nn::os::FinalizeReaderWriterLock(): 指定された rwlock オブジェクトは Readers ロック状態です。") );
    NN_SDK_REQUIRES( rwlock->_WriteLockCount == 0, NN_TEXT_OS("nn::os::FinalizeReaderWriterLock(): 指定された rwlock オブジェクトは Writer ロック状態です。") );

    // ReaderWriterLockType を NotInitialized 状態へ変更（最初に行なう）
    rwlock->_state  = ReaderWriterLockType::State_NotInitialized;

    // 排他リソースを破棄
    Get(rwlock->_cvReadLockWaiter ).~InternalConditionVariable();
    Get(rwlock->_cvWriteLockWaiter).~InternalConditionVariable();
    Get(rwlock->_cs).~InternalCriticalSection();
}


//--------------------------------------------------------------------------
//  Readers ロックの取得
void AcquireReadLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    auto* current = detail::GetCurrentThread();

    // 自スレッドがロックオーナーなら、Reader ロックカウントをインクリメント
    // この if 文が成立するときは既にクリティカルセクション内であるため、
    // このままロックは開放せずにリターンする。
    if (rwlock->_LockOwner == current)
    {
        NN_SDK_ASSERT(rwlock->_ReadLockCount < ReaderWriterLockCountMax);
        ++rwlock->_ReadLockCount;
        return;
    }
    else
    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(rwlock->_cs);
        std::lock_guard<detail::InternalCriticalSection> lock(cs);

        // クリティカルセクション内に入れた時点で Writer ロック中ではない
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        NN_SDK_ASSERT(rwlock->_LockOwner == NULL);

        // Writer ロック待ちがいるなら待機
        while (rwlock->_WriteLockWaiterCount > 0)
        {
            ++rwlock->_ReadLockWaiterCount;
            Get(rwlock->_cvReadLockWaiter).Wait( &cs );
            --rwlock->_ReadLockWaiterCount;
        }

        // Readers ロックを獲得
        NN_SDK_ASSERT(rwlock->_ReadLockCount < ReaderWriterLockCountMax);
        ++rwlock->_ReadLockCount;
    }
}


//--------------------------------------------------------------------------
//  Readers ロックの取得（ポーリング）
bool TryAcquireReadLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    auto* current = detail::GetCurrentThread();

    // 自スレッドがロックオーナーなら、Reader ロックカウントをインクリメント
    // この if 文が成立するときは既にクリティカルセクション内であるため、
    // このままロックは開放せずにリターンする。
    if (rwlock->_LockOwner == current)
    {
        NN_SDK_ASSERT(rwlock->_ReadLockCount < ReaderWriterLockCountMax);
        ++rwlock->_ReadLockCount;
        return true;
    }
    else
    {   // クリティカルセクション
        if (!Get(rwlock->_cs).TryEnter())
        {
            return false;
        }
        NN_UTIL_SCOPE_EXIT
        {
            Get(rwlock->_cs).Leave();
        };

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
        // クリティカルセクション内に入れた時点で WriteLock 中ではない
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        NN_SDK_ASSERT(rwlock->_LockOwner == NULL);
#elif defined(NN_BUILD_CONFIG_OS_WIN)
        // Win 版だと InternalCriticalSection を多重ロックできてしまうため、
        // Writer ロック中でもクリティカルセクションに入れるので判定が必要。
        if (rwlock->_WriteLockCount > 0)
        {
            return false;
        }
#endif

        // Writer ロック待ちがいるなら false を返す
        if (rwlock->_WriteLockWaiterCount > 0)
        {
            return false;
        }

        // Readers ロックを獲得
        NN_SDK_ASSERT(rwlock->_ReadLockCount < ReaderWriterLockCountMax);
        ++rwlock->_ReadLockCount;
    }
    return true;
}


//--------------------------------------------------------------------------
//  Readers ロックを開放
void ReleaseReadLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_REQUIRES(rwlock->_ReadLockCount > 0);

    auto* current = detail::GetCurrentThread();

    if (rwlock->_LockOwner == current)
    {
        // この if 文が成立するときは既にクリティカルセクション内である
        --rwlock->_ReadLockCount;

        // この先の処理は ReleaseWriteLock() と同等
        if (rwlock->_WriteLockCount == 0 && rwlock->_ReadLockCount == 0)
        {
            rwlock->_LockOwner = NULL;

            if (rwlock->_WriteLockWaiterCount > 0)
            {
                // Writer ロック待機スレッドを起床
                Get(rwlock->_cvWriteLockWaiter).Signal();
            }
            else if (rwlock->_ReadLockWaiterCount > 0)
            {
                // Readers ロック待機スレッドを起床
                Get(rwlock->_cvReadLockWaiter).Broadcast();
            }

            // クリティカルセクションを抜ける
            Get(rwlock->_cs).Leave();
        }
    }
    else
    {   // クリティカルセクション
        detail::InternalCriticalSection&    cs = Get(rwlock->_cs);
        std::lock_guard<detail::InternalCriticalSection> lock(cs);

        // クリティカルセクション内に入れた時点で WriteLock 中ではない
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        NN_SDK_ASSERT(rwlock->_LockOwner == NULL);

        // Readers ロックカウンタをデクリメント
        --rwlock->_ReadLockCount;
        if (rwlock->_ReadLockCount == 0)
        {
            if (rwlock->_WriteLockWaiterCount > 0)
            {
                // Writer ロック待機スレッドを起床
                Get(rwlock->_cvWriteLockWaiter).Signal();

                // この時点で残っている Reader ロック待ちは、
                // ReleaseWriteLock() の中から起床される。
            }
        }
    }
}


//--------------------------------------------------------------------------
//  Writer ロックの取得
void AcquireWriteLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    auto* current = detail::GetCurrentThread();

    // 再帰ロックなら Writer ロックカウントをインクリメント
    // この if 文が成立するときは既に Write ロックを獲得済み状態であるため、
    // このままロックは開放せずにリターンする。
    if (rwlock->_LockOwner == current)
    {
        NN_SDK_ASSERT(rwlock->_WriteLockCount < ReaderWriterLockCountMax);
        ++rwlock->_WriteLockCount;
        return;
    }
    else
    {   // クリティカルセクション
        Get(rwlock->_cs).Enter();

        // クリティカルセクション内に入れた時点で Writer ロック中ではない
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        NN_SDK_ASSERT(rwlock->_LockOwner == NULL);

        // Readers ロック中なら待機
        while (rwlock->_ReadLockCount > 0)
        {
            ++rwlock->_WriteLockWaiterCount;
            Get(rwlock->_cvWriteLockWaiter).Wait( &Get(rwlock->_cs) );
            --rwlock->_WriteLockWaiterCount;
        }

        // Writer ロックを獲得、優先度継承が効くようにロックは開放しない。
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        rwlock->_WriteLockCount = 1;
        rwlock->_LockOwner = current;
    }
}


//--------------------------------------------------------------------------
//  Writer ロックの取得（ポーリング）
bool TryAcquireWriteLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    auto* current = detail::GetCurrentThread();

    // 再帰ロックなら Writer ロックカウントをインクリメント
    // この if 文が成立するときは既に Write ロックを獲得済み状態であるため、
    // このままロックは開放せずにリターンする。
    if (rwlock->_LockOwner == current)
    {
        NN_SDK_ASSERT(rwlock->_WriteLockCount < ReaderWriterLockCountMax);
        ++rwlock->_WriteLockCount;
        return true;
    }
    else
    {   // クリティカルセクション
        if (!Get(rwlock->_cs).TryEnter())
        {
            return false;
        }

        // クリティカルセクション内に入れた時点で Writer ロック中ではない
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        NN_SDK_ASSERT(rwlock->_LockOwner == NULL);

        // Readers ロック中なら false を返す
        if (rwlock->_ReadLockCount > 0)
        {
            Get(rwlock->_cs).Leave();
            return false;
        }

        // Writer ロックを獲得、優先度継承が効くようにロックは開放しない。
        NN_SDK_ASSERT(rwlock->_WriteLockCount == 0);
        rwlock->_WriteLockCount = 1;
        rwlock->_LockOwner = current;
    }
    return true;
}


//--------------------------------------------------------------------------
//  Writer ロックを開放
void ReleaseWriteLock(ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);
    NN_SDK_REQUIRES(rwlock->_WriteLockCount > 0);
    NN_ABORT_UNLESS(rwlock->_LockOwner == detail::GetCurrentThread());

    {   // 上記判定を通過できた時点で、既にクリティカルセクション内のはず

        // Writer ロックカウンタをデクリメント
        --rwlock->_WriteLockCount;
        if (rwlock->_WriteLockCount == 0 && rwlock->_ReadLockCount == 0)
        {
            rwlock->_LockOwner = NULL;

            if (rwlock->_WriteLockWaiterCount > 0)
            {
                // Writer ロック待機スレッドを起床
                Get(rwlock->_cvWriteLockWaiter).Signal();
            }
            else if (rwlock->_ReadLockWaiterCount > 0)
            {
                // Readers ロック待機スレッドを起床
                Get(rwlock->_cvReadLockWaiter).Broadcast();
            }

            // クリティカルセクションを抜ける
            Get(rwlock->_cs).Leave();
        }
    }
}


//--------------------------------------------------------------------------
//  Readers ロック状態か否か
bool IsReadLockHeld(const ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    return rwlock->_ReadLockCount > 0;
}


//--------------------------------------------------------------------------
//  自スレッドによる Writer ロック取得のチェック
bool IsWriteLockHeldByCurrentThread(const ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    return rwlock->_LockOwner == detail::GetCurrentThread() && rwlock->_WriteLockCount > 0;
}


//--------------------------------------------------------------------------
//  自スレッドがロックオーナーか否か
bool IsReaderWriterLockOwnerThread(const ReaderWriterLockType* rwlock) NN_NOEXCEPT
{
    NN_OS_ASSERT_RWLOCK_IS_INITIALIZED(rwlock, NN_CURRENT_FUNCTION_NAME);

    return rwlock->_LockOwner == detail::GetCurrentThread();
}


//---------------------------------------------------------------------------

}} // namespace nn::os

