﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_MutexTypes.h>
#include <nn/os/os_MutexApi.h>

#include "detail/os_Diag.h"
#include "detail/os_Common.h"
#include "detail/os_ThreadManager.h"
#include "detail/os_MutexImpl.h"

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os {

namespace detail {

#if !defined(NN_SDK_BUILD_RELEASE)
    // ロック履歴を検査し、足跡を残す
    // ネストカウンタや所有者を更新する前に呼び出す必要がある
    void PushAndCheckLockLevel(MutexType* mutex) NN_NOEXCEPT
    {
        // ロックレベルのチェック（Lock）
        if (mutex->_lockLevel == 0)
        {
            return;
        }

        ThreadType* thread = detail::GetCurrentThread();

        // ロック履歴はビットパターンで管理。右シフトで下位履歴を削除
        std::uint32_t  lockHistory = (thread->_lockHistory >> mutex->_lockLevel);

        // 指定された mutex のロックレベル値以上の取得履歴を確認
        if (lockHistory == 0x0)
        {
            // その mutex は初回ロックでなければならない
            NN_SDK_ASSERT(mutex->_nestCount == 0, NN_TEXT_OS("内部エラー：MutexType オブジェクトの内部状態が異常です。") );

            // ロック履歴に足跡を残す
            thread->_lockHistory |= ( 0x1 << mutex->_lockLevel );

        }
        else if (lockHistory == 0x1)
        {
            // 指定された mutex と同じロックレベル値のものが取得済みの場合、
            // 当該 mutex の再帰ロックならＯＫ、他の mutex ならＮＧ。

            // nestCount >= 1 なら、当該 mutex による再帰ロックなので問題なし。
            // そうでなければ、他の mutex で同一レベルを取得済みでＮＧ。
            NN_SDK_ASSERT( mutex->_nestCount >= 1, NN_TEXT_OS("ロックレベル違反を検出：同一ロックレベル値を持つ他のミューテックスが既にロックされています。") );

            // 対象 Mutex は再帰ロック可能でなければならない
            NN_SDK_ASSERT( mutex->_isRecursive, NN_TEXT_OS("内部エラー：MutexType オブジェクトの内部状態が異常です。") );

        }
        else
        {
            // 指定された mutex のロックレベル値以上のものが取得済み（停止）
            // つまり、ロックレベルの逆転を検出
            NN_ABORT(NN_TEXT_OS("ロックレベル違反を検出：ロックレベルの逆転を検出しました。") );
        }
    }

    // ロック履歴を検査し、足跡を消す
    // ネストカウンタや所有者を更新する前に呼び出す必要がある
    void PopAndCheckLockLevel(MutexType* mutex) NN_NOEXCEPT
    {
        // ロックレベルのチェック（Unlock）
        if (mutex->_lockLevel == 0)
        {
            return;
        }

        ThreadType* thread = detail::GetCurrentThread();

        // ロック履歴はビットパターンで管理。右シフトで下位履歴を削除
        std::uint32_t  lockHistory = thread->_lockHistory >> mutex->_lockLevel;

        // 指定された mutex と同じロックレベル値のものだけが取得済みの場合
        if (lockHistory == 0x1)
        {
            // ロックカウンタは 1 以上でなければアンロック順序がおかしい
            NN_SDK_ASSERT( mutex->_nestCount >= 1, NN_TEXT_OS("ロックレベル違反を検出：ロック順序と異なる不正なアンロックを検出しました。") );

            // ネストカウンタが最後のアンロックなら足跡を消す
            if ( mutex->_nestCount == 1 )
            {
                thread->_lockHistory &= ~( 0x1 << mutex->_lockLevel );
            }
        }
        else
        {
            // 上記以外なら停止
            NN_ABORT(NN_TEXT_OS("ロックレベル違反を検出：ロックされていないミューテックスをアンロックしようとしました。") );
        }
    }
#endif

    //  Mutex ロック成功後の共通処理
    //  正規の手順でロックに成功した状態であることが前提
    inline void AfterLockMutex(MutexType* mutex, ThreadType* current) NN_NOEXCEPT
    {
        // 再帰カウンタが限界値に達していたらアサート
        NN_SDK_ASSERT( mutex->_nestCount < MutexRecursiveLockCountMax, NN_TEXT_OS("再帰ロックカウンタが最大値（%d）に到達しています。"), MutexRecursiveLockCountMax );

        // ロックレベルのチェック
        detail::PushAndCheckLockLevel( mutex );

        // ロックカウンタをインクリメント、所有者を自スレッドへ
        mutex->_nestCount++;
        mutex->_ownerThread = current;
    }

}   // namespace detail


//---------------------------------------------------------------------------
//  MutexType オブジェクトのイニシャライズ
void InitializeMutex(MutexType* mutex, bool isRecursive, int lockLevel) NN_NOEXCEPT
{
    // 引数エラーチェック
    NN_SDK_REQUIRES( (lockLevel == 0) ||
                     ( (lockLevel >= MutexLockLevelMin) &&
                       (lockLevel <= MutexLockLevelMax) ),
        NN_TEXT_OS("nn::os::InitializeMutex(): lockLevel が不正です（lockLevel=%d）"), lockLevel);

    // クリティカルセクションの初期化
    new( &mutex->_mutex ) detail::InternalCriticalSection;

    // MutexType オブジェクトを初期化
    mutex->_isRecursive  = isRecursive;
    mutex->_lockLevel    = lockLevel;
    mutex->_nestCount    = 0;
    mutex->_ownerThread  = NULL;

    // MutexType を Initialized 状態へ（最後に行なう）
    mutex->_state        = MutexType::State_Initialized;
}


//---------------------------------------------------------------------------
//  MutexType オブジェクトのファイナライズ
void FinalizeMutex(MutexType* mutex) NN_NOEXCEPT
{
    // MutexType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::FinalizeMutex(): 指定されたミューテックスが初期化されていません。"));

    // MutexType オブジェクトを破棄（最初に行なう）
    mutex->_state  = MutexType::State_NotInitialized;

    // クリティカルセクションを破棄
    Get(mutex->_mutex).~InternalCriticalSection();
}


//---------------------------------------------------------------------------
//  Mutex のロック
void LockMutex(MutexType* mutex) NN_NOEXCEPT
{
    // MutexType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::LockMutex(): 指定されたミューテックスが初期化されていません。"));

    auto* current = detail::GetCurrentThread();

    // 再帰可能かどうかで処理を分ける
    if ( !mutex->_isRecursive )
    {
        // 再帰不可であれば、初ロックか他スレッドロック状態のいずれかである。
        // つまり、所有者は自スレッド以外である。なお、所有者を自スレッドに
        // 設定できるのは、自スレッドのみであるため、この事前条件チェックは
        // ロックスコープ外であってもスレッド安全である。
        NN_SDK_ASSERT( !(mutex->_ownerThread == current), NN_TEXT_OS("nn::os::LockMutex(): 再帰ロック不可のミューテックスを再帰ロックしようとしました。"));

        // ロックを獲得（待ちに入る可能性がある）
        Get(mutex->_mutex).Enter();
    }
    else
    {
        // 再帰可能の場合、初ロックか再帰ロックかで処理を分ける。
        // 所有者を自スレッドに設定できるのは、自スレッドによるロック成功のみ
        // であるため、この判定はロックスコープ外であってもスレッド安全である。
        if ( mutex->_ownerThread == current )
        {
            // 自スレッドによって既にロック済みなら、内部状態をチェック
            NN_SDK_ASSERT( mutex->_nestCount >= 1, NN_TEXT_OS("nn::os::LockMutex(): ミューテックスの再帰カウンタ値（%d）が異常です。"), mutex->_nestCount);

            // このケースでは、mutex->_mutex.Enter() は呼び出さない。
            // 下位実装関数は再帰ロックに対応していないものとして扱う。
        }
        else
        {
            // 所有者が自スレッドでない場合は、初ロックか、他スレッドによる
            // ロック中の両方のケースがあり、かつその状態は他スレッドによって
            // 過渡状態である可能性もある。いずれの場合もロックを試みる必要
            // があるため、ロックを獲得（待ちに入る可能性がある）
            Get(mutex->_mutex).Enter();
        }
    }

    // ロックに成功したら共通処理を呼出す
    detail::AfterLockMutex(mutex, current);
}


//  Mutex のロック（ポーリング）
bool TryLockMutex(MutexType* mutex) NN_NOEXCEPT
{
    // MutexType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::TryLockMutex(): 指定されたミューテックスが初期化されていません。"));

    auto* current = detail::GetCurrentThread();

    // 再帰可能かどうかで処理を分ける
    if ( !mutex->_isRecursive )
    {
        // 再帰不可であれば、初ロックか他スレッドロック状態のいずれかである。
        // つまり、所有者は自スレッド以外である。なお、所有者を自スレッドに
        // 設定できるのは、自スレッドのみであるため、この事前条件チェックは
        // ロックスコープ外であってもスレッド安全である。
        NN_SDK_ASSERT( !(mutex->_ownerThread == current), NN_TEXT_OS("nn::os::TryLockMutex(): 再帰ロック不可のミューテックスを再帰ロックしようとしました。"));

        // ミューテックスをポーリングする
        if ( !(Get(mutex->_mutex).TryEnter()) )
        {
            // ポーリング失敗の場合、false を返す
            return false;
        }
    }
    else
    {
        // 再帰可能の場合、初ロックか再帰ロックかで処理を分ける。
        // 所有者を自スレッドに設定できるのは、自スレッドによるロック成功のみ
        // であるため、この判定はロックスコープ外であってもスレッド安全である。
        if ( mutex->_ownerThread == current )
        {
            // 自スレッドによって既にロック済みなら、内部状態をチェック
            NN_SDK_ASSERT( mutex->_nestCount >= 1, NN_TEXT_OS("nn::os::TryLockMutex(): ミューテックスの再帰カウンタ値（%d）が異常です。"), mutex->_nestCount);

            // このケースでは、mutex->_mutex.TryEnter() は呼び出さない。
            // 下位実装関数は再帰ロックに対応していないものとして扱う。
        }
        else
        {
            // 所有者が自スレッドでない場合は、初ロックか、他スレッドによる
            // ロック中の両方のケースがあり、かつその状態は他スレッドによって
            // 過渡状態である可能性もある。いずれの場合もロックを試みる必要
            // があるため、ここでポーリングを実施
            if ( !(Get(mutex->_mutex).TryEnter()) )
            {
                // ポーリング失敗の場合、false を返す
                return false;
            }
        }
    }

    // ロックに成功したら共通処理を呼出す
    detail::AfterLockMutex(mutex, current);

    return  true;
}


//---------------------------------------------------------------------------
//  Mutex のアンロック
void UnlockMutex(MutexType* mutex) NN_NOEXCEPT
{
    // MutexType オブジェクトが Initialized 状態でなければ実行停止
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::UnlockMutex(): 指定されたミューテックスが初期化されていません。"));

    // ロック所有者でなければエラー
    NN_SDK_REQUIRES( mutex->_nestCount > 0, NN_TEXT_OS("nn::os::UnlockMutex(): 指定されたミューテックスがロックされていません。"));
    NN_SDK_REQUIRES( mutex->_ownerThread == detail::GetCurrentThread(), NN_TEXT_OS("nn::os::UnlockMutex(): 所有者でないミューテックスをアンロックしようとしました。"));

    // ロックレベルのチェック
    detail::PopAndCheckLockLevel( mutex );

    // ネストカウンタをデクリメントし、所有者をクリア
    mutex->_nestCount--;
    if (mutex->_nestCount == 0)
    {
        // 所有者をクリア
        mutex->_ownerThread = NULL;

        // Mutex を返却する
        Get(mutex->_mutex).Leave();
    }
}


//---------------------------------------------------------------------------
//  Mutex が自スレッドによってロックされているか
bool IsMutexLockedByCurrentThread(const MutexType* mutex) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( mutex->_state == MutexType::State_Initialized, NN_TEXT_OS("nn::os::IsMutexLockedByCurrentThread(): 指定されたミューテックスが初期化されていません。"));

    return mutex->_ownerThread == detail::GetCurrentThread();
}

//---------------------------------------------------------------------------

}} // namespace nn::os
