﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_ThreadLocalRegion.h>
#include <nn/os/os_DefaultUserExceptionHandler.h>

#include "os_Diag.h"
#include "os_UserExceptionHandlerImpl.h"
#include "os_UserExceptionHandlerImpl-os.horizon.h"
#include "os_DefaultUserExceptionHandlerImpl.h"

namespace nn { namespace os { namespace detail {

//-----------------------------------------------------------------------------
// nn::os::UserExceptionInfo 構造体をスタック上から確保する場合を考慮して、
// sp が 8 byte アライメントからずれないように、サイズも 8 の倍数であることを
// チェックしておく。
//
static_assert( (sizeof(UserExceptionInfo) % 8) == 0, "" );

#if defined(NN_OS_CPU_ARM_AARCH32_ARMV7A) || \
    defined(NN_OS_CPU_ARM_AARCH32_ARMV8A)

    #if NN_BUILD_CONFIG_FPU_NUM_DOUBLE_REGISTERS != 32
        #error "FPU レジスタ数 16 には未対応です。"
        // nn::os::UserExceptionInfo 構造体の定義なども含めて対応必要
    #endif
#endif

#if 0
//  この #if 0 で括られた関数群は、代替実装が rtld 側にフルアセンブラで
//  用意されています。こちらのコードは、メンテナンス用に残しておきます。


inline bool IsMemoryAccessible(uintptr_t addr) NN_NOEXCEPT
{
    svc::MemoryInfo block;
    svc::PageInfo   page;

    // QueryMemory() は失敗しないので result チェックを行なわない。
    svc::QueryMemory(&block, &page, addr);

    return block.permission == svc::MemoryPermission_ReadWrite;
}

//-----------------------------------------------------------------------------
// rtld.S のユーザ例外エントリから利用される内部関数
//
//  例外発生直後の nn::svc::ExceptionInfo 構造体や他の情報を受け取り、
//  nn::os:UserExceptionInfo 構造体へ情報をコピーする準備を行なう。
//
//  この関数は、ユーザ例外発生直後、svc::ReturnFromException() 直前に呼ばれる。
//  返値が false ならスタックオーバーフローだったことを示す。
//
//  exceptionType:
//      例外要因
//
//  pSvcExceptionInfo:
//      カーネルから受取った nn::svc::ExceptionInfo 構造体へのポインタ
//-----------------------------------------------------------------------------
UserExceptionInfoDetail* PrepareToCopyUserExceptionInfo(
                            Bit32               exceptionType,
                            svc::ExceptionInfo* pSvcExceptionInfo) NN_NOEXCEPT
{
    //----------------------------------------------------------------
    // まず、ハンドラスタックと例外情報格納領域のアドレスを決定する。

    UserExceptionInfo*  info                = g_UserExceptionInfo;
    uintptr_t           handlerStackBottom  = g_UserExceptionStackBottom;
    uintptr_t           threadStack         = pSvcExceptionInfo->sp;

    // 以下の 1)～3) の順序は変更してはならない。
    // 1) 例外情報格納領域をスレッドスタックから確保
    if (info == UserExceptionInfoUsesThreadStack)
    {
        threadStack -= sizeof(nn::os::UserExceptionInfo);
        info = reinterpret_cast<UserExceptionInfo*>(threadStack);
    }

    // 2) 例外ハンドラスタックの位置確定
    if (handlerStackBottom == reinterpret_cast<uintptr_t>(HandlerStackUsesThreadStack))
    {
        handlerStackBottom = threadStack;
    }

    // 3) 例外情報格納領域をハンドラスタックから確保
    if (info == UserExceptionInfoUsesHandlerStack)
    {
        handlerStackBottom -= sizeof(nn::os::UserExceptionInfo);
        info = reinterpret_cast<UserExceptionInfo*>(handlerStackBottom);
        if (g_UserExceptionStackBottom != reinterpret_cast<uintptr_t>(HandlerStackUsesThreadStack) && handlerStackBottom <= g_UserExceptionStackTop)
        {
            return NULL;
        }
    }

    if (!IsMemoryAccessible(reinterpret_cast<uintptr_t>(info)) ||
        !IsMemoryAccessible(handlerStackBottom))
    {
        // ハンドラスタックや ExceptionInfo のメモリが利用できない
        return NULL;
    }


    //----------------------------------------------------------------
    // nn::os::UserExceptionInfo に例外発生時の情報を埋める

    // 例外要因を代入
    info->exceptionType = exceptionType;

    {
        //------------------------------------------------------------------
        // ユーザ例外起動用に nn::svc::ExceptionInfo を書き換える

        pSvcExceptionInfo->r[0] = reinterpret_cast<uintptr_t>(info);
        pSvcExceptionInfo->sp   = handlerStackBottom;
        // pSvcExceptionInfo->pc は rtld 内部で書き替える。
    }

    return &info->detail;
}
#endif

//-----------------------------------------------------------------------------

}}} // namespace nn::os::detail

