﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_TransferMemoryApi.h>
#include <nn/os/os_TransferMemoryTypes.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_TransferMemoryImpl.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_MemoryMapSelect.h>

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------

namespace {

svc::MemoryPermission ConvertToSvcMemoryPermission(os::MemoryPermission permission) NN_NOEXCEPT
{
    switch (permission)
    {
    case os::MemoryPermission_None:
            return svc::MemoryPermission_None;

    case os::MemoryPermission_ReadOnly:
            return svc::MemoryPermission_Read;

    case os::MemoryPermission_WriteOnly:
            return svc::MemoryPermission_Write;

    case os::MemoryPermission_ReadWrite:
            return svc::MemoryPermission_ReadWrite;

    default:
            break;
    }

    NN_ABORT(NN_TEXT_OS("nn::os::CreateTransferMemory(): 指定された permission=0x%x が不正です。"), permission);
}

}   // namespace

//--------------------------------------------------------------------------
//  Transfer memory の作成（ハンドルの生成）
nn::Result TransferMemoryImplByHorizon::Create(NativeHandle* outHandle, void* address, size_t size, os::MemoryPermission permission) NN_NOEXCEPT
{
    auto svcPermission    = ConvertToSvcMemoryPermission(permission);

    svc::Handle handle;
    auto result = svc::CreateTransferMemory(&handle, reinterpret_cast<uintptr_t>(address), size, svcPermission);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultMaxHandle )
        {
            return os::ResultMaxHandle();
        }
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            return os::ResultOutOfTransferMemory();
        }
        NN_RESULT_CATCH( svc::ResultLimit )
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS( os::ResultResourceLimit() );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    *outHandle = static_cast<NativeHandle>( static_cast<nnHandle>(handle).value );
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
//  Transfer memory の削除（ハンドルを削除）
void TransferMemoryImplByHorizon::Close(NativeHandle handle) NN_NOEXCEPT
{
    svc::Handle svcHandle(handle);

    auto result = svc::CloseHandle(svcHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

//--------------------------------------------------------------------------
//  Transfer memory をマップ
Result TransferMemoryImplByHorizon::Map(void** pOutAddress, NativeHandle handle, void* address, size_t size, os::MemoryPermission ownerPermission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(address != NULL, NN_TEXT_OS("nn::os::MapTransferMemory(): address が NULL になっています。"));

    svc::Handle svcHandle(handle);
    auto svcOwnerPermission = ConvertToSvcMemoryPermission(ownerPermission);

    auto result = svc::MapTransferMemory(svcHandle,
                                         reinterpret_cast<uintptr_t>(address),
                                         size,
                                         svcOwnerPermission);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidHandle )
        {
            return os::ResultInvalidHandle();
        }
        NN_RESULT_CATCH( svc::ResultInvalidSize )
        {
            return os::ResultInvalidTransferMemorySize();
        }
        NN_RESULT_CATCH( svc::ResultInvalidState )
        {
            return os::ResultInvalidTransferMemoryState();
        }
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            // Map しようとした場所のメモリ状態が Free でなかった
            // Kernel が IPC 用に使っている場所だった可能性が高い
            // 別のアドレスでリトライする
            return os::ResultInvalidCurrentMemoryState();
        }
        NN_RESULT_CATCH( svc::ResultInvalidRegion )
        {
            // ResultInvalidRegion の場合も上記と同様
            return os::ResultInvalidCurrentMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    *pOutAddress = address;
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
//  Transfer memory をアンマップ
void TransferMemoryImplByHorizon::Unmap(NativeHandle handle, void* address, size_t size) NN_NOEXCEPT
{
    svc::Handle svcHandle(handle);

    auto result = svc::UnmapTransferMemory(svcHandle,
                                           reinterpret_cast<uintptr_t>(address),
                                           size);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

