﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdint>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/util/util_TypedStorage.h>
#include "os_Diag.h"
#include "os_Common.h"
#include "os_TlsManager.h"
#include "os_ThreadManager.h"



namespace nn { namespace os {
namespace detail {

//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace {

// InvokeTlsDestructors 関数にて NULL でない TLS 値を持つ TLS デストラクタ呼び出しの試行回数
const int TryCallDestructorCount = 4;

// デフォルトの TLS デストラクタ（何もしない）
void DefaultTlsDestructor(uintptr_t value)
{
    NN_UNUSED(value);
}

}   // namespace

//---------------------------------------------------------------------------
// 空いている TLS スロットを検索
int TlsManager::SearchUnusedTlsSlotUnsafe(bool sdkInternal) const NN_NOEXCEPT
{
    if (sdkInternal)
    {
        // SDK 内部用に配列の後ろから空きを探す
        for (int slotNum=TotalTlsSlotCountMax - 1; slotNum>=TlsSlotCountMax + SdkInternalTlsCount; --slotNum)
        {
            if (m_TlsDestructorArray[ slotNum ] == NULL)
            {
                return slotNum;
            }
        }
        NN_ABORT(NN_TEXT_OS("nn::os::SdkAllocateTlsSlot(): SDK 内部用の空き TLS スロットが見つかりません"));
    }
    else
    {
        // ユーザー用に配列の前から空きを探す
        for (int slotNum=0; slotNum<TlsSlotCountMax; ++slotNum)
        {
            if (m_TlsDestructorArray[ slotNum ] == NULL)
            {
                return slotNum;
            }
        }
        NN_ABORT(NN_TEXT_OS("nn::os::AllocateTlsSlot(): 空き TLS スロットが見つかりません"));
    }
    return -1;
}

//---------------------------------------------------------------------------
// TlsManager のコンストラクタ
TlsManager::TlsManager() NN_NOEXCEPT
    : m_NumUsedTlsSlots( 0 )
{
    // TLS アレイの初期化
    for (int slotNum=0; slotNum<TotalTlsSlotCountMax; ++slotNum)
    {
        m_TlsDestructorArray[slotNum] = NULL;
    }
}


//---------------------------------------------------------------------------
// 全ての TLS デストラクタを順番に呼出す
void TlsManager::InvokeTlsDestructors() NN_NOEXCEPT
{
    ThreadType* thread = detail::GetCurrentThread();

    // 全ての TLS スロットを順番に調べる
    for (int slotNum=0; slotNum<TotalTlsSlotCountMax; ++slotNum)
    {
        TlsDestructor   pDestructor = m_TlsDestructorArray[ slotNum ];

        // デストラクタを呼び出す
        if (pDestructor != NULL)
        {
            pDestructor( thread->_tlsValueArray[ slotNum ] );
        }

        // TLS 値を NULL に設定
        thread->_tlsValueArray[ slotNum ] = static_cast<uintptr_t>(NULL);
    }

    // 当該スレッドの全 TLS スロットの TLS 値が全て NULL になるまで、
    // 最大で TotalTlsSlotCountMax 回繰り返す
    for (int i=0; i<TryCallDestructorCount; i++)
    {
        bool isAllTlsValueNull = true;

        for (int slotNum=0; slotNum<TotalTlsSlotCountMax; ++slotNum)
        {
            TlsDestructor   pDestructor = m_TlsDestructorArray[ slotNum ];

            // 前のループにあるデストラクタの中で他の TLS スロットに値を設定する可能性があるため、
            // 全デストラクタ呼び出し後に TLS スロットに値がセットされていないかチェックする
            if (thread->_tlsValueArray[ slotNum ] != static_cast<uintptr_t>(NULL))
            {
                isAllTlsValueNull = false;

                if (pDestructor != NULL)
                {
                    pDestructor( thread->_tlsValueArray[ slotNum ] );
                }

                // 再度 TLS 値を NULL に設定
                thread->_tlsValueArray[ slotNum ] = static_cast<uintptr_t>(NULL);
            }
        }

        // 全て NULL なら終了
        if (isAllTlsValueNull)
        {
            break;
        }
    }
}


//---------------------------------------------------------------------------
// TLS スロットを確保し、デストラクタを登録する
bool TlsManager::AllocateTlsSlot(TlsSlot* outTlsSlot, TlsDestructor pDestructor, bool sdkInternal) NN_NOEXCEPT
{
    int slotNum;

    {    // クリティカルセクション区間
        std::lock_guard<InternalCriticalSection> lockForSuspend( Get(GetCurrentThread()->_csThread) );
        std::lock_guard<InternalCriticalSection> lock( m_TlsCriticalSection );

        // ユーザー側の TLS 使用数を確認
        if ((!sdkInternal && m_NumUsedTlsSlots >= TlsSlotCountMax))
        {
            return false;
        }

        // 空いている TLS スロットを検索（ユーザー用は必ず見つかる。SDK 用が見つからない場合は内部でアボート）
        slotNum = SearchUnusedTlsSlotUnsafe(sdkInternal);

        // TLS スロット確保
        m_TlsDestructorArray[slotNum] = (pDestructor == NULL) ?  DefaultTlsDestructor : pDestructor;

        // TLS スロット使用数をインクリメント
        if (!sdkInternal)
        {
            ++m_NumUsedTlsSlots;
        }
    }
    // この時点で TLS スロットの確保を完了

    // 全スレッドの当該 TLS スロットの TLS 値をゼロに設定
    GetThreadManagerInstance()->SetZeroToAllThreadsTlsSafe( slotNum );

    // 最後にユーザ変数に返す
    outTlsSlot->_innerValue = slotNum;
    return true;
}


//---------------------------------------------------------------------------
// TLS スロットを返却する
void TlsManager::FreeTlsSlot(TlsSlot tlsSlot) NN_NOEXCEPT
{
    uint32_t slotNum = tlsSlot._innerValue;

    {   // クリティカルセクション区間
        std::lock_guard<InternalCriticalSection> lockForSuspend( Get(GetCurrentThread()->_csThread) );
        std::lock_guard<InternalCriticalSection> lock( m_TlsCriticalSection );

        if (m_TlsDestructorArray[ slotNum ] == NULL)
        {
            NN_ABORT(NN_TEXT_OS("nn::os::FreeTlsSlot(): 未使用の TLS スロットを返却しようとしました。"));
        }

        // TLS スロットを解放
        m_TlsDestructorArray[ slotNum ] = NULL;

        // TLS 使用数をデクリメント
        if (slotNum < TlsSlotCountMax)
        {
            --m_NumUsedTlsSlots;
        }
    }
}

}   // namespace detail
}}  // namespace nn::os
