﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_TimerEventTypes.h>
#include <nn/os/os_TimerEventApi.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_TimerEventHelper.h"


namespace nn { namespace os {
namespace detail {

//-----------------------------------------------------------------------------
// タイマー時間の飽和加算
TimeSpan SaturatedAdd(TimeSpan t1, TimeSpan t2) NN_NOEXCEPT
{
    NN_SDK_ASSERT( (t1 >= 0) && (t2 >= 0) );

    uint64_t ut1    = t1.GetNanoSeconds();
    uint64_t ut2    = t2.GetNanoSeconds();

    // 元の値が int64_t 同士の加算なので added はオーバーフローしない
    uint64_t added  = ut1 + ut2;

    return TimeSpan::FromNanoSeconds(std::min<uint64_t>(INT64_MAX, added));
}

//-----------------------------------------------------------------------------
// タイマー動作を停止する
void StopTimerUnsafe(TimerEventType* event) NN_NOEXCEPT
{
    Get(event->_nextTimeToWakeup) = 0;
    event->_timerState            = TimerEventType::TimerState_Stop;
}

//-----------------------------------------------------------------------------
// タイマー動作の状態をチェックして _signalState へ反映させる。
//
// 最後に設定された _nextTimeToWakeup から現在までの間に、
// TimerEventType オブジェクトへの操作はないことが設計上保証されているので、
// タイマー動作以外にシグナル化される要因はない。
//
// その後、タイマーの次のシグナル時間を算出して設定しておく。
bool UpdateSignalStateAndRecalcNextTimeToWakeupUnsafe(TimerEventType* event, TimeSpan currTime) NN_NOEXCEPT
{
    TimeSpan nextTime = Get(event->_nextTimeToWakeup);

    switch (event->_timerState)
    {
    case TimerEventType::TimerState_Stop:
            break;

    case TimerEventType::TimerState_OneShot:
            if (nextTime < currTime)
            {
                event->_signalState = true;
                StopTimerUnsafe(event);
                return true;
            }
            break;

    case TimerEventType::TimerState_Periodic:
            if (nextTime < currTime)
            {
                event->_signalState = true;

                nextTime = SaturatedAdd(nextTime, Get(event->_interval));
                // 1 回の加算で追い越せない場合は計算し直す
                if (!(nextTime >= currTime))
                {
                    uint64_t elapseFromFirst = (currTime - Get(event->_first)).GetNanoSeconds();
                    uint64_t interval        = Get(event->_interval).GetNanoSeconds();
                    // オーバーフローを加味した次の interval への繰り上げ
                    uint64_t t = ((elapseFromFirst / interval) + 1) * interval;
                             t = std::min<uint64_t>(INT64_MAX, t);

                    // この加算ではオーバーフローの可能性があるので飽和加算する
                    nextTime = SaturatedAdd(TimeSpan::FromNanoSeconds(t), Get(event->_first));
                }

                Get(event->_nextTimeToWakeup) = nextTime;
                return true;
            }
            break;

    default: NN_UNEXPECTED_DEFAULT;
    }

    return false;
}

//-----------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

