﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Config.h>

#if !defined(NN_BUILD_CONFIG_OS_SUPPORTS_WIN32)
    #error  "OS 種別として Win32 が指定されていません。"
#endif

#include <nn/nn_Windows.h>
#include <nn/nn_Macro.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_ThreadTypes.private.h>
#include <nn/os/detail/os_InternalCriticalSection.h>

//--------------------------------------------------------------------------
//  C++ 向けのプロトタイプ宣言
//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//---------------------------------------------------------------------------
//  C++ クラスの定義
//---------------------------------------------------------------------------

//
//  ThreadManagerImplByWin32 クラスです。
//

class ThreadManagerImplByWin32
{
    // コピー、ムーブを禁止する
    NN_DISALLOW_COPY( ThreadManagerImplByWin32 );
    NN_DISALLOW_MOVE( ThreadManagerImplByWin32 );

public:
    // コンストラクタ
    explicit ThreadManagerImplByWin32(ThreadType* mainThread)  NN_NOEXCEPT;

    // Thread 生成（coreNumber あり）
    Result  CreateThread(ThreadType* thread, void (*function)(ThreadType *), int coreNumber) NN_NOEXCEPT;

    // Thread 開始
    void    StartThread(const ThreadType* thread)   NN_NOEXCEPT;

    // Thread 削除
    void    DestroyThreadUnsafe(ThreadType* thread)   NN_NOEXCEPT;

    // Thread が終了するのを待機
    void    WaitForExitThread(ThreadType *thread)   NN_NOEXCEPT;

    // Thread の Yield
    void    YieldThread()   NN_NOEXCEPT;

    // Thread の優先度変更
    bool    ChangePriority(ThreadType *thread, int priority)    NN_NOEXCEPT;

    // Thread の現在優先度を返す
    int     GetCurrentPriority(const ThreadType *thread) const NN_NOEXCEPT;

    // Thread の ID を取得する
    uint64_t GetThreadId(const ThreadType* thread) const NN_NOEXCEPT;

    // スレッドの Suspend/Resume 関連
    void    SuspendThreadUnsafe(ThreadType* thread) NN_NOEXCEPT;
    void    ResumeThreadUnsafe(ThreadType* thread) NN_NOEXCEPT;
    void    GetThreadContextUnsafe(ThreadContextInfo* context, const ThreadType* thread) NN_NOEXCEPT;

    // スレッド名をデバッガ等に登録する
    void    NotifyThreadNameChangedImpl(const ThreadType* thread) const NN_NOEXCEPT;

    // 指定された thread をカレントスレッドとして TLS にセットする
    void    SetCurrentThread(const ThreadType* thread) const NN_NOEXCEPT;

    // 現在の thread を TLS から取得する
    ThreadType* GetCurrentThread() const NN_NOEXCEPT;

    // 現在動作中のプロセッサ番号の取得
    int     GetCurrentCoreNumber() const NN_NOEXCEPT;

    // スレッド生成時のデフォルトプロセッサ番号を取得
    int     GetDefaultCoreNumber() const NN_NOEXCEPT
    {
        // 便宜上、デフォルトプロセッサ番号は 0 とする
        return 0;
    }

    // コアアフィニティの設定
    void    SetThreadCoreMask(ThreadType* thread, int idealCore, Bit64 affinityMask) const NN_NOEXCEPT;

    // コアアフィニティの取得
    void    GetThreadCoreMask(int* pOutIdealCore, Bit64* pOutAffinityMask, const ThreadType* thread) const NN_NOEXCEPT;

    // 割り当て可能なコアの取得
    nn::Bit64    GetThreadAvailableCoreMask() const NN_NOEXCEPT;

    bool MapAliasStack(const void* to, const void* from, size_t size) const NN_NOEXCEPT
    {
        NN_UNUSED(to);
        NN_UNUSED(from);
        NN_UNUSED(size);
        return true;
    }
    bool UnmapAliasStack(const void* to, const void* from, size_t size) const NN_NOEXCEPT
    {
        NN_UNUSED(to);
        NN_UNUSED(from);
        NN_UNUSED(size);
        return true;
    }

    NN_NORETURN void QuickExit() NN_NOEXCEPT;
    NN_NORETURN void ExitProcessImpl() NN_NOEXCEPT;

private:
    static const size_t DefaultReservedStackSize = 0x100000;

private:
    DWORD   m_currentThreadTlsIndex;  // CurrentThread ポインタ格納用TLS
    mutable InternalCriticalSection m_criticalSectionWin32;
};

typedef ThreadManagerImplByWin32    ThreadManagerImpl;

}  // namespace detail
}} // namespace nn::os

