﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>

#include <nn/os/os_Config.h>
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#include <cfenv>
#endif

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/os/os_ThreadTypes.private.h>
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Thread.h>
#include <nn/svc/svc_MemoryMapSelect.h>
#include <nn/svc/svc_ThreadLocalRegion.h>

#include "os_Diag.h"

namespace nn { namespace os { namespace detail {

//--------------------------------------------------------------------------
//  カーネルが管理するスレッド優先度全体の範囲
const int   NativeThreadPriorityRangeSize = svc::LowestThreadPriority - svc::HighestThreadPriority + 1;

//--------------------------------------------------------------------------
//  カーネル用や特権処理用に予約されているスレッド優先度の範囲
const int   ReservedThreadPriorityRangeSize = svc::LibraryThreadPriorityHighest;

//--------------------------------------------------------------------------
//  システムプロセス用のスレッド優先度の範囲
const int   SystemThreadPriorityRangeSize = NativeThreadPriorityRangeSize - ReservedThreadPriorityRangeSize;

//--------------------------------------------------------------------------
//  アプリケーションプロセス用のスレッド優先度のオフセット値（ゲタ）
const int   UserThreadPriorityOffset = 28;

//--------------------------------------------------------------------------
//  カーネルが提供するスレッド優先度（現状では定数値で持っておく）
const int   HighestNativeThreadPriority = 0;
const int   LowestNativeThreadPriority  = NativeThreadPriorityRangeSize - 1;


//--------------------------------------------------------------------------

class ThreadManagerImplByHorizon
{
    NN_DISALLOW_COPY(ThreadManagerImplByHorizon);
    NN_DISALLOW_MOVE(ThreadManagerImplByHorizon);

public:
    explicit ThreadManagerImplByHorizon(ThreadType* mainThread) NN_NOEXCEPT;

    Result CreateThread(ThreadType* thread, void (*function)(ThreadType*), int idealCoreNumber) NN_NOEXCEPT;
    void StartThread(const ThreadType* thread)                  NN_NOEXCEPT;
    void DestroyThreadUnsafe(ThreadType* thread)                NN_NOEXCEPT;
    void WaitForExitThread(ThreadType* thread)                  NN_NOEXCEPT;
    void YieldThread()                                          NN_NOEXCEPT;
    bool ChangePriority(ThreadType* thread, int priority)       NN_NOEXCEPT;
    int  GetCurrentPriority(const ThreadType* thread) const NN_NOEXCEPT;
    uint64_t GetThreadId(const ThreadType* thread) const NN_NOEXCEPT;

    void SuspendThreadUnsafe(ThreadType* thread) NN_NOEXCEPT;
    void ResumeThreadUnsafe(ThreadType* thread) NN_NOEXCEPT;
    void GetThreadContextUnsafe(ThreadContextInfo* context, const ThreadType* thread) NN_NOEXCEPT;

    void NotifyThreadNameChangedImpl(const ThreadType* thread) const NN_NOEXCEPT
    {
        NN_UNUSED(thread);
    }
    void SetCurrentThread(const ThreadType* thread) const NN_NOEXCEPT
    {
        nn::svc::GetThreadLocalRegion()->pThreadType = reinterpret_cast<uintptr_t>(thread);
    }
    ThreadType* GetCurrentThread() const NN_NOEXCEPT
    {
        return reinterpret_cast<ThreadType*>(nn::svc::GetThreadLocalRegion()->pThreadType);
    }
    int  GetCurrentCoreNumber() const NN_NOEXCEPT;
    int  GetDefaultCoreNumber() const NN_NOEXCEPT
    {
        return nn::svc::IdealCoreUseProcessValue;
    }
    void SetThreadCoreMask(ThreadType* thread, int idealCore, Bit64 affinityMask) const NN_NOEXCEPT;
    void GetThreadCoreMask(int* pOutIdealCore, Bit64* pOutAffinityMask, const ThreadType* thread) const NN_NOEXCEPT;
    nn::Bit64    GetThreadAvailableCoreMask() const NN_NOEXCEPT;

    bool MapAliasStack(const void* to, const void* from, size_t size) const NN_NOEXCEPT;
    bool UnmapAliasStack(const void* to, const void* from, size_t size) const NN_NOEXCEPT;

    NN_NORETURN void QuickExit() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
        // clang の時だけ std::quick_exit() を呼ぶ。
        std::quick_exit(0);
#else
        // gcc の場合は std::quick_exit() が未サポートだが、
        // std::at_quick_exit() も未サポートなので、そのままプロセス終了する。
        svc::ExitProcess();
        NN_ABORT("Unreachable code");
#endif
    }

    NN_NORETURN void ExitProcessImpl() NN_NOEXCEPT
    {
        svc::ExitProcess();
        NN_ABORT("Unreachable code");
    }
};

typedef ThreadManagerImplByHorizon ThreadManagerImpl;

extern uintptr_t   g_OsBootParamter;

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
typedef std::fenv_t FpuEnvType;
#elif defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
struct FpuEnvType
{
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
        nn::Bit32   _fpscr;
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
        nn::Bit32   _fpcr;
        nn::Bit32   _fpsr;
    #endif
};
#endif

inline void SetFenv(FpuEnvType* pFenv) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
    std::fesetenv( pFenv );
#elif defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
    asm volatile (" vmsr fpscr, %0" : : "r"(pFenv->_fpscr) : "memory");
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
    asm volatile (" msr fpcr, %0" : : "r"(static_cast<Bit64>(pFenv->_fpcr)) : "memory");
    asm volatile (" msr fpsr, %0" : : "r"(static_cast<Bit64>(pFenv->_fpsr)) : "memory");
    #endif
#endif  // CLANG or GCC
}

inline void GetFenv(FpuEnvType* pFenv) NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
    std::fegetenv( pFenv );
#elif defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
    #if defined(NN_BUILD_CONFIG_CPU_ARM)
    Bit32 value32;
    asm volatile (" vmrs %0, fpscr" : "=r"(value32) : : "memory");
    pFenv->_fpscr = value32;
    #elif defined(NN_BUILD_CONFIG_CPU_ARM64)
    Bit64 value64;
    asm volatile (" mrs %0, fpcr" : "=r"(value64) : : "memory");
    pFenv->_fpcr = value64 & 0xffffffff;
    asm volatile (" mrs %0, fpsr" : "=r"(value64) : : "memory");
    pFenv->_fpsr = value64 & 0xffffffff;
    #endif
#endif  // CLANG or GCC
}

//--------------------------------------------------------------------------

}}} // namespace nn::os::detail

