﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_SharedMemoryApi.h>
#include <nn/os/os_SharedMemoryTypes.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_SharedMemoryImpl.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dd.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_MemoryMapSelect.h>

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------

namespace {

svc::MemoryPermission ConvertToSvcMemoryPermission(os::MemoryPermission permission) NN_NOEXCEPT
{
    switch (permission)
    {
    case os::MemoryPermission_ReadOnly:
            return svc::MemoryPermission_Read;

    case os::MemoryPermission_ReadWrite:
            return svc::MemoryPermission_ReadWrite;

    default:
            break;
    }

    NN_ABORT(NN_TEXT_OS("nn::os::CreateSharedMemory(): 指定された permission=0x%x が不正です。"), permission);
}

}   // namespace

//--------------------------------------------------------------------------
//  共有メモリの作成（ハンドルの生成）
nn::Result SharedMemoryImplByHorizon::Create(NativeHandle* outHandle, size_t size, os::MemoryPermission myPermission, os::MemoryPermission otherPermission) NN_NOEXCEPT
{
    auto svcMyPermission    = ConvertToSvcMemoryPermission(myPermission);
    auto svcOtherPermission = ConvertToSvcMemoryPermission(otherPermission);

    svc::Handle handle;
    auto result = svc::CreateSharedMemory(&handle, size, svcMyPermission, svcOtherPermission);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultOutOfMemory )
        {
            return os::ResultOutOfMemory();
        }
        NN_RESULT_CATCH( svc::ResultOutOfResource )
        {
            return os::ResultOutOfResource();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    *outHandle = static_cast<NativeHandle>( static_cast<nnHandle>(handle).value );
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
//  共有メモリの削除（ハンドルを削除）
void SharedMemoryImplByHorizon::Close(NativeHandle handle) NN_NOEXCEPT
{
    svc::Handle svcHandle(handle);

    auto result = svc::CloseHandle(svcHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------
//  共有メモリをマップ
Result SharedMemoryImplByHorizon::Map(void** pOutAddress, NativeHandle handle, void* address, size_t size, os::MemoryPermission myPermission) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(address != NULL, NN_TEXT_OS("nn::os::MapSharedMemory(): address が NULL になっています。"));

    svc::Handle svcHandle(handle);
    auto svcMyPermission = ConvertToSvcMemoryPermission(myPermission);

    auto result = svc::MapSharedMemory(svcHandle,
                                       reinterpret_cast<uintptr_t>(address),
                                       size,
                                       svcMyPermission);

    NN_RESULT_TRY(result)
        NN_RESULT_CATCH( svc::ResultInvalidCurrentMemory )
        {
            // Map しようとした場所のメモリ状態が Free でなかった
            // Kernel が IPC 用に使っている場所だった可能性が高い
            // 別のアドレスでリトライする
            return os::ResultInvalidCurrentMemoryState();
        }
        NN_RESULT_CATCH( svc::ResultInvalidRegion )
        {
            // ResultInvalidRegion の場合も上記と同様
            return os::ResultInvalidCurrentMemoryState();
        }
        NN_RESULT_CATCH_ALL
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    NN_RESULT_END_TRY

    *pOutAddress = address;
    NN_RESULT_SUCCESS;
}

//--------------------------------------------------------------------------
//  共有メモリをアンマップ
void SharedMemoryImplByHorizon::Unmap(NativeHandle handle, void* address, size_t size) NN_NOEXCEPT
{
    svc::Handle svcHandle(handle);

    auto result = svc::UnmapSharedMemory(svcHandle,
                                         reinterpret_cast<uintptr_t>(address),
                                         size);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

//--------------------------------------------------------------------------

}  // namespace detail
}} // namespace nn::os

