﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/os/os_NativeHandleTypes.h>

#include "os_Common.h"

//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

//--------------------------------------------------------------------------
//  MultiWaitHolderType 構造体の実装用の基底クラス（抽象クラス）
//
class   MultiWaitImpl;
class   MultiWaitObjectList;

class   MultiWaitHolderBase
{
    // IntrusiveList 用の frined 宣言
    friend class    MultiWaitImpl;
    friend class    MultiWaitObjectList;

public:
    // コンストラクタ
    MultiWaitHolderBase() NN_NOEXCEPT : m_pMultiWait( NULL ) {}

    // 純粋仮想関数
    virtual TriBool IsSignaled() const  NN_NOEXCEPT = 0;
    virtual TriBool AddToObjectList() NN_NOEXCEPT = 0;
    virtual void RemoveFromObjectList() NN_NOEXCEPT = 0;
    virtual bool GetNativeHandle(NativeHandle* pOut) const NN_NOEXCEPT = 0;
    virtual TimeSpan GetAbsoluteTimeToWakeup() const NN_NOEXCEPT
    {
        return TimeSpan::FromNanoSeconds(INT64_MAX);
    }

    // リンク先の MultiWait へのポインタを設定
    void SetMultiWait(MultiWaitImpl* p) NN_NOEXCEPT
    {
        m_pMultiWait = p;
    }

    // リンク先の MultiWait へのポインタを取得
    MultiWaitImpl* GetMultiWait() const NN_NOEXCEPT
    {
        return m_pMultiWait;
    }

    // 何かしらの MultiWait にリンクしているかを取得
    bool IsLinked() const NN_NOEXCEPT
    {
        return m_pMultiWait != NULL;
    }

    // どの MultiWait にもリンクしていないかを取得
    bool IsNotLinked() const NN_NOEXCEPT
    {
        return m_pMultiWait == NULL;
    }

public:
    // MultiWait リスト および 待ち対象オブジェクトリスト のノード
    util::IntrusiveListNode m_MultiWaitNode;  // MultiWaitType リストのノード
    util::IntrusiveListNode m_WaitObjectNode; // 対象オブジェクトのリストノード

private:
    MultiWaitImpl*  m_pMultiWait;   // MultiWaitType ヘッダへのポインタ
};

//--------------------------------------------------------------------------
//  MultiWaitHolderType 構造体の実装用の基底クラス（ユーザランドオブジェクト用）
//

class MultiWaitHolderOfUserWaitObject : public MultiWaitHolderBase
{
public:
    virtual bool GetNativeHandle(NativeHandle* pOut) const NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(pOut);
        return false;
    }
};

//--------------------------------------------------------------------------
//  MultiWaitHolderType 構造体の実装用の基底クラス（カーネルオブジェクト用）
//

class MultiWaitHolderOfNativeWaitObject : public MultiWaitHolderBase
{
public:
    virtual TriBool AddToObjectList() NN_NOEXCEPT NN_OVERRIDE { return TriBool::Indeterminate; }
    virtual void RemoveFromObjectList() NN_NOEXCEPT NN_OVERRIDE { /* nop */ }
};


}   // namespace detail
}}  // namespace nn::os

