﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_MemoryPermission.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_MemoryHeapManager.h"

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_MemoryMapSelect.h>


//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

namespace {

//----------------------------------------------------------------------------
// SetMemoryPermissionImplUnsafe() 関数
//  指定された beginAddress ～ (beginAddress + regionSize) のメモリ領域の
//  メモリアクセス権を変更する。本関数では svc::QueryMemory() が返す
//  カーネルのメモリ管理単位に分割して svc::SetMemoryPermission() を発行する。
//
void SetMemoryPermissionImplUnsafe(uintptr_t beginAddress, size_t regionSize, svc::MemoryPermission permission) NN_NOEXCEPT
{
    auto address = beginAddress;

    size_t leftSize = regionSize;
    while (leftSize > 0)
    {
        // メモリ情報を取得
        svc::MemoryInfo  memoryInfo;
        svc::PageInfo    pageInfo;
        auto result = svc::QueryMemory(&memoryInfo, &pageInfo, address);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ページ属性変更対象領域のサイズ計算
        size_t size = (memoryInfo.baseAddress + memoryInfo.size) - address;
               size = (size > leftSize) ? leftSize : size;

        // メモリ属性の変更
        if (memoryInfo.permission != permission)
        {
            result = svc::SetMemoryPermission(address, size, permission);
            if (!result.IsSuccess())
            {
                NN_SDK_LOG(NN_TEXT_OS("nn::os::SetMemoryPermission(): メモリアクセス権の変更に失敗しました\n"));
                NN_SDK_LOG("   Address    = 0x%p\n", address);
                NN_SDK_LOG("   Size       = 0x%zx\n", size);
                NN_SDK_LOG("   State      = 0x%x\n", memoryInfo.state);
                NN_SDK_LOG("   Permission = 0x%x -> 0x%x\n", memoryInfo.permission, permission);
                NN_SDK_LOG("   Attribute  = 0x%x\n", memoryInfo.attribute);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        }

        address  += size;
        leftSize -= size;
    }
}

} // namespace


//----------------------------------------------------------------------------
// メモリアクセス権の変更
void SetMemoryPermissionImpl(uintptr_t address, size_t size, MemoryPermission permission) NN_NOEXCEPT
{
    switch (permission)
    {
    case MemoryPermission_None:
            // Horizon: メモリ属性を None に変更
            SetMemoryPermissionImplUnsafe(address, size, svc::MemoryPermission_None );
            break;

    case MemoryPermission_ReadOnly:
            // Horizon: メモリ属性を ReadOnly に変更
            SetMemoryPermissionImplUnsafe(address, size, svc::MemoryPermission_Read );
            break;

    case MemoryPermission_ReadWrite:
            // Horizon: メモリ属性を ReadWrite に変更
            SetMemoryPermissionImplUnsafe(address, size, svc::MemoryPermission_ReadWrite );
            break;

    default:
            NN_ABORT(NN_TEXT_OS("nn::os::SetMemoryPermission(): 不正な permission=0x%x が指定されました。\n"), permission);
            break;
    }
}

//----------------------------------------------------------------------------

}   // namespace detail
}}  // namespace nn::os

