﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"
#include <nn/os/os_Config.h>
#include <nn/diag/text/diag_SdkTextOs.h>

#if !NN_BUILD_CONFIG_OS_SUPPORTS_WIN32
    #error "OS 種別として Win32 が指定されていません。"
#endif

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Windows.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_MemoryHeapCommon.h>

#include "os_Diag.h"
#include "os_MemoryPermissionImpl-os.win32.h"
#include "os_MemoryHeapManager.h"

//---------------------------------------------------------------------------
// const 定数同士の比較に対する Warnng C4127 の抑制
#pragma warning( disable : 4127 )

//--------------------------------------------------------------------------

namespace nn { namespace os {
namespace detail {

namespace {

    static const size_t VritualAllocUnitSize = 64 * 1024;

    inline uintptr_t RoundUp(uintptr_t value, uintptr_t scale) NN_NOEXCEPT
    {
        return (value + scale - 1) & ~(scale - 1);
    }

#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    //-------------------------------------------------------------------------
    // ユーザランドの空間サイズの判定。4GB なら true を返す。
    // TORIAEZU:
    //  正しい判定方法ではないが、32bit 4GB(LAA) モードか否かは、
    //  0x80000000 固定番地への VirtualAlloc() に成功するかどうかで判定する。
    inline bool IsLargeAddressAware() NN_NOEXCEPT
    {
        LPVOID beginAddress = VirtualAlloc(reinterpret_cast<LPVOID>(0x80000000),
                                           VritualAllocUnitSize,
                                           MEM_RESERVE,
                                           PAGE_READWRITE);
        if (beginAddress != 0)
        {
            BOOL result = VirtualFree( beginAddress, 0, MEM_RELEASE );
            NN_ABORT_UNLESS(result != 0);
        }

        return beginAddress == reinterpret_cast<LPVOID>(0x80000000);
    }
#endif

}   // namespace

//--------------------------------------------------------------------------
// 仮想メモリ空間の予約
//
// 仮想メモリ空間を予約し、成否を返す。
//
// Win32 の VirtualAlloc() は 64KB アライメントでメモリ獲得されるため、
// MemoryHeapUnitSize >= 64KB の場合は以下のように対策を行なう。
//
// まず、初めてメモリ空間を予約する場合、size より多めに空間を予約し、
// （MemoryHeapUnitSize でアライメントした先頭アドレス＋size）が確実に
// 予約されるようにする。２回目以降は MemoryHeapUnitSize でアライメント
// されたところから size 分だけ上書きするような感じでメモリ空間を予約する。
// 前後のはみ出た部分の空間は予約したままとする。
//
// MemoryHeapUnitSize < 64KB の場合は、Windows の 64KB アライメントに
// 従ってメモリ空間を予約するため、前後の余分な空間は発生しない。
//
// アドレスやサイズを格納している変数は以下の位置を示す。
//
// |                   |
// +-------------------+ ← beginAddress   ( VirtualAlloc: 64KB 境界 )
// +-------------↑----+ ← alignedAddress ( MemoryHeapUnitSize 境界 )
// |  ↑         ｜    |
// | size    extraSize |
// |  ↓         ｜    |
// +-------------↓----+ ← tailAddress    ( MemoryHeapUnitSize 境界 )
// +-------------------+ ← endAddress     ( VirtualAlloc: 64KB 境界 )
// |                   |
//
// 本関数は既に 1byte でも予約済みの状態で呼ぶことは許されない。
// 予約サイズを変更したい場合は、いったん ReleaseVirtualSpace() で予約を開放
// してから再度本関数を呼ぶ必要がある。
//
bool MemoryHeapManagerImplByWin32::ReserveVirtualSpace(uintptr_t address, size_t size) NN_NOEXCEPT
{
    // 既に予約済みならアボート
    NN_ABORT_UNLESS(m_RealReservedSize == 0);
    NN_ABORT_UNLESS(m_RealReservedAddress == NULL);

    size_t reserveSize = RoundUp(size, MemoryHeapUnitSize);
    if (MemoryHeapUnitSize > VritualAllocUnitSize)
    {
        // 指定サイズ＋アライメント調整分の領域をサイズに加えておく
        reserveSize += MemoryHeapUnitSize - VritualAllocUnitSize;
    }

    // まっさらな状態から予約を試みる
    LPVOID beginAddress = VirtualAlloc( reinterpret_cast<LPVOID>(address),
                                        reserveSize,
                                        MEM_RESERVE,
                                        PAGE_READWRITE);
    // 予約に失敗
    if (beginAddress == NULL)
    {
        return false;
    }

    // 予約に成功
    m_RealReservedAddress = beginAddress;
    m_RealReservedSize    = reserveSize;

    // ヒープのアドレスは一度予約したら、それ以降は変化しない
    m_AlignedReservedHeapAddress = reinterpret_cast<LPVOID>(RoundUp( reinterpret_cast<uintptr_t>(m_RealReservedAddress), MemoryHeapUnitSize ));
    m_AlignedReservedHeapSize    = size;

    return true;
}

//--------------------------------------------------------------------------
// 仮想メモリ空間の開放
//
void MemoryHeapManagerImplByWin32::ReleaseVirtualSpace() NN_NOEXCEPT
{
    if (m_RealReservedAddress != NULL)
    {
        BOOL result = VirtualFree( m_RealReservedAddress, 0, MEM_RELEASE );
        NN_SDK_ASSERT(result, NN_TEXT_OS("内部エラー：メモリ空間の開放に失敗しました。"));
        NN_UNUSED(result);

        m_RealReservedAddress = NULL;
        m_RealReservedSize    = 0;

        m_AlignedReservedHeapAddress = NULL;
        m_AlignedReservedHeapSize    = 0;
    }
}

//--------------------------------------------------------------------------
// コンストラクタ
//  os ライブラリ起動時にヒープ空間を予約してしまう。
//  Windows 環境では以下の３つのユーザランド空間サイズがあるので、
//  それぞれの空間サイズに合わせて予約するヒープ空間サイズを決定する。
//  LAA とは /LARGEADDRESSAWARE モードを指す。
//
//  - 32bit アプリ non-LAA 環境: UserLand   2GB, Heap  1GB
//  - 32bit アプリ LAA     環境: UserLand   4GB, Heap  2GB
//  - 64bit アプリ LAA     環境: UserLand 128TB, Heap 32GB
//
MemoryHeapManagerImplByWin32::MemoryHeapManagerImplByWin32() NN_NOEXCEPT
    : m_RealReservedAddress(NULL),
      m_RealReservedSize(0),
      m_AlignedReservedHeapAddress(0),
      m_AlignedReservedHeapSize(0),
      m_CommitedSize(0)
{
#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    size_t reserveSize  = IsLargeAddressAware() ? DefaultReserveHeapSizeLAA
                                                : DefaultReserveHeapSize;
#elif defined(NN_BUILD_CONFIG_ADDRESS_64)
    size_t reserveSize  = DefaultReserveHeapSize;
#endif

    // 期待サイズを超えない最大サイズの空間予約を試みる。
    // 32bit-LAA 環境の場合のみ、アドレスを固定にして試みる。
    size_t  tryReserveSize   = 0;
    size_t  tryReserveAddend = reserveSize;

    while (tryReserveAddend >= MemoryHeapUnitSize)
    {
        bool result = ReserveVirtualSpace(0, tryReserveSize + tryReserveAddend);
        if (result)
        {
            // 一旦予約を開放する
            ReleaseVirtualSpace();
            tryReserveSize += tryReserveAddend;
            if (tryReserveSize >= reserveSize)
            {
                break;
            }
        }

        tryReserveAddend /= 2;
    }

    // 検査の結果、最終的に予約できた最大サイズを予約する
    bool result = ReserveVirtualSpace(0, tryReserveSize);
    NN_ABORT_UNLESS(result, NN_TEXT_OS("内部エラー：メモリヒープ空間の予約に失敗しました（size=0x%p）"), tryReserveSize);
}


//--------------------------------------------------------------------------
// 物理メモリの獲得
// → 予約済みヒープ空間の先頭から size 分だけのメモリ領域をコミットする。
//
bool MemoryHeapManagerImplByWin32::CommitMemory(size_t size) NN_NOEXCEPT
{
    // Win32: 物理メモリを確保
    LPVOID address = VirtualAlloc( m_AlignedReservedHeapAddress,
                                   static_cast<SIZE_T>( size ),
                                   MEM_COMMIT,
                                   PAGE_READWRITE );

    // NULL が返った場合は false を返す
    if (address == NULL)
    {
        return false;
    }

    // 一致していなければ想定外の状況に陥るのでアボートでチェック
    NN_ABORT_UNLESS(address == m_AlignedReservedHeapAddress);
    return true;
}

//--------------------------------------------------------------------------
// 物理メモリの返却
// → 予約＆コミット済みヒープ領域の末尾にあるはずの、
//    address から size 分の物理メモリを返却する。空間は予約したままとする。
//
void MemoryHeapManagerImplByWin32::DecommitMemory(uintptr_t address, size_t size) NN_NOEXCEPT
{
    // Win32: 物理メモリを返却
    BOOL result = VirtualFree( reinterpret_cast<LPVOID>( address ),
                               static_cast<SIZE_T>( size ),
                               MEM_DECOMMIT );

    // result !=0 なら成功
    NN_SDK_ASSERT( result != 0, NN_TEXT_OS("内部エラー：Windows に物理メモリを返却できません。"));
    NN_UNUSED(result);
}

//-----------------------------------------------------------------------------
//  ヒープサイズの変更
Result MemoryHeapManagerImplByWin32::SetHeapSize(uintptr_t* pOutAddress, size_t size) NN_NOEXCEPT
{
    // メモリ空間が予約できていない
    NN_RESULT_THROW_UNLESS(m_RealReservedAddress, os::ResultOutOfMemory());

    // 現状のコミット済みサイズと比較
    if (size > m_CommitedSize)
    {
        // 追加の物理メモリを確保（size はトータルサイズ）
        if ( !CommitMemory( size ) )
        {
            NN_RESULT_THROW(os::ResultOutOfMemory());
        }
        m_CommitedSize = size;
    }
    else if (size < m_CommitedSize)
    {
        // デコミットするアドレスとサイズを計算
        auto   decommitAddress = reinterpret_cast<uintptr_t>(m_AlignedReservedHeapAddress) + size;
        size_t decommitSize    = m_CommitedSize - size;

        // Win32: 物理メモリを返却
        DecommitMemory( decommitAddress, decommitSize );
        m_CommitedSize = size;
    }

    *pOutAddress = reinterpret_cast<uintptr_t>( m_AlignedReservedHeapAddress );
    NN_RESULT_SUCCESS;
}


}   // namespace detail
}}  // namespace nn::os

//  TORIAEZU: Win32 環境では自動的に呼ぶようにする
//  TORIAEZU: Win32 環境用のスタートアップが出来るまでの暫定対応
extern "C" void nnosInitialize(void);
NN_OS_DETAIL_ATTACH_EARLY_INITIALIZER( nnosInitialize );

