﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "os_Common.h"

#include <nn/nn_Windows.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <cstring>
#include <cstdlib>

#include <nn/nn_Macro.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/os/os_Config.h>
#include <nn/os/os_Argument.h>


namespace nn { namespace os {

    void SetHostArgc(int    argc) NN_NOEXCEPT;
    void SetHostArgv(char** argv) NN_NOEXCEPT;

}}   // namespace nn::os

namespace nn { namespace os {
    NN_NORETURN void QuickExit() NN_NOEXCEPT;
}}

extern "C" void nnMain();

// TORIAEZU: 無効にしておく（SIGLO-6425 および PR#2496 参照）
#undef  NN_OS_ENABLE_MAIN_FOR_UNICODE

#if defined(NN_OS_ENABLE_MAIN_FOR_UNICODE)
namespace {

//----------------------------------------------------------------------------
//  wargv[n] (0 <= n < argc) の各要素のワイドコード文字列を UTF-8 に変換し、
//  pOutArgv[n] にその格納先アドレスをセットする。変換先の文字列を格納する
//  領域は自動的に new され、そのアドレスがセットされる。
//  pOutArgv[n] が指す領域の開放は呼び出し元の責任で行なわなければならない。
//
//  返値は変換に成功した n の数を返す。
//  pOutArgv[] には 0～(n-1) までの変換に成功した結果が格納された状態で返る。
//----------------------------------------------------------------------------

int ConvertWcharToUtf8(char** pOutArgv, int argc, wchar_t** wargv) NN_NOEXCEPT
{
    // ワイド文字をマルチバイト文字へ変換
    for (int i=0; i<argc; ++i)
    {
        auto* wstr = reinterpret_cast<const uint16_t*>( wargv[i] );
        int   wlen = wcslen( wargv[i] );

        int  len;
        auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&len, wstr, wlen);
        if (!(result == nn::util::CharacterEncodingResult_Success))
        {
            return i;
        }

        char* dst = new char[ len + 1 ];
        result = nn::util::ConvertStringUtf16NativeToUtf8(dst, len, wstr, wlen);
        if (!(result == nn::util::CharacterEncodingResult_Success))
        {
            delete [] dst;
            return i;
        }
        dst[len]    = '\0';

        pOutArgv[i] = dst;
    }

    return argc;
}

}   // namespace <anonymous>
#endif  // NN_OS_ENABLE_MAIN_FOR_UNICODE

//--------------------------------------------------------------------------
//  Windows コンソールアプリケーション用の main 関数（ANSI 用）です。
//  argc/argv 引数を保存して nnMain() を呼出します。
//--------------------------------------------------------------------------

int main(int argc, char** argv)
{
    nn::os::SetHostArgc(argc);
    nn::os::SetHostArgv(argv);

    nnMain();

    nn::os::QuickExit();
}

#if defined(NN_OS_ENABLE_MAIN_FOR_UNICODE)
//--------------------------------------------------------------------------
//  Windows コンソールアプリケーション用の wmain 関数（UNICODE 用）です。
//  argc/argv 引数を保存して nnMain() を呼出します。
//--------------------------------------------------------------------------

int wmain(int argc, wchar_t** wargv)
{
    char** argv = new char*[argc];
    std::memset(argv, 0, sizeof(char*) * argc);

    int count = ConvertWcharToUtf8(argv, argc, wargv);
    if (count < argc)
    {
        NN_SDK_LOG(NN_TEXT_OS("wargv[%d] に UTF-8 に変換できない文字がありました\n"), count);
    }
    else
    {
        nn::os::SetHostArgc(argc);
        nn::os::SetHostArgv(argv);
    }

    nnMain();

    for (int i=0; i<argc; ++i)
    {
        delete [] argv[i];
    }
    delete [] argv;

    nn::os::QuickExit();
}
#endif  // NN_OS_ENABLE_MAIN_FOR_UNICODE

//--------------------------------------------------------------------------
//  Windows GUI アプリケーション用の WinMain 関数（ANSI 用）です。
//  argc/argv 引数をセットして nnMain() を呼出します。
//--------------------------------------------------------------------------

int WINAPI WinMain(
    HINSTANCE hInstance,      // 現在のインスタンスのハンドル（別途取得可能）
    HINSTANCE hPrevInstance,  // Win32 アプリケーションでは常に NULL
    LPSTR     lpCmdLine,      // コマンドライン
    int       nCmdShow)       // 表示状態
{
    NN_UNUSED(hInstance);
    NN_UNUSED(hPrevInstance);
    NN_UNUSED(lpCmdLine);
    NN_UNUSED(nCmdShow);

    // lpCmdLine と同じものは GetCommandLine() で取得可能なため、
    // ここでは引数ごとに分解されたものを渡す。詳細は以下を参照。
    // https://msdn.microsoft.com/ja-jp/library/dn727674.aspx
    nn::os::SetHostArgc(__argc);
    nn::os::SetHostArgv(__argv);

    nnMain();

    nn::os::QuickExit();
}

#if defined(NN_OS_ENABLE_MAIN_FOR_UNICODE)
//--------------------------------------------------------------------------
//  Windows GUI アプリケーション用の wWinMain 関数（UNICODE 用）です。
//  argc/argv 引数をセットして nnMain() を呼出します。
//--------------------------------------------------------------------------

int WINAPI wWinMain(
    HINSTANCE hInstance,      // 現在のインスタンスのハンドル（別途取得可能）
    HINSTANCE hPrevInstance,  // Win32 アプリケーションでは常に NULL
    PWSTR     pCmdLine,       // コマンドライン
    int       nCmdShow)       // 表示状態
{
    NN_UNUSED(hInstance);
    NN_UNUSED(hPrevInstance);
    NN_UNUSED(pCmdLine);
    NN_UNUSED(nCmdShow);

    // lpCmdLine と同じものは GetCommandLineW() で取得可能なため、
    // ここでは引数ごとに分解されたものを渡す。詳細は以下を参照。
    // https://msdn.microsoft.com/ja-jp/library/dn727674.aspx

    int    argc = __argc;
    char** argv = new char*[argc];
    std::memset(argv, 0, sizeof(char*) * argc);

    int count = ConvertWcharToUtf8(argv, argc, __wargv);
    if (count < argc)
    {
        NN_SDK_LOG(NN_TEXT_OS("__wargv[%d] に UTF-8 に変換できない文字がありました\n"), count);
    }
    else
    {
        nn::os::SetHostArgc(argc);
        nn::os::SetHostArgv(argv);
    }

    nnMain();

    for (int i=0; i<argc; ++i)
    {
        delete [] argv[i];
    }
    delete [] argv;

    nn::os::QuickExit();
}
#endif  // NN_OS_ENABLE_MAIN_FOR_UNICODE

//  TORIAEZU: Win32 環境では自動的に呼ぶようにする
//  TORIAEZU: Win32 環境用のスタートアップが出来るまでの暫定対応
//
//  【重要】
//      os の機能を全く使っていないアプリやテストなどでも、
//      Win32 環境では nnosInitialize() の呼出しを保証する必要がある。
//      そこで、main 関数に引きずられて libnn_os が必ずリンクされるように、
//      ここに専用のマクロを置いておく。
extern "C" void nnosInitialize();
NN_OS_DETAIL_ATTACH_EARLY_INITIALIZER( nnosInitialize );

