﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/diag/text/diag_SdkTextOs.h>
#include <nn/nn_Result.h>
#include <nn/util/util_TypedStorage.h>
#include <nn/os/os_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_EventCommon.h>
#include <nn/os/os_SystemEventTypes.h>

#include "os_Diag.h"
#include "os_Common.h"
#include "os_TimeoutHelper.h"
#include "os_InterProcessEventImpl.h"
#include "os_MultipleWaitHelper.h"
#include "os_MultipleWaitHolderOfInterProcessEvent.h"
#include "os_MultipleWaitHolderImpl.h"
#include "os_InterProcessEvent.h"


//---------------------------------------------------------------------------
//  C++ 関数の定義
//---------------------------------------------------------------------------

namespace nn { namespace os { namespace detail {

//---------------------------------------------------------------------------

namespace {

inline void SetupInterProcessEventType(InterProcessEventType* event, NativeHandle readableHandle, bool isReadableHandleManaged, NativeHandle writableHandle, bool isWritableHandleManaged, EventClearMode clearMode) NN_NOEXCEPT
{
    // メンバの初期化
    event->_readableHandle          = readableHandle;
    event->_isReadableHandleManaged = isReadableHandleManaged;
    event->_writableHandle          = writableHandle;
    event->_isWritableHandleManaged = isWritableHandleManaged;

    event->_autoClear = (clearMode == EventClearMode_AutoClear);

    // コンストラクタを呼ぶ
    new( &event->_multiWaitObjectList ) detail::MultiWaitObjectList;

    // プロセス間同期イベントを Initialized 状態へ（最後に行なう）
    event->_state = InterProcessEventType::State_Initialized;
}

}   // namespace

//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトの生成
Result CreateInterProcessEvent(InterProcessEventType* event, EventClearMode clearMode) NN_NOEXCEPT
{
    // イベントの生成
    NativeHandle    writableHandle;
    NativeHandle    readableHandle;
    auto result = detail::InterProcessEventImpl::Create(&writableHandle, &readableHandle);
    if (!result.IsSuccess())
    {
        return result;
    }

    // 初期化
    SetupInterProcessEventType(event, readableHandle, true, writableHandle, true, clearMode);
    return nn::ResultSuccess();
}


//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトの破棄
void DestroyInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == InterProcessEventType::State_Initialized, NN_TEXT_OS("指定されたプロセス間同期イベントは初期化されていません。") );

    event->_state  = InterProcessEventType::State_NotInitialized;

    if (event->_isReadableHandleManaged)
    {
        if (event->_readableHandle != InvalidNativeHandle)
        {
            detail::InterProcessEventImpl::Close(event->_readableHandle);
        }
        event->_isReadableHandleManaged = false;
    }

    if (event->_isWritableHandleManaged)
    {
        if (event->_writableHandle != InvalidNativeHandle)
        {
            detail::InterProcessEventImpl::Close(event->_writableHandle);
        }
        event->_isWritableHandleManaged = false;
    }

    // デストラクタ呼出し
    Get(event->_multiWaitObjectList).~MultiWaitObjectList();
}


//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトへのハンドルのアタッチ
void AttachInterProcessEvent(InterProcessEventType* event, NativeHandle readableHandle, bool isReadableHandleManaged, NativeHandle writableHandle, bool isWritableHandleManaged, EventClearMode clearMode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( readableHandle != InvalidNativeHandle ||
                     writableHandle != InvalidNativeHandle,
        NN_TEXT_OS("いずれかのハンドルは有効な値でなければなりません。"));

    // 初期化
    SetupInterProcessEventType(event, readableHandle, isReadableHandleManaged, writableHandle, isWritableHandleManaged, clearMode);
}


//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトからの ReadableHandle のデタッチ
NativeHandle DetachReadableHandleOfInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == InterProcessEventType::State_Initialized, NN_TEXT_OS("指定されたプロセス間同期イベントは初期化されていません。") );

    // ハンドルを切り離す
    NativeHandle    handle = event->_readableHandle;

    event->_readableHandle          = InvalidNativeHandle;
    event->_isReadableHandleManaged = false;

    return handle;
}


//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトからの WritableHandle のデタッチ
NativeHandle DetachWritableHandleOfInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == InterProcessEventType::State_Initialized, NN_TEXT_OS("指定されたプロセス間同期イベントは初期化されていません。") );

    // ハンドルを切り離す
    NativeHandle    handle = event->_writableHandle;

    event->_writableHandle          = InvalidNativeHandle;
    event->_isWritableHandleManaged = false;

    return handle;
}


//---------------------------------------------------------------------------
//  InterProcessEventType がシグナル状態になるまで待機
void WaitInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == InterProcessEventType::State_Initialized, NN_TEXT_OS("指定されたプロセス間同期イベントは初期化されていません。") );

    // ターゲット依存の待機処理
    detail::InterProcessEventImpl::Wait(event->_readableHandle, event->_autoClear);
}


//---------------------------------------------------------------------------
//  InterProcessEventType がシグナル状態になるまで待機（ポーリング）
bool TryWaitInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == InterProcessEventType::State_Initialized, NN_TEXT_OS("指定されたプロセス間同期イベントは初期化されていません。") );

    // ターゲット依存のポーリング
    return detail::InterProcessEventImpl::TryWait(event->_readableHandle, event->_autoClear);
}


//---------------------------------------------------------------------------
//  InterProcessEventType がシグナル状態になるまで待機（タイムアウト付き）
bool TimedWaitInterProcessEvent(InterProcessEventType* event, TimeSpan timeout) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state == InterProcessEventType::State_Initialized, NN_TEXT_OS("指定されたプロセス間同期イベントは初期化されていません。") );
    NN_SDK_REQUIRES( timeout.GetNanoSeconds() >= 0, NN_TEXT_OS("timeout 値が不正です。") );

    // ターゲット依存のタイムアウト付き待機処理
    return detail::InterProcessEventImpl::TimedWait(event->_readableHandle, event->_autoClear, timeout);
}


//---------------------------------------------------------------------------
//  InterProcessEventType をシグナル状態へセット
void SignalInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state != InterProcessEventType::State_NotInitialized, NN_TEXT_OS("指定されたプロセス間同期イベントは生成もしくは初期化されていません。") );

    // ターゲット依存のシグナル処理
    detail::InterProcessEventImpl::Signal(event->_writableHandle);
}


//---------------------------------------------------------------------------
//  InterProcessEventType を非シグナル状態へクリア
void ClearInterProcessEvent(InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state != InterProcessEventType::State_NotInitialized, NN_TEXT_OS("指定されたプロセス間同期イベントは生成もしくは初期化されていません。") );

    // ターゲット依存のクリア処理
    // readableHandle と writableHandle のいずれか有効なハンドルで行なう
    // 両方は無効なハンドルの場合には他の API と同じように Impl の中でアサート
    auto handle = event->_readableHandle;
    if (handle == InvalidNativeHandle)
    {
        handle = event->_writableHandle;
    }
    detail::InterProcessEventImpl::Clear(handle);
}


//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトの ReadableHandle を取得する
NativeHandle GetReadableHandleOfInterProcessEvent(const InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state != InterProcessEventType::State_NotInitialized, NN_TEXT_OS("指定されたプロセス間同期イベントは生成もしくは初期化されていません。") );

    return event->_readableHandle;
}


//---------------------------------------------------------------------------
//  InterProcessEventType オブジェクトの WritableHandle を取得する
NativeHandle GetWritableHandleOfInterProcessEvent(const InterProcessEventType* event) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( event->_state != InterProcessEventType::State_NotInitialized, NN_TEXT_OS("指定されたプロセス間同期イベントは生成もしくは初期化されていません。") );

    return event->_writableHandle;
}


//---------------------------------------------------------------------------

}}} // namespace nn::os::detail

