﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_DeltaApplier.h>
#include <nn/ncm/detail/ncm_Log.h>

namespace nn { namespace ncm {

NN_DEFINE_STATIC_CONSTANT(const int64_t DeltaApplier::ExtendSizeMax);

/**
* @brief   コンストラクタです。
*/
DeltaApplier::DeltaApplier() NN_NOEXCEPT
    : m_pStoragePatch(nullptr),
      m_Command(0),
      m_CommandOffset(0),
      m_WriteCommandOffset(0),
      m_WriteCommandSize(0)
{
    std::memset(&m_PlaceHolderIdPatch, 0, sizeof(m_PlaceHolderIdPatch));
    std::memset(&m_Header, 0, sizeof(m_Header));
    std::memset(&m_StatusData, 0, sizeof(m_StatusData));
}

/**
* @brief   デストラクタです。
*/
DeltaApplier::~DeltaApplier() NN_NOEXCEPT
{
}

/**
* @brief   初期化します。
*
* @param[out]  pOutOffsetDelta パッチ間差分データの開始オフセット
* @param[in]   pFilePatch                    パッチ間差分を適用するパッチファイル
* @param[in]   pStatus                       パッチ間差分の適用状態を保存するバッファー
* @param[in]   sizeStatus                    パッチ間差分の適用状態を保存するバッファーの有効なデータのサイズ
*
* @retval  ResuleSuccess                       正常に処理が終了しました。
* @retval  ncm:::ResultInvalidApplyDeltaStatus パッチ間差分の適用状態が異常です。
*
* @pre
*      - pOutOffsetDelta != nullptr
*      - pStoragePatch != nullptr
*      - pStatus != nullptr
*/
Result DeltaApplier::Initialize(
    int64_t* pOutOffsetDelta,
    ncm::ContentStorage* pStoragePatch,
    const ncm::PlaceHolderId& placeHolderIdPatch,
    const char* pStatus,
    size_t sizeStatus
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutOffsetDelta != nullptr);
    NN_SDK_REQUIRES(pStoragePatch != nullptr);
    NN_SDK_REQUIRES(pStatus != nullptr);

    m_pStoragePatch = pStoragePatch;
    m_PlaceHolderIdPatch = placeHolderIdPatch;
    m_Command = 0;
    m_CommandOffset = 0;
    m_WriteCommandOffset = 0;
    m_WriteCommandSize = 0;

    if (sizeStatus == 0)
    {
        // 前回の情報なし
        std::memset(&m_StatusData, 0, sizeof(m_StatusData));
        std::memset(&m_Header, 0, sizeof(m_Header));
    }
    else
    {
        NN_RESULT_DO(SetStatus(pStatus, sizeStatus));
    }

    *pOutOffsetDelta = m_StatusData.offsetDelta;

    NN_RESULT_SUCCESS;
}

/**
* @brief   パッチ間差分を適用します。
*
* @param[out]  pOutStatus          パッチ間差分の適用状態を保存するバッファー
* @param[in]   sizeStatus          パッチ間差分の適用状態を保存するバッファーのサイズ
* @param[in]   pDelta              パッチ間差分データ
* @param[in]   sizeDelta           パッチ間差分データのサイズ
*
* @retval  ResuleSuccess                 正常に処理が終了しました。
* @retval  ncm::ResultInvalidDeltaFormat パッチ間差分が異常です。
* @retval  上記以外                      パッチ間差分の適用に失敗しました。
*
* @pre
*      - pOutStatus != nullptr
*      - StatusSize <= sizeStatus
*      - pDelta != nullptr
*      - 0 < sizeDelta
*/
Result DeltaApplier::ApplyDelta(
    size_t* pOutProcessedSize,
    char* pOutStatus,
    size_t sizeStatus,
    const void* pDelta,
    size_t sizeDelta
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutStatus != nullptr);
    NN_SDK_REQUIRES(StatusSize <= sizeStatus);
    NN_SDK_REQUIRES(pDelta != nullptr);
    NN_SDK_REQUIRES(0 < sizeDelta);

    size_t offsetBuffer = 0;

    // ヘッダー部分の処理
    if (m_StatusData.offsetDelta < fssystem::DeltaHeader::Size)
    {
        // 既知のヘッダーデータであればコピーする
        if (m_StatusData.offsetDelta < sizeof(m_Header))
        {
            auto sizeCopy = std::min<size_t>(
                sizeof(m_Header) - static_cast<size_t>(m_StatusData.offsetDelta),
                sizeDelta
                );
            std::memcpy(
                reinterpret_cast<unsigned char*>(&m_Header) + m_StatusData.offsetDelta,
                pDelta,
                sizeCopy
            );
        }

        // ヘッダー部分だけスキップ (ヘッダーより後の部分はスキップしない)
        {
            auto sizeMove = std::min<size_t>(
                static_cast<size_t>(fssystem::DeltaHeader::Size - m_StatusData.offsetDelta),
                sizeDelta - offsetBuffer
                );
            m_StatusData.offsetDelta += sizeMove;
            offsetBuffer += sizeMove;
        }

        // ヘッダー部分の読み取りが完了したら、ヘッダー部分を検証
        if (m_StatusData.offsetDelta == fssystem::DeltaHeader::Size)
        {
            NN_RESULT_DO(VerifyHeader());
        }
    }

    // ヘッダーのオフセット情報に従い、データがくるまでバッファーをスキップ
    if (m_StatusData.offsetDelta < m_Header.offsetBody)
    {
        auto sizeSeek = std::min<size_t>(static_cast<size_t>(m_Header.offsetBody - m_StatusData.offsetDelta), sizeDelta - offsetBuffer);
        m_StatusData.offsetDelta += sizeSeek;
        offsetBuffer += sizeSeek;
    }

    int64_t currentDestinationSize;
    NN_RESULT_DO(m_pStoragePatch->GetSize(&currentDestinationSize, m_PlaceHolderIdPatch));

    while (offsetBuffer < sizeDelta)
    {
        // ヘッダーで要求された以上のデータが来たらエラーと見做す
        NN_RESULT_THROW_UNLESS(m_StatusData.offsetDelta + static_cast<int64_t>(sizeDelta - offsetBuffer) <= m_Header.offsetBody + m_Header.sizeBody, ResultInvalidDeltaFormat());

        // コマンド先頭ならコマンド データを受けとり
        if (m_CommandOffset == 0)
        {
            m_Command = reinterpret_cast<const unsigned char*>(pDelta)[offsetBuffer];
            m_WriteCommandOffset = 0;
            m_WriteCommandSize = 0;
            ++m_CommandOffset;
            ++offsetBuffer;
            if (offsetBuffer >= sizeDelta)
            {
                break;
            }
        }

        auto commandType = fssystem::DeltaCommand::GetCommandType(m_Command);

        switch (commandType)
        {
        case fssystem::DeltaCommandType::Write:
            {
                auto sizeDataSize = fssystem::DeltaCommandWrite::GetSizeOfDataSize(m_Command);
                auto sizeDataOffset = fssystem::DeltaCommandWrite::GetSizeOfDataOffset(m_Command);
                auto sizeCommand = 1 + sizeDataSize + sizeDataOffset;

                // データはコマンドのデータサイズ定義部分
                for (int64_t i = m_CommandOffset - 1; i < sizeDataSize && offsetBuffer < sizeDelta; ++i)
                {
                    m_WriteCommandSize |= (static_cast<int64_t>(reinterpret_cast<const unsigned char*>(pDelta)[offsetBuffer]) << (8 * i));
                    ++m_CommandOffset;
                    ++offsetBuffer;
                }

                // データはコマンドのデータオフセット定義部分
                for (int64_t i = m_CommandOffset - (1 + sizeDataSize); i < sizeDataOffset && offsetBuffer < sizeDelta; ++i)
                {
                    m_WriteCommandOffset |= (static_cast<int64_t>(reinterpret_cast<const unsigned char*>(pDelta)[offsetBuffer]) << (8 * i));
                    ++m_CommandOffset;
                    ++offsetBuffer;
                }

                // データがなければコマンド処理を抜ける (その後、ループから抜ける)
                if (offsetBuffer >= sizeDelta)
                {
                    break;
                }

                // 書き込みサイズによってシークのみか、シーク + 書き込みか、判断
                if (m_WriteCommandSize == 0)
                {
                    // シークのみ
                    if (m_StatusData.offsetPatch + m_WriteCommandOffset > m_Header.sizeDestination)
                    {
                        NN_RESULT_THROW(ResultInvalidDeltaFormat());
                    }

                    m_StatusData.offsetDelta += sizeCommand;
                    m_StatusData.offsetPatch += m_WriteCommandOffset;

                    // コマンドの実行を終えたのでステータスを更新する
                    m_CommandOffset = 0;
                }
                else
                {
                    // シーク + 書き込み
                    if (m_StatusData.offsetPatch + m_WriteCommandOffset + m_WriteCommandSize > m_Header.sizeDestination)
                    {
                        NN_RESULT_THROW(ResultInvalidDeltaFormat());
                    }

                    auto sizeWrite = static_cast<size_t>(std::min(
                        static_cast<int64_t>(sizeDelta - offsetBuffer),
                        static_cast<int64_t>(m_WriteCommandSize - (m_CommandOffset - sizeCommand))
                    ));

                    // 16KB 以上の書き込み && 次読み込んだ時の書き込み先アライメントが 16KB 境界にない && 今回の書き込みでコマンドが終わらない場合
                    // 末尾のサイズを調整して、次の書き込みアドレスが 16KB 境界になるようにする
                    // そして残りは次に回す
                    bool isSizeModified = false;
                    size_t modifiedSize = 0;
                    const size_t WriteAlignment = 16 * 1024;
                    auto nextWriteAddress = (m_StatusData.offsetPatch + m_WriteCommandOffset + (m_CommandOffset - sizeCommand)) + sizeWrite;
                    if (sizeWrite >= WriteAlignment
                        && !(util::is_aligned(nextWriteAddress, WriteAlignment))
                        && sizeWrite != static_cast<size_t>(m_WriteCommandSize - (m_CommandOffset - sizeCommand)))
                    {
                        isSizeModified = true;
                        modifiedSize = nextWriteAddress % WriteAlignment;
                        NN_DETAIL_NCM_TRACE("[DeltaApplier] Modify size: %llu -> %llu, next write addr %llx -> %llx\n", sizeWrite, sizeWrite - modifiedSize, nextWriteAddress, nextWriteAddress - modifiedSize);
                        sizeWrite -= modifiedSize;
                    }

                    const auto offsetWrite = m_StatusData.offsetPatch + m_WriteCommandOffset + (m_CommandOffset - sizeCommand);

                    if( currentDestinationSize < offsetWrite + static_cast<int64_t>(sizeWrite) )
                    {
                        auto nextDestinationSize = std::min(offsetWrite + std::max(static_cast<int64_t>(sizeWrite), ExtendSizeMax), m_Header.sizeDestination);
                        NN_DETAIL_NCM_TRACE("[DeltaApplier] Extend %lld -> %lld\n", currentDestinationSize, nextDestinationSize);

                        NN_RESULT_DO(m_pStoragePatch->FlushPlaceHolder());
                        NN_RESULT_TRY(m_pStoragePatch->SetPlaceHolderSize(m_PlaceHolderIdPatch, nextDestinationSize));
                            NN_RESULT_CATCH_CONVERT(fs::ResultUsableSpaceNotEnough, ResultNotEnoughSpaceToApplyDelta());
                        NN_RESULT_END_TRY;
                        currentDestinationSize = nextDestinationSize;
                    }

                    NN_RESULT_DO(m_pStoragePatch->WritePlaceHolder(
                        m_PlaceHolderIdPatch,
                        offsetWrite,
                        reinterpret_cast<const char*>(pDelta) + offsetBuffer,
                        sizeWrite
                    ));

                    offsetBuffer += sizeWrite;
                    m_CommandOffset += sizeWrite;

                    if (isSizeModified)
                    {
                        // 適用状態を取得する
                        GetStatus(pOutStatus, sizeStatus);
                        *pOutProcessedSize = sizeDelta - modifiedSize;
                        NN_RESULT_SUCCESS;
                    }

                    // 全て書き終えたのでステータスを更新する
                    if(m_CommandOffset == sizeCommand + m_WriteCommandSize)
                    {
                        m_StatusData.offsetDelta += m_CommandOffset;
                        m_StatusData.offsetPatch += m_WriteCommandOffset + m_WriteCommandSize;
                        m_CommandOffset = 0;
                    }
                }
            }
            break;

        default:
            NN_RESULT_THROW(ResultInvalidDeltaFormat());
        }
    }

    // 適用状態を取得する
    GetStatus(pOutStatus, sizeStatus);
    *pOutProcessedSize = sizeDelta;

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
* @brief   パッチ間差分の適用状態をバッファーから読み込みます。
*
* @param[in]   pStatus             パッチ間差分の適用状態バッファー
* @param[in]   sizeStatus          パッチ間差分の適用状態バッファーのサイズ
*
* @retval  ResuleSuccess                      正常に処理が終了しました。
* @retval  ncm::ResultInvalidApplyDeltaStatus パッチ間差分の適用状態が異常です。
* @retval  上記以外                           読み込みに失敗しました。
*
* @pre
*      - pStatus != nullptr
*/
Result DeltaApplier::SetStatus(const char* pStatus, size_t sizeStatus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pStatus != nullptr);
    NN_RESULT_THROW_UNLESS(sizeStatus >= sizeof(m_StatusData), ResultInvalidApplyDeltaStatus());

    std::memcpy(&m_StatusData, pStatus, sizeof(m_StatusData));

    // ステータス情報の検証
    NN_RESULT_THROW_UNLESS(
        (m_StatusData.sizeHeader == static_cast<int32_t>(sizeof(m_Header)) &&
         m_StatusData.offsetHeader >= static_cast<int32_t>(sizeof(m_StatusData)) &&
         m_StatusData.offsetDelta >= 0 &&
         m_StatusData.offsetPatch >= 0 &&
         sizeStatus >= static_cast<size_t>(m_StatusData.offsetHeader + m_StatusData.sizeHeader)),
        ResultInvalidApplyDeltaStatus()
    );

    std::memcpy(&m_Header, pStatus + m_StatusData.offsetHeader, m_StatusData.sizeHeader);
    NN_RESULT_THROW_UNLESS(VerifyHeader().IsSuccess(), ResultInvalidApplyDeltaStatus());

    NN_RESULT_SUCCESS;
}

/**
* @brief   パッチ間差分の適用状態をバッファーに書き込みます。
*
* @param[out]  pOutStatus          パッチ間差分の適用状態を保存するバッファー
* @param[in]   sizeStatus          パッチ間差分の適用状態を保存するバッファーのサイズ
*
* @pre
*      - pOutStatus != nullptr
*      - StatusSize <= sizeStatus
*/
void DeltaApplier::GetStatus(char* pOutStatus, size_t sizeStatus) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(pOutStatus != nullptr);
    NN_SDK_REQUIRES(StatusSize <= sizeStatus);

    // パッチ ヘッダーの保存
    m_StatusData.offsetHeader = sizeof(m_StatusData);
    m_StatusData.sizeHeader = sizeof(m_Header);

    // 書き込み
    std::memset(pOutStatus, 0, sizeStatus);
    std::memcpy(pOutStatus, &m_StatusData, sizeof(m_StatusData));
    std::memcpy(pOutStatus + m_StatusData.offsetHeader, &m_Header, sizeof(m_Header));
}

/**
* @brief   パッチ間差分のヘッダーを検証します。
*
* @retval  ResuleSuccess                 正常に処理が終了しました。
* @retval  ncm::ResultInvalidDeltaFormat 検証に失敗しました。
*/
Result DeltaApplier::VerifyHeader() NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(
        m_Header.signature == static_cast<uint32_t>(fssystem::DeltaHeader::Signature::V0),
        ResultInvalidDeltaFormat()
    );
    NN_RESULT_THROW_UNLESS(
        (m_Header.sizeSource >= 0 &&
         m_Header.sizeDestination >= 0 &&
         m_Header.offsetBody >= fssystem::DeltaHeader::Size &&
         m_Header.sizeBody > 0),
        ResultInvalidDeltaFormat()
    );

    NN_RESULT_SUCCESS;
}

}}
