﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>

#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/ncm/detail/ncm_Log.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ContentIdUtil.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentManagementUtil.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/ncm/ncm_ApplyDeltaTaskBase.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include "ncm_FileSystemUtility.h"

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Content.h>


namespace nn { namespace ncm {

namespace{
    const char* FragmentName = "fragment";

    Result GetFragmentPath(Path* outPath, ContentMetaDatabase* db, ContentStorage* storage, const ContentMetaKey& key) NN_NOEXCEPT
    {
        int numContent;
        ContentInfo contentInfo;
        for(int i = 0; i < 16; ++i)
        {
            NN_RESULT_DO(db->ListContentInfo(&numContent, &contentInfo, 1, key, i));
            if(numContent == 0)
            {
                break;
            }
            if(contentInfo.type == ContentType::DeltaFragment)
            {
                storage->GetPath(outPath, contentInfo.id);
                NN_RESULT_SUCCESS;
            }
        }
        // TODO: 正しくない
        NN_RESULT_THROW(ResultContentMetaNotFound());
    }

    Result Delete(ContentMetaDatabase* db, ContentStorage* storage, const ContentMetaKey& key) NN_NOEXCEPT
    {
        // meta 以外の fragment が残存している場合もあるので、一通り削除
        nn::ncm::ContentInfo buffer[16];
        int offset = 0;
        for (;;)
        {
            int outCount;
            const int count = sizeof(buffer) / sizeof(nn::ncm::ContentInfo);
            NN_RESULT_DO(db->ListContentInfo(&outCount, buffer, count, key, offset));
            for (int i = 0; i < outCount; i++)
            {
                NN_RESULT_TRY(storage->Delete(buffer[i].id))
                    NN_RESULT_CATCH(nn::ncm::ResultContentNotFound) {}
                NN_RESULT_END_TRY
            }
            if (outCount < count)
            {
                break;
            }
            offset += outCount;
        }
        NN_RESULT_DO(db->Remove(key));
        NN_RESULT_DO(db->Commit());
        NN_RESULT_SUCCESS;
    }
    Result DeleteDeltaFragments(ContentMetaDatabase* db, ContentStorage* storage, const ContentMetaKey& key) NN_NOEXCEPT
    {
        nn::ncm::ContentInfo buffer[16];
        int offset = 0;
        for (;;)
        {
            int outCount;
            const int count = sizeof(buffer) / sizeof(nn::ncm::ContentInfo);
            NN_RESULT_DO(db->ListContentInfo(&outCount, buffer, count, key, offset));
            for (int i = 0; i < outCount; i++)
            {
                if (buffer[i].type == ContentType::DeltaFragment)
                {
                    NN_RESULT_TRY(storage->Delete(buffer[i].id))
                        NN_RESULT_CATCH(nn::ncm::ResultContentNotFound) {}
                    NN_RESULT_END_TRY
                }
            }
            if (outCount < count)
            {
                break;
            }
            offset += outCount;
        }
        NN_RESULT_SUCCESS;
    }
    bool IsEmptyPlaceHolder(PlaceHolderId id) NN_NOEXCEPT
    {
        return id.uuid == util::InvalidUuid;
    }

    Result CalculateActualOccupiedSize(int64_t* outSize, ContentStorage* storage, const PlaceHolderId& id) NN_NOEXCEPT
    {
        int64_t size = 0;
        bool has;
        NN_RESULT_DO(storage->HasPlaceHolder(&has, id));
        if (has)
        {
            NN_RESULT_DO(storage->GetSize(&size, id));
        }
        *outSize = size;

        NN_RESULT_SUCCESS;
    }
}

Result ApplyDeltaTaskBase::Cleanup() NN_NOEXCEPT
{
    m_MetaBuffer.Reset();
    {
        ContentStorage storage;
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));
        for (int i = 0; i < m_TaskState->fragmentSetCount; ++i)
        {
            NN_RESULT_TRY(storage.DeletePlaceHolder(m_TaskState->placeHolders[i]))
                NN_RESULT_CATCH(ResultPlaceHolderNotFound) {}
            NN_RESULT_END_TRY
        }
        NN_RESULT_TRY(storage.DeletePlaceHolder(m_TaskState->metaPlaceHolder))
            NN_RESULT_CATCH(ResultPlaceHolderNotFound) {}
        NN_RESULT_END_TRY
    }
    {
        ContentMetaDatabase db;
        ContentStorage storage;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->deltaKey.storageId));
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));

        bool has;
        NN_RESULT_DO(db.Has(&has, m_TaskState->deltaKey.key));
        if (has)
        {
            ContentManagerAccessor accessor(&db, &storage);
            NN_RESULT_DO(accessor.DeleteRedundant(m_TaskState->deltaKey.key, nullptr));
            NN_RESULT_DO(db.Commit());
        }
    }
    CleanupProgress();

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::Execute() NN_NOEXCEPT
{
    auto state = GetProgress().state;
    NN_RESULT_THROW_UNLESS(state == ApplyDeltaProgressState_Prepared || state == ApplyDeltaProgressState_Suspended, ResultUnexecutableState());
    SetProgressState(ApplyDeltaProgressState_Applying);
    for(;;)
    {
        auto fragmentSetIndex  = m_TaskState->fragmentSetIndex;
        auto fragmentIndex     = m_TaskState->fragmentIndex;
        auto fragmentSetOffset = m_TaskState->fragmentSetProcessedSize;
        auto fragmentOffset    = m_TaskState->fragmentProcessedSize;
        auto fragmentSet       = m_DeltaReader->GetFragmentSet(fragmentSetIndex);
        auto fragment          = m_DeltaReader->FindFragmentIndicator(fragmentSetIndex, fragmentIndex);
        auto contentInfo       = m_DeltaReader->GetContentInfo(fragment->contentInfoIndex);
        auto& placeHolder      = m_TaskState->placeHolders[fragmentSetIndex];

        bool completeFragment;
        int64_t appliedSize;
        nn::Result applyResult;
        if (fragmentSet->updateType == UpdateType::ApplyAsDelta)
        {
            applyResult = ApplyDeltaFragment(&completeFragment, &appliedSize, fragmentSetOffset, fragmentOffset, placeHolder, contentInfo->info);
        }
        else
        {
            applyResult = ApplyDataFragment(&completeFragment, &appliedSize, fragmentSet->GetDestinationSize(), fragmentSetOffset, fragmentOffset, placeHolder, contentInfo->info);
        }
        // 容量不足時は再開を可能にするために状態を Suspend に変更する
        NN_RESULT_TRY(applyResult)
            NN_RESULT_CATCH(ResultNotEnoughSpaceToApplyDelta)
            {
                SetProgressState(ApplyDeltaProgressState_Suspended);
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;

        m_TaskState->fragmentProcessedSize = appliedSize;

        if (completeFragment)
        {
            ContentStorage storage;
            NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));

            // fragment の削除
            NN_RESULT_DO(storage.Delete(contentInfo->info.id));
            bool allComplete;
            NN_RESULT_DO(MoveNext(&allComplete));
            if (allComplete)
            {
                SetProgressState(ApplyDeltaProgressState_DeltaApplied);
                break;
            }
        }
        if (IsCancelRequested())
        {
            std::lock_guard<os::Mutex> locker(m_CancelMutex);
            SetProgressState(ApplyDeltaProgressState_Suspended);
            ClearCancelRequest();
            NN_RESULT_THROW(ResultCancelled());
        }
    }

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::Commit(bool doCleanup) NN_NOEXCEPT
{
    for(int i = 0; i < m_TaskState->fragmentSetCount; ++i)
    {
        auto fragmentSet = m_DeltaReader->GetFragmentSet(i);
        auto placeHolder = m_TaskState->placeHolders[i];
        if(fragmentSet->targetContentType == ContentType::Meta)
        {
            // ContentType::Meta は特別
            continue;
        }
        NN_RESULT_DO(Register(i, placeHolder));
    }
    // この時点で m_DeltaReader はもう利用しない。m_MetaBuffer も開放する
    m_DeltaReader.reset();
    m_MetaBuffer.Reset();

    // W/A for SIGLO-63062
    // 2NUP で作成したタスクの場合、m_TaskState->metaPlaceHolder が存在しないため、このあとの処理でエラーになる
    // そのため、metaPlaceHolder が 0 埋めだった場合に Copy をする
    // Commit でコピーはできるだけしたくないので、0 埋めの時のみとする
    if (IsEmptyPlaceHolder(m_TaskState->metaPlaceHolder))
    {
        ContentStorage storage;
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

        m_TaskState->metaPlaceHolder = storage.GeneratePlaceHolderId();
        NN_RESULT_DO(CopyContentMeta(&storage, m_TaskState->metaPlaceHolder));
    }

    // ContentType::Meta の処理
    AutoBuffer contentMetaBuffer;
    ContentMetaKey key;

    // ApplyPatchDeltaTask の場合、先に delta を消しておく
    if (m_TaskState->deltaKey.key.type == ContentMetaType::Patch)
    {
        ContentMetaDatabase db;
        ContentStorage storage;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->deltaKey.storageId));
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));

        ContentManagerAccessor accessor(&db, &storage);
        NN_RESULT_DO(accessor.DeleteRedundant(m_TaskState->deltaKey.key, nullptr));
        NN_RESULT_DO(db.Commit());
    }
    {
        ContentMetaDatabase db;
        ContentStorage storage;
        NN_RESULT_DO(OpenContentMetaDatabase(&db, m_TaskState->sourceKey.storageId));
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

        NN_RESULT_DO(CreateContentMeta(&key, &contentMetaBuffer, m_TaskState->metaPlaceHolder, m_TaskState->metaContentInfo));
        NN_RESULT_DO(storage.Register(m_TaskState->metaPlaceHolder, m_TaskState->metaContentInfo.id));
#if 0
        NN_DETAIL_NCM_TRACE("[ApplyDelta] key: id: %016llx, type: %u, ver.: %u, installType: %u\n", key.id, static_cast<uint32_t>(key.type), key.version, static_cast<uint32_t>(key.installType));
        NN_DETAIL_NCM_TRACE("[ApplyDelta] size: %u\n", contentMetaBuffer.GetSize());
        for(int i = 0; i < static_cast<int>(contentMetaBuffer.GetSize()); ++i)
        {
            if(i % 16 == 0)
            {
                NN_DETAIL_NCM_TRACE("\n[ApplyDelta]");
            }
            NN_DETAIL_NCM_TRACE("%02x", reinterpret_cast<const char*>(contentMetaBuffer.Get())[i]);
        }
        NN_DETAIL_NCM_TRACE("\n");
#endif
        NN_RESULT_DO(db.Set(key, contentMetaBuffer.Get(), contentMetaBuffer.GetSize()));
        NN_RESULT_DO(db.Commit());
    }

    if (doCleanup)
    {
        NN_RESULT_DO(Cleanup());
    }

    SetProgressState(ApplyDeltaProgressState_Commited);

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::Cancel() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> locker(m_CancelMutex);
    auto progress = GetProgress();
    if (progress.state == ApplyDeltaProgressState_Applying)
    {
        m_CancelToken = true;
    }
    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::CalculateOccupiedPlaceholderSize(int64_t* outValue, StorageId storageId) NN_NOEXCEPT
{
    int64_t size = 0;
    if (m_TaskState->sourceKey.storageId == storageId)
    {
        ContentStorage storage;
        NN_RESULT_DO(OpenContentStorage(&storage, storageId));
        for (int i = 0; i < MaxPlaceHolderSize; ++i)
        {
            int64_t occupied;
            NN_RESULT_DO(CalculateActualOccupiedSize(&occupied, &storage, m_TaskState->placeHolders[i]));
            size += occupied;
        }
        {
            int64_t occupied;
            NN_RESULT_DO(CalculateActualOccupiedSize(&occupied, &storage, m_TaskState->metaPlaceHolder));
            size += occupied;
        }
    }
    *outValue = size;

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::CalculateRequiredSizeMaxForPrepareInternal(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES(m_DeltaReader);
    NN_SDK_REQUIRES_NOT_NULL(m_TaskState);

    static const int64_t MarginForRevert = 2 * MaxClusterSize;

    int64_t sizeMax = 0;
    int64_t sizeCurrent = 0;

    auto header = m_DeltaReader->GetHeader();

    for( int fragmentSetIndex = 0; fragmentSetIndex < header->fragmentSetCount; ++fragmentSetIndex )
    {
        auto fragmentSet = m_DeltaReader->GetFragmentSet(fragmentSetIndex);
        if( fragmentSet->updateType == UpdateType::Create )
        {
            sizeCurrent += CalculateRequiredSize(0);
        }
        else
        {
            sizeCurrent += MarginForRevert;
        }

        if( sizeMax < sizeCurrent )
        {
            sizeMax = sizeCurrent;
        }
    }

    {
        ContentStorage storage;
        NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));

        int64_t fileSize;
        NN_RESULT_DO(storage.GetSize(&fileSize, m_TaskState->metaContentInfo.id));

        sizeMax += CalculateRequiredSize(fileSize);
    }

    *outValue = sizeMax;

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::CalculateRequiredSizeMaxForPrepare(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);

    int64_t sizeRequiredSizeMaxForPrepareInternal;
    NN_RESULT_DO(CalculateRequiredSizeMaxForPrepareInternal(&sizeRequiredSizeMaxForPrepareInternal));

    int64_t sizeRequiredSizeMaxForResume;
    NN_RESULT_DO(CalculateRequiredSizeMaxForResume(&sizeRequiredSizeMaxForResume));

    *outValue = sizeRequiredSizeMaxForPrepareInternal + sizeRequiredSizeMaxForResume;

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::CalculateRequiredSizeMaxForResume(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outValue);
    NN_SDK_REQUIRES(m_DeltaReader);
    NN_SDK_REQUIRES_NOT_NULL(m_TaskState);

    int64_t sizeMax = 0;

    auto header = m_DeltaReader->GetHeader();
    int64_t sizeAdded = 0;
    int64_t sizeRemoved = 0;
    for( int fragmentSetIndex = m_TaskState->fragmentSetIndex; fragmentSetIndex < header->fragmentSetCount; ++fragmentSetIndex )
    {
        auto fragmentSet = m_DeltaReader->GetFragmentSet(fragmentSetIndex);

        int64_t sizeChangedByCurrentFragmentSet = std::max<int64_t>(0, fragmentSet->GetDestinationSize() - fragmentSet->GetSourceSize());
        int64_t sizeAddedByCurrentFragmentSet = 0;

        auto fragmentIndex = fragmentSetIndex == m_TaskState->fragmentSetIndex ? m_TaskState->fragmentIndex : 0;
        for( ; fragmentIndex < fragmentSet->fragmentCount; ++fragmentIndex )
        {
            auto fragment = m_DeltaReader->FindFragmentIndicator(fragmentSetIndex, fragmentIndex);
            auto contentInfo = m_DeltaReader->GetContentInfo(fragment->contentInfoIndex);

            // 最悪ケースである、パッチ間差分が全て拡張データである場合を考えてファイルサイズ分加算
            int64_t sizeAddedByCurrentFragment = contentInfo->info.GetSize();
            if( fragmentIndex + 1 < fragmentSet->fragmentCount )
            {
                // 次のフラグメント以降のデータのために拡張が起こりうるので ExtendSizeMax を追加で加算
                sizeAddedByCurrentFragment += DeltaApplier::ExtendSizeMax;
            }

            // 現在の追加サイズを計算して最大値を更新
            int64_t sizeCurrent =
                CalculateRequiredSizeForExtension(
                    sizeAdded
                    + std::min(
                        sizeAddedByCurrentFragmentSet + sizeAddedByCurrentFragment,
                        sizeChangedByCurrentFragmentSet
                    )
                ) - sizeRemoved;
            if( sizeMax < sizeCurrent )
            {
                sizeMax = sizeCurrent;
            }

            // 更新
            sizeAddedByCurrentFragmentSet += contentInfo->info.GetSize();
            if( m_TaskState->sourceKey.storageId == m_TaskState->deltaKey.storageId )
            {
                sizeRemoved += contentInfo->info.GetSize();
            }
        }

        // 更新
        sizeAdded += sizeChangedByCurrentFragmentSet;
    }

    *outValue = sizeMax;

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::InitializeCommon() NN_NOEXCEPT
{
    m_CancelToken = false;

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::MoveNext(bool* outComplete) NN_NOEXCEPT
{
    *outComplete = false;

    auto header = m_DeltaReader->GetHeader();
    auto fragmentSet = m_DeltaReader->GetFragmentSet(m_TaskState->fragmentSetIndex);

    m_TaskState->fragmentSetProcessedSize += m_TaskState->fragmentProcessedSize;
    m_TaskState->fragmentProcessedSize = 0;

    m_TaskState->fragmentIndex++;
    if (m_TaskState->fragmentIndex == fragmentSet->fragmentCount)
    {
        m_TaskState->fragmentIndex = 0;
        m_TaskState->fragmentSetProcessedSize = 0;
        m_TaskState->fragmentSetIndex++;
        if (m_TaskState->fragmentSetIndex == header->fragmentSetCount)
        {
            *outComplete = true;
        }
    }

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::CreatePlaceHolder(int fragmentSetIndex, const PlaceHolderId& placeHolder) NN_NOEXCEPT
{
    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    auto fragmentSet = m_DeltaReader->GetFragmentSet(fragmentSetIndex);
    if (fragmentSet->updateType == UpdateType::Create)
    {
        NN_RESULT_TRY(storage.CreatePlaceHolder(placeHolder, fragmentSet->destinationContentId, 0))
            NN_RESULT_CATCH_CONVERT(fs::ResultUsableSpaceNotEnough, ResultNotEnoughSpaceToApplyDelta());
        NN_RESULT_END_TRY;
    }
    else
    {
        NN_RESULT_TRY(storage.RevertToPlaceHolder(placeHolder, fragmentSet->sourceContentId, fragmentSet->destinationContentId))
            NN_RESULT_CATCH_CONVERT(ResultContentNotFound, ResultContentNotFoundToRevert())
            NN_RESULT_CATCH_CONVERT(fs::ResultUsableSpaceNotEnough, ResultNotEnoughSpaceToApplyDelta());
        NN_RESULT_END_TRY;
        if (fragmentSet->GetSourceSize() > fragmentSet->GetDestinationSize())
        {
            NN_DETAIL_NCM_TRACE("[ApplyDelta] Shrink %lld -> %lld\n", fragmentSet->GetSourceSize(), fragmentSet->GetDestinationSize());
            NN_RESULT_DO(storage.SetPlaceHolderSize(placeHolder, fragmentSet->GetDestinationSize()));
        }
    }
    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::Register(int fragmentSetIndex, const PlaceHolderId& placeHolder) NN_NOEXCEPT
{
    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    auto fragmentSet = m_DeltaReader->GetFragmentSet(fragmentSetIndex);
    NN_RESULT_DO(storage.Register(placeHolder, fragmentSet->destinationContentId));
    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::CreateContentMeta(ContentMetaKey* key, AutoBuffer* buffer, const PlaceHolderId& placeHolder, const ContentInfo& metaContentInfo) NN_NOEXCEPT
{
    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    AutoBuffer meta;
    Path path;
    storage.GetPlaceHolderPath(&path, placeHolder);
    NN_RESULT_DO(ReadContentMetaPath(&meta, path.string));

    PackagedContentMetaReader reader(meta.Get(), meta.GetSize());
    NN_RESULT_DO(buffer->Initialize(reader.CalculateConvertContentMetaSize()));
    reader.ConvertToContentMeta(buffer->Get(), buffer->GetSize(), metaContentInfo);

    *key = reader.GetKey();
    NN_DETAIL_NCM_TRACE("[ApplyDelta] key: id: %016llx, type: %d, ver.: %u\n", key->id, key->type, key->version);
    NN_DETAIL_NCM_TRACE("[ApplyDelta] size: %u\n", buffer->GetSize());

    NN_RESULT_SUCCESS;
}
Result ApplyDeltaTaskBase::CreateContentMeta(ContentMetaKey* key, AutoBuffer* buffer, const ContentId& contentId, const ContentInfo& metaContentInfo) NN_NOEXCEPT
{
    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    AutoBuffer meta;
    Path path;
    storage.GetPath(&path, contentId);
    NN_RESULT_DO(ReadContentMetaPath(&meta, path.string));

    PackagedContentMetaReader reader(meta.Get(), meta.GetSize());
    NN_RESULT_DO(buffer->Initialize(reader.CalculateConvertContentMetaSize()));
    reader.ConvertToContentMeta(buffer->Get(), buffer->GetSize(), metaContentInfo);

    *key = reader.GetKey();
    NN_DETAIL_NCM_TRACE("[ApplyDelta] key: id: %016llx, type: %d, ver.: %u\n", key->id, key->type, key->version);
    NN_DETAIL_NCM_TRACE("[ApplyDelta] size: %u\n", buffer->GetSize());

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::OpenFragment(detail::MountName* pOutMountName, fs::FileHandle* pOutFile, const ContentInfo& fragment) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pOutMountName);
    NN_SDK_ASSERT_NOT_NULL(pOutFile);

    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));

    Path path;
    storage.GetPath(&path, fragment.id);

    NN_DETAIL_NCM_TRACE("[ApplyDelta] fragment nca : %s\n", path.string);
    *pOutMountName = detail::CreateUniqueMountName();

    // delta fragment は、更新後のパッチの PatchId で開くことができる
    // とはいえ、更新前後で PatchId が変わることはないので、更新前の PatchId としておく
    ncm::ProgramId patchId = { m_TaskState->sourceKey.key.id };
    NN_RESULT_DO(fs::MountContent(pOutMountName->string, path.string, patchId, fs::ContentType_Data));

    auto rootDirectoryPath = detail::GetRootDirectoryPath(*pOutMountName);
    char filePath[64];

    auto length = util::SNPrintf(filePath, sizeof(filePath), "%s%s", rootDirectoryPath.string, FragmentName);
    NN_SDK_ASSERT(length < sizeof(filePath));
    NN_RESULT_DO(fs::OpenFile(pOutFile, filePath, fs::OpenMode_Read));
    NN_UNUSED(length);

    NN_DETAIL_NCM_TRACE("[ApplyDelta] Open file: %s\n", filePath);

    NN_RESULT_SUCCESS;
}

void ApplyDeltaTaskBase::CloseFragment(const detail::MountName& mountName, fs::FileHandle file) NN_NOEXCEPT
{
    fs::CloseFile(file);
    fs::Unmount(mountName.string);
}

Result ApplyDeltaTaskBase::ApplyDataFragment(bool* outCompleteFragment, int64_t* outProcessedOffset, int64_t targetDestinationSize, int64_t fragmentSetOffset, int64_t fragmentOffset, const PlaceHolderId& placeHolder, const ContentInfo& fragment) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ApplyBuffer != nullptr, ResultBufferNotEnough());

    *outCompleteFragment = false;

    detail::MountName mountName;
    fs::FileHandle file;
    NN_RESULT_DO(OpenFragment(&mountName, &file, fragment));
    NN_UTIL_SCOPE_EXIT { CloseFragment(mountName, file); };

    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    int64_t currentDestinationSize;
    NN_RESULT_DO(storage.GetSize(&currentDestinationSize, placeHolder));

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, file));

    int64_t totalRead = fragmentOffset;
    while(totalRead < fileSize)
    {
        size_t readSize;
        NN_RESULT_DO(fs::ReadFile(&readSize, file, totalRead, m_ApplyBuffer, m_ApplyBufferSize));
        if( currentDestinationSize < fragmentSetOffset + totalRead + static_cast<int64_t>(readSize) )
        {
            auto nextDestinationSize = std::min(currentDestinationSize + DeltaApplier::ExtendSizeMax, targetDestinationSize);
            NN_DETAIL_NCM_TRACE("[ApplyDelta] Extend %lld -> %lld\n", currentDestinationSize, nextDestinationSize);
            NN_RESULT_DO(storage.FlushPlaceHolder());
            NN_RESULT_TRY(storage.SetPlaceHolderSize(placeHolder, nextDestinationSize))
                NN_RESULT_CATCH_CONVERT(fs::ResultUsableSpaceNotEnough, ResultNotEnoughSpaceToApplyDelta());
            NN_RESULT_END_TRY;
            currentDestinationSize = nextDestinationSize;
        }
        NN_RESULT_DO(storage.WritePlaceHolder(placeHolder, fragmentSetOffset + totalRead, m_ApplyBuffer, readSize));
#if 0
        NN_DETAIL_NCM_TRACE("[ApplyDelta] %lld / %llu, read: %llu\n", totalRead, fileSize, readSize);
#endif
        totalRead += readSize;
        IncrementAppliedDeltaProgress(readSize);

        if (IsCancelRequested())
        {
            break;
        }
    }
    *outProcessedOffset = totalRead;
    // nca サイズと fragment サイズの差分を加算
    if (totalRead == fileSize)
    {
        IncrementAppliedDeltaProgress(fragment.GetSize() - fileSize);
        *outCompleteFragment = true;
    }

    NN_RESULT_SUCCESS;
}

Result ApplyDeltaTaskBase::ApplyDeltaFragment(bool* outCompleteFragment, int64_t* outProcessedOffset, int64_t fragmentSetOffset, int64_t fragmentOffset, const PlaceHolderId& placeHolder, const ContentInfo& fragment) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(m_ApplyBuffer != nullptr, ResultBufferNotEnough());

    *outCompleteFragment = false;

    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->sourceKey.storageId));

    DeltaApplier applier;
    {
        size_t statusSize = 0;
        if (fragmentSetOffset > 0 || fragmentOffset > 0)
        {
            statusSize = sizeof(m_TaskState->deltaApplierStatus);
        }
        NN_RESULT_DO(applier.Initialize(
            &(m_TaskState->deltaApplierOffset),
            &storage,
            placeHolder,
            m_TaskState->deltaApplierStatus,
            statusSize
        ));
    }

    detail::MountName mountName;
    fs::FileHandle file;
    NN_RESULT_DO(OpenFragment(&mountName, &file, fragment));
    NN_UTIL_SCOPE_EXIT { CloseFragment(mountName, file);  };

    int64_t fileSize;
    NN_RESULT_DO(fs::GetFileSize(&fileSize, file));
    NN_RESULT_THROW_UNLESS(m_TaskState->deltaApplierOffset < fragmentSetOffset + fileSize, ResultUnexpectedDeltaOffset());

    // deltaApplier に指定された箇所から読み始める。それに伴いプログレスも調整する
    int64_t totalRead = m_TaskState->deltaApplierOffset - fragmentSetOffset;
    IncrementAppliedDeltaProgress(totalRead - fragmentOffset);
    while(totalRead < fileSize)
    {
        size_t readSize;
        NN_RESULT_DO(fs::ReadFile(&readSize, file, totalRead, m_ApplyBuffer, m_ApplyBufferSize));

        size_t processedSize;
        NN_RESULT_DO(applier.ApplyDelta(
            &processedSize,
            m_TaskState->deltaApplierStatus,
            sizeof(m_TaskState->deltaApplierStatus),
            m_ApplyBuffer,
            readSize
        ));
#if 0
        NN_DETAIL_NCM_TRACE("[ApplyDelta] %lld / %llu, read: %llu\n", totalRead, fileSize, readSize);
#endif
        totalRead += processedSize;
        IncrementAppliedDeltaProgress(processedSize);

        if (IsCancelRequested())
        {
            break;
        }
    }
    *outProcessedOffset = totalRead;
    // nca サイズと fragment サイズの差分を加算
    if (totalRead == fileSize)
    {
        IncrementAppliedDeltaProgress(fragment.GetSize() - fileSize);
        *outCompleteFragment = true;
    }

    NN_RESULT_SUCCESS;
}

ApplyDeltaProgress ApplyDeltaTaskBase::GetProgress() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> locker(m_ProgressMutex);

    return m_TaskState->progress;
}

void ApplyDeltaTaskBase::SetTotalDeltaSize(int64_t totalSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> locker(m_ProgressMutex);

    m_TaskState->progress.totalDeltaSize = totalSize;
}

void ApplyDeltaTaskBase::IncrementAppliedDeltaProgress(int64_t appliedSize) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> locker(m_ProgressMutex);

    m_TaskState->progress.appliedDeltaSize += appliedSize;
}

void ApplyDeltaTaskBase::SetProgressState(ApplyDeltaProgressState state) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> locker(m_ProgressMutex);

    m_TaskState->progress.state = state;
}

void ApplyDeltaTaskBase::CleanupProgress() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> locker(m_ProgressMutex);

    m_TaskState->progress = ApplyDeltaProgress();
}

Result ApplyDeltaTaskBase::CopyContentMeta(ContentStorage* destinationStorage, const PlaceHolderId& placeHolder) NN_NOEXCEPT
{
    // TODO: Prepare 時に m_ApplyBuffer を使える保証がないので、少量を AutoBuffer で確保している
    //       InitialPrepare が無ければ確実に m_ApplyBuffer を利用できるのだが、やむなし
    const int BufferSize = 4 * 1024;
    AutoBuffer buffer;
    NN_RESULT_DO(buffer.Initialize(BufferSize));

    ContentStorage storage;
    NN_RESULT_DO(OpenContentStorage(&storage, m_TaskState->deltaKey.storageId));

    int64_t fileSize;
    NN_RESULT_DO(storage.GetSize(&fileSize, m_TaskState->metaContentInfo.id));

    NN_RESULT_TRY(destinationStorage->CreatePlaceHolder(placeHolder, m_TaskState->metaContentInfo.id, fileSize))
        NN_RESULT_CATCH_CONVERT(fs::ResultUsableSpaceNotEnough, ResultNotEnoughSpaceToApplyDelta());
    NN_RESULT_END_TRY;

    int64_t readOffset = 0;
    while(readOffset < fileSize)
    {
        size_t readSize = std::min(static_cast<size_t>(fileSize - readOffset), buffer.GetSize());
        NN_RESULT_DO(storage.ReadContentIdFile(buffer.Get(), readSize, m_TaskState->metaContentInfo.id, readOffset));
        NN_RESULT_DO(destinationStorage->WritePlaceHolder(placeHolder, readOffset, buffer.Get(), readSize));
        readOffset += readSize;
    }

    NN_RESULT_SUCCESS;
}


}}
