﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief NumberLineAllocator の実装です。
 */

#include <cerrno>
#include <cstdlib>
#include <new>
#include <mutex>
#include <algorithm>

#include <nn/mem/mem_NumberLineAllocator.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>

namespace nn { namespace mem {

namespace {

uint64_t g_Size = 0;
uint64_t g_TrieNodeAllocNum = 0;
uint64_t g_SpanAllocNum = 0;

/**
 * @brief   Span の状態を表す列挙型です。
 */
enum SpanStatus
{
    SpanStatus_Head = 0,      // Span の先頭(NumberLineAllocator が実態を持つ Span につけられる）
    SpanStatus_NotExist,      // フリーリストにも確保領域にもいない Span である（確保不可能）
    SpanStatus_OnFreeList,    // フリーリストにいる Span である
    SpanStatus_Allocated      // 確保されている Span である
};

/**
 * @brief   双方向循環リストにいろいろなメタデータを持たせた構造体。部分数直線を表します。
 */
struct Span
{
    uint32_t first;     // 自身に割り当てられているインデックスの先頭
    uint32_t last;      // 自身に割り当てられているインデックスの末尾
    uint32_t status;    // SpanStatus
    // フリーリストにつながっているときはフリー Span の双方向リストに、
    // 確保時のときはユーザデータの格納場所になる共用体
    union
    {
        struct
        {
            Span* prev;
            Span* next;
        } FreeLink;
        struct
        {
            void* pData;
            void (*dtor)(void*);
        } UserData;
    } u;
    Span* prev;
    Span* next;
};

/**
 * @brief   pList の次に pItem を挿入します。
 */
void InsertSpanAfter(Span* pList, Span* pItem) NN_NOEXCEPT
{
    Span* next = pList->next;
    NN_SDK_ASSERT_NOT_NULL(next);
    pItem->next = next;
    pItem->prev = pList;
    pList->next = pItem;
    next->prev = pItem;
}

/**
 * @brief   pItem がつながっているリストから pItem を削除します。
 */
void RemoveSpan(Span* pItem) NN_NOEXCEPT
{
    Span* prev = pItem->prev;
    Span* next = pItem->next;
    NN_SDK_ASSERT_NOT_NULL(prev);
    NN_SDK_ASSERT_NOT_NULL(next);
    prev->next = next;
    next->prev = prev;
    pItem->prev = NULL;
    pItem->next = NULL;
}

/**
 * @brief   確保時の各ノードを表す構造体
 */
struct TrieNode
{
    union
    {
        uintptr_t val;
        TrieNode* child;
        Span* span;         // Span として利用している場合、最下位ビットを 1 にする
    } u[256];      // 子ノードを 256 まで持てる
};

/**
 * @brief   確保済みの Span を管理する高さ 3 の木構造です。
 * @details 一つのノード（節）につき持てるノードは 256 = 256 までです。
 *          よって、このクラスで管理できる部分数直線の数は 256 ^ 3 = 16Ki個までとなります。
 */
class Trie8x3
{
public:
    Trie8x3(nn::mem::MallocCallback mallocFunc, void* pMallocParam, nn::mem::FreeCallback freeFunc, void* pFreeParam) NN_NOEXCEPT : m_MallocFunc(mallocFunc), m_pMallocParam(pMallocParam), m_FreeFunc(freeFunc), m_pFreeParam(pFreeParam)
    {
        memset(&m_Lv0, 0, sizeof(m_Lv0));
    }

    ~Trie8x3() NN_NOEXCEPT
    {
        // すべてのノード(TrieNode を格納している領域)を解放
        for (int i = 0; i < 256; ++i)
        {
            if (m_Lv0.u[i].val == 0 || m_Lv0.u[i].val & 1)
            {
                continue;
            }
            TrieNode* lv1 = m_Lv0.u[i].child;
            for (int j = 0; j < 256; ++j)
            {
                if (lv1->u[j].val == 0 || lv1->u[j].val & 1)
                {
                    continue;
                }
                TrieNode* lv2 = lv1->u[j].child;
                m_FreeFunc(lv2, m_pFreeParam);
            }
            m_FreeFunc(lv1, m_pFreeParam);
        }
    }

    /**
     * @brief   Span の状況をダンプします。
     * @details 1 段目、 2 段目、 3 段目と深さ優先で Dump
     */
    void Dump(uint64_t offset, int factor) NN_NOEXCEPT
    {
        NN_UNUSED(offset);
        NN_UNUSED(factor);

        uint64_t firstForDisp, lastForDisp;
        size_t sizeForDisp;
        NN_UNUSED(firstForDisp);
        NN_UNUSED(lastForDisp);
        NN_UNUSED(sizeForDisp);
        // ログ出力時の引数の準備
        auto CalcDispVariable = [&](uint32_t first, uint32_t last) {
            firstForDisp = static_cast<uint64_t>(first) * static_cast<uint64_t>(factor) + offset;
            lastForDisp = static_cast<uint64_t>(last) * static_cast<uint64_t>(factor) + offset;
            NN_SDK_ASSERT(last - first >= 0);
            sizeForDisp = static_cast<uint64_t>(last - first) * static_cast<size_t>(factor);
        };

        for (int i = 0; i < 256; ++i)
        {
            if (m_Lv0.u[i].val == 0)
            {
                continue;
            }
            if (m_Lv0.u[i].val & 1)
            {
                // 最下位ビットの 1 を取る
                Span* rval = reinterpret_cast<Span*>(m_Lv0.u[i].val & ~static_cast<uintptr_t>(1));
                NN_UNUSED(rval);
                CalcDispVariable(rval->first, rval->last);
                NN_DETAIL_MEM_INFO("    <alloc level0='%d' level1='-' level2='-' first='%016llx' last='%016llx' size='%zu'/>\n",
                    i, firstForDisp, lastForDisp, sizeForDisp);
                continue;
            }
            TrieNode* lv1 = m_Lv0.u[i].child;
            for (int j = 0; j < 256; ++j)
            {
                if (lv1->u[j].val == 0)
                {
                    continue;
                }
                if (lv1->u[j].val & 1)
                {
                    Span* rval = reinterpret_cast<Span*>(lv1->u[j].val & ~static_cast<uintptr_t>(1));
                    NN_UNUSED(rval);
                    CalcDispVariable(rval->first, rval->last);
                    NN_DETAIL_MEM_INFO("    <alloc level0='%d' level1='%d' level2='-' first='%016llx' last='%016llx' size='%zu'/>\n",
                        i, j, firstForDisp, lastForDisp, sizeForDisp);
                    continue;
                }
                TrieNode* lv2 = lv1->u[j].child;
                for (int k = 0; k < 256; ++k)
                {
                    if (lv2->u[k].span)
                    {
                        Span* rval = lv2->u[k].span;
                        NN_UNUSED(rval);
                        CalcDispVariable(rval->first, rval->last);
                        NN_DETAIL_MEM_INFO("    <alloc level0='%d' level1='%d' level2='%d' first='%016llx' last='%016llx' size='%zu'/>\n",
                            i, j, k, firstForDisp, lastForDisp, sizeForDisp);
                    }
                }
            }
        }
    }

    /**
     * @brief   first から導き出せる Span を見つけ、返します。
     * @param[in]   first   Span を一意に識別できる 32bit 値
     * @param[in]   remove  見つけた Span を木から削除するか否か
     */
    Span* FindSpan(uint32_t first, bool remove) NN_NOEXCEPT
    {
        Span* rval;
        // レベル 0 (最上位) に割り当てられているビット→ 24 ～ 17 ビット目
        int lv0Idx = first >> 16;
        if (m_Lv0.u[lv0Idx].val & 1)
        {
            if (first & 0xFFFF) return NULL;
            // 最下位ビットを 0 に戻す（ span として利用しているフラグを外す ）
            rval = reinterpret_cast<Span*>(m_Lv0.u[lv0Idx].val & ~static_cast<uintptr_t>(1));
            if (remove)
            {
                m_Lv0.u[lv0Idx].span = NULL;
            }
            return rval;
        }
        TrieNode* lv1 = m_Lv0.u[lv0Idx].child;
        if (!lv1) {
            return NULL;
        }
        // レベル 1 (中段) に割り当てられているビット→ 16 ～ 9 ビット目
        int lv1Idx = (first >> 8) & 0xFF;
        if (lv1->u[lv1Idx].val & 1) {
            if (first & 0xFF)
            {
                return NULL;
            }
            rval = reinterpret_cast<Span*>(lv1->u[lv1Idx].val & ~static_cast<uintptr_t>(1));
            if (remove)
            {
                lv1->u[lv1Idx].span = NULL;
            }
            return rval;
        }
        TrieNode* lv2 = lv1->u[lv1Idx].child;
        if (!lv2)
        {
            return NULL;
        }
        // レベル 2 (最下位・葉) に割り当てられているビット→ 8 ～ 1 ビット目
        int lv2Idx = first & 0xFF;
        rval = lv2->u[lv2Idx].span;
        if (remove)
        {
            lv2->u[lv2Idx].span = NULL;
        }
        return rval;
    }

    /**
     * @brief   Span を適切なレベルに追加します。
     * @param   pItem   追加する Span
     */
    errno_t AddSpan(Span* pItem) NN_NOEXCEPT
    {
        // Spanのオーバーラップとかはないものとする(内部クラスなので)
        uint32_t first = pItem->first;
        uint32_t size = pItem->last - pItem->first;

        int lv0Idx = first >> 16;
        NN_SDK_ASSERT(!(m_Lv0.u[lv0Idx].val & 1));
        if (!m_Lv0.u[lv0Idx].child)
        {
            // 大きいサイズの Span ならレベル 0 の領域そのまま明け渡す
            if (!(first & 0xFFFF) && size >= 0x10000)
            {
                m_Lv0.u[lv0Idx].span = pItem;
                m_Lv0.u[lv0Idx].val |= 1;
                return 0;
            }
            TrieNode* tmp = NewTrieNode();
            if (!tmp)
            {
                return ENOMEM;
            }
            m_Lv0.u[lv0Idx].child = tmp;
        }

        TrieNode* lv1 = m_Lv0.u[lv0Idx].child;
        int lv1Idx = (first >> 8) & 0xFF;
        NN_SDK_ASSERT(!(lv1->u[lv1Idx].val & 1));
        if (!lv1->u[lv1Idx].child)
        {
            if (!(first & 0xFF) && size >= 0x100)
            {
                lv1->u[lv1Idx].span = pItem;
                lv1->u[lv1Idx].val |= 1;
                return 0;
            }
            TrieNode* tmp = NewTrieNode();
            if (!tmp)
            {
                return ENOMEM;
            }
            lv1->u[lv1Idx].child = tmp;
        }

        TrieNode* lv2 = lv1->u[lv1Idx].child;
        int lv2Idx = first & 0xFF;
        NN_SDK_ASSERT(lv2->u[lv2Idx].span == NULL);
        lv2->u[lv2Idx].span = pItem;
        return 0;
    }

    /**
     * @brief   確保済みの最大サイズを返します。
     */
    int GetAllocatedSize() NN_NOEXCEPT
    {
        int retVal = 0;
        for (int i = 0; i < 256; ++i)
        {
            if (m_Lv0.u[i].val == 0)
            {
                continue;
            }
            if (m_Lv0.u[i].val & 1)
            {
                // 最下位ビットの 1 を取る（Span として利用しているかどうかのバイナリハック）
                Span* rval = reinterpret_cast<Span*>(m_Lv0.u[i].val & ~static_cast<uintptr_t>(1));
                retVal += (rval->last - rval->first);
                continue;
            }
            TrieNode* lv1 = m_Lv0.u[i].child;
            for (int j = 0; j < 256; ++j)
            {
                if (lv1->u[j].val == 0)
                {
                    continue;
                }
                if (lv1->u[j].val & 1)
                {
                    Span* rval = reinterpret_cast<Span*>(lv1->u[j].val & ~static_cast<uintptr_t>(1));
                    retVal += (rval->last - rval->first);
                    continue;
                }
                TrieNode* lv2 = lv1->u[j].child;
                for (int k = 0; k < 256; ++k)
                {
                    if (lv2->u[k].span)
                    {
                        Span* rval = lv2->u[k].span;
                        retVal += (rval->last - rval->first);
                    }
                }
            }
        }
        return retVal;
    }

private:
    /**
     * @brief   登録されている malloc コールバックを利用して TrieNode 用の領域を確保します。
     */
    TrieNode* NewTrieNode() NN_NOEXCEPT
    {
        g_TrieNodeAllocNum += 1;
        g_Size += static_cast<uint64_t>(sizeof(TrieNode));
        TrieNode* p = reinterpret_cast<TrieNode*>(m_MallocFunc(sizeof(*p), m_pMallocParam));
        if (p)
        {
            memset(p, 0, sizeof(*p));
        }
        return p;
    }

    nn::mem::MallocCallback m_MallocFunc;
    void*                   m_pMallocParam;
    nn::mem::FreeCallback   m_FreeFunc;
    void*                   m_pFreeParam;
    TrieNode                m_Lv0;
};

/**
 * @brief   Segregated Free List です。
 * @details 自身でSpanのメモリ割り当ては行いません。
 *          Spanのステートを変えることもしません。
 *          - サイズごとに 256 つのグループに分け、それぞれをフリーリストとして管理している
 *          - グループ 0 ～ 254 : 長さ 1 ～ 255 のサイズのフリーリスト
 *          - グループ 255 : 長さ 256 以上のサイズのフリーリスト
 */
class FreeList
{
public:
    FreeList() NN_NOEXCEPT
    {
        memset(&m_pNextIdx[0], 0xFF, sizeof(m_pNextIdx));
        memset(&m_pFreeListHead[0], 0, sizeof(m_pFreeListHead));
        for (int i = 0; i < 256; ++i)
        {
            m_pFreeListHead[i].u.FreeLink.prev = &m_pFreeListHead[i];
            m_pFreeListHead[i].u.FreeLink.next = &m_pFreeListHead[i];
        }
    }
    ~FreeList() NN_NOEXCEPT{}

    /**
     * @brief   フリーリストの先頭に pItem をつなぎます
     */
    void Insert(Span* pItem) NN_NOEXCEPT
    {
        size_t idx = pItem->last - pItem->first - 1;
        if (idx > 255)
        {
            idx = 255;
        }
        Span& head = m_pFreeListHead[idx];
        Span* next = head.u.FreeLink.next;
        head.u.FreeLink.next = pItem;
        pItem->u.FreeLink.prev = &head;
        pItem->u.FreeLink.next = next;
        next->u.FreeLink.prev = pItem;
        if (next == &head)
        {
            // もともとのフリーリストが空だった場合、 m_pNextIdx[] を更新する
            // m_pNextIdx[] は空のエントリはスキップする
            // スキップされないよう、自分より小さなサイズを管理するフリーリストに自分を登録させる
            int to = static_cast<int>(idx);
            int i = to - 1;
            while (i >= 0)
            {
                if (m_pNextIdx[i] <= to)
                {
                    --i;
                    continue;
                }
                m_pNextIdx[i] = static_cast<uint8_t>(to);
                --i;
            }
        }
    }

    /**
     * @brief   リストから pItem を削除します。
     */
    void Remove(Span* pItem) NN_NOEXCEPT
    {
        Span* prev = pItem->u.FreeLink.prev;
        Span* next = pItem->u.FreeLink.next;
        prev->u.FreeLink.next = next;
        next->u.FreeLink.prev = prev;
        pItem->u.FreeLink.prev = pItem->u.FreeLink.next = NULL;
    }

    /**
     * @brief   size に適するフリーリストの一番先頭のものを削除します。
     * @return  Span を返します。
     *          フリーリストに Span がなければ NULL を返します。
     */
    Span* FindAndRemove(size_t size) NN_NOEXCEPT
    {
        size_t idx = size >= 256 ? 255 : size - 1;
        Span* head = &m_pFreeListHead[idx];
        while (idx < 255)
        {
            Span* pItem = head->u.FreeLink.next;
            // 見つけたフリーリストに要素が一つでもつながっているなら先頭を消して終わり
            if (pItem != head)
            {
                Remove(pItem);
                return pItem;
            }
            // 要素がなければ次に大きいサイズのフリーリストを見にいく
            size_t next_idx = m_pNextIdx[idx];
            for (;;)
            {
                head = &m_pFreeListHead[next_idx];
                if (head->u.FreeLink.next != head)
                {
                    break;
                }
                if (next_idx == 255)
                {
                    // 256 以上のサイズが格納されているフリーリストを見にいく
                    next_idx = 256;
                    break;
                }
                // m_pNextIdx[] から要素が全くつながっていないリストのインデックスを削除
                next_idx = m_pNextIdx[next_idx];
                m_pNextIdx[idx] = static_cast<uint8_t>(next_idx);
            }
            idx = next_idx;
        }
        // 256 以上のサイズが格納されているフリーリストを線形サーチしなくてはならない
        Span* it = head->u.FreeLink.next;
        while (it != head)
        {
            if (it->last - it->first >= size)
            {
                Remove(it);
                return it;
            }
            it = it->u.FreeLink.next;
        }
        // 削除できるフリーリストはなかった
        return NULL;
    }

    /**
    * @brief   size に適するフリーリストのアライメントが合っているもののうち、一番先頭のものを削除します。
    * @return  Span を返します。
    *          フリーリストに Span がなければ NULL を返します。
    */
    Span* FindAndRemove(size_t size, uint32_t align) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(size >= 1 && size <= 0x01000000);

        if (align == 1)
        {
            return FindAndRemove(size);
        }
        size_t idx = size >= 256 ? 255 : size - 1;
        Span* head = &m_pFreeListHead[idx];
        while (idx < 255)
        {
            Span* pItem = head->u.FreeLink.next;
            while (pItem != head)
            {
                for (uint32_t i = pItem->first; pItem->last - i >= size; ++i)
                {
                    if (i % align == 0)
                    {
                        Remove(pItem);
                        return pItem;
                    }
                }
                pItem = pItem->u.FreeLink.next;
            }
            // 要素がなければ次に大きいサイズのフリーリストを見にいく
            size_t next_idx = m_pNextIdx[idx];
            for (;;)
            {
                head = &m_pFreeListHead[next_idx];
                if (head->u.FreeLink.next != head)
                {
                    break;
                }
                if (next_idx == 255)
                {
                    // 256 以上のサイズが格納されているフリーリストを見にいく
                    next_idx = 256;
                    break;
                }
                // m_pNextIdx[] から要素が全くつながっていないリストのインデックスを削除
                next_idx = m_pNextIdx[next_idx];
                m_pNextIdx[idx] = static_cast<uint8_t>(next_idx);
            }
            idx = next_idx;
        }
        // 256 以上のサイズが格納されているフリーリストを線形サーチしなくてはならない
        Span* it = head->u.FreeLink.next;
        while (it != head)
        {
            for (uint32_t i = it->first; it->last - i >= size; ++i)
            {
                if (i % align == 0)
                {
                    Remove(it);
                    return it;
                }
            }
            it = it->u.FreeLink.next;
        }
        // 削除できるフリーリストはなかった
        return NULL;
    }

    /**
     * @brief   空きサイズの合計値と確保可能な最大サイズを取得します。
     */
    void GetSizeInfo(int* outTotalFreeSize, int* outAllocatableSize) NN_NOEXCEPT
    {
        int totalFreeSize = 0;
        int allocatableSize = 0;
        int index = 0;
        while (index < 256)
        {
            Span* pHead = &m_pFreeListHead[index];
            Span* pItem = pHead->u.FreeLink.next;
            while (pItem != pHead)
            {
                totalFreeSize += (pItem->last - pItem->first);
                allocatableSize = std::max(allocatableSize, static_cast<int>(pItem->last - pItem->first));
                pItem = pItem->u.FreeLink.next;
            }
            if (index >= 255)
            {
                break;
            }
            index = m_pNextIdx[index];
        }
        NN_SDK_ASSERT(totalFreeSize >= 0);
        NN_SDK_ASSERT(allocatableSize >= 0);
        *outTotalFreeSize = totalFreeSize;
        *outAllocatableSize = allocatableSize;
    }

    /**
     * @brief   フリーリストの状況をダンプします。
     */
    void Dump(uint64_t offset, int factor) NN_NOEXCEPT
    {
        NN_UNUSED(offset);  // LOG 出力にしか使われないため NN_UNUSED が必要
        NN_UNUSED(factor);
        int index = 0;
        while (index < 256)
        {
            Span* pHead = &m_pFreeListHead[index];
            Span* pItem = pHead->u.FreeLink.next;
            if (pItem == pHead)     // このサイズのフリーリストに要素がない
            {
                index = (index + 1 < m_pNextIdx[index]) ? m_pNextIdx[index] : (index + 1);
                continue;
            }
            while (pItem != pHead)
            {
                uint64_t firstForDisp = static_cast<uint64_t>(pItem->first) * static_cast<uint64_t>(factor) + offset;
                uint64_t lastForDisp = static_cast<uint64_t>(pItem->last) * static_cast<uint64_t>(factor) + offset;
                size_t sizeForDisp = static_cast<uint64_t>(pItem->last - pItem->first) * static_cast<size_t>(factor);
                NN_UNUSED(firstForDisp);
                NN_UNUSED(lastForDisp);
                NN_UNUSED(sizeForDisp);
                NN_DETAIL_MEM_INFO("    <free index='%d' first='%016llx' last='%016llx' size='%zu'/>\n",
                    index, firstForDisp, lastForDisp, sizeForDisp);
                pItem = pItem->u.FreeLink.next;
            }
            index = (index + 1 < m_pNextIdx[index]) ? m_pNextIdx[index] : (index + 1);
        }
    }

private:
    uint8_t m_pNextIdx[256];  // 自身より大きいサイズのリストで空でない可能性のあるものを指す
    Span m_pFreeListHead[256];
};

/**
 * @brief   Span ポインタ用の領域を与えるクラスです。
 * @details 領域解放後も 128 個まではこのクラス内で保持しておきます。
 */
class SpanCache
{
public:
    SpanCache(nn::mem::MallocCallback mallocFunc, void* pMallocParam, nn::mem::FreeCallback freeFunc, void* pFreeParam) NN_NOEXCEPT
        : m_Count(0), m_MallocFunc(mallocFunc), m_pMallocParam(pMallocParam), m_FreeFunc(freeFunc), m_pFreeParam(pFreeParam), m_pHead(NULL)
    {
    }
    ~SpanCache() NN_NOEXCEPT
    {
        // 全ての Span を解放
        Span* it = m_pHead;
        while (it)
        {
            Span* next = it->next;
            m_FreeFunc(it, m_pFreeParam);
            it = next;
        }
    }

    /**
     * @brief   Span のポインタ用の領域を確保します。
     */
    Span* Alloc() NN_NOEXCEPT
    {
        Span* pItem;
        if (m_pHead)
        {
            // 既に確保済みの領域があるなら、そこを使う
            --m_Count;
            pItem = m_pHead;
            m_pHead = m_pHead->next;
        }
        else
        {
            g_SpanAllocNum += 1;
            g_Size += static_cast<uint64_t>(sizeof(*pItem));
            pItem = reinterpret_cast<Span*>(m_MallocFunc(sizeof(*pItem), m_pMallocParam));
            if (!pItem)
            {
                return NULL;
            }
        }
        memset(pItem, 0, sizeof(*pItem));
        return pItem;
    }

    /**
     * @brief   Span のポインタ用の領域を解放します。
     */
    void Free(Span* pItem) NN_NOEXCEPT
    {
        if (m_Count < 128)
        {
            pItem->next = m_pHead;
            m_pHead = pItem;
            ++m_Count;
        }
        else
        {
            m_FreeFunc(pItem, m_pFreeParam);
        }
    }

private:
    size_t                  m_Count;
    nn::mem::MallocCallback m_MallocFunc;
    void*                   m_pMallocParam;
    nn::mem::FreeCallback   m_FreeFunc;
    void*                   m_pFreeParam;
    Span*                   m_pHead;        // Span* 用フリーリスト（単方向で運用）
};

}   // unnamed namespace

/**
 * @brief   NumberLineAllocator の実態です。
 */
class NumberLineAllocatorPrv
{
public:
    NumberLineAllocatorPrv(nn::mem::MallocCallback mallocFunc, void* pMallocParam, nn::mem::FreeCallback freeFunc, void* pFreeParam) NN_NOEXCEPT
        : m_Trie(mallocFunc, pMallocParam, freeFunc, pFreeParam), m_FreeList(), m_SpanCache(mallocFunc, pMallocParam, freeFunc, pFreeParam)
    {
        memset(&m_SpanList, 0, sizeof(m_SpanList));
    }

    ~NumberLineAllocatorPrv() NN_NOEXCEPT
    {
        // m_SpanListをfreeしていく
        Span* end = &m_SpanList;
        Span* it = end->next;
        while (it != end)
        {
            Span* next = it->next;
            if (it->status == SpanStatus_Allocated)
            {
                if (it->u.UserData.dtor)
                {
                    it->u.UserData.dtor(it->u.UserData.pData);
                }
            }
            FreeSpan(it);
            it = next;
        }
    }

    /**
     * @brief   一つの大きな Span を m_SpanList につなぎます。
     */
    errno_t Init() NN_NOEXCEPT
    {
        Span* pAll = NewSpan();
        if (!pAll)
        {
            return ENOMEM;
        }
        memset(pAll, 0, sizeof(*pAll));

        m_SpanList.prev = &m_SpanList;
        m_SpanList.next = pAll;
        pAll->prev = &m_SpanList;
        pAll->next = &m_SpanList;
        pAll->first = 0;
        pAll->last = 0x01000000U;
        pAll->status = SpanStatus_NotExist;

        return 0;
    }

    /**
     * @brief       size 分の区間を割り当て、 pAllocated に先頭を入れます。
     * @param[out]  pAllocated  確保された Span*
     * @param[in]   size        確保したいサイズ
     */
    errno_t Alloc(Span** pAllocated, uint32_t size) NN_NOEXCEPT
    {
        Span* pItem = m_FreeList.FindAndRemove(size);
        if (!pItem)
        {
            // 適切な領域が見つけられないなら、コンパクション
            Compaction();
            pItem = m_FreeList.FindAndRemove(size);
            if (!pItem)
            {
                return ENOENT;
            }
        }
        bool justfit = (pItem->first + size == pItem->last);
        if (!justfit)
        {
            if (size < 256)
            {
                // 後ろを切り取ってフリーリストに戻す
                errno_t e;
                // フリーリストに返す分の span ポインタ用の領域を確保
                Span* back = NewSpan();
                if (!back)
                {
                    // 管理領域が確保できなかった場合、フリーリストに span をそのまま戻す
                    m_FreeList.Insert(pItem);
                    return ENOMEM;
                }
                back->first = pItem->first + size;
                back->last = pItem->last;
                back->status = SpanStatus_OnFreeList;

                pItem->last = back->first;
                if ((e = m_Trie.AddSpan(pItem)) != 0)
                {
                    // trie に追加失敗したならフリーリストに全部返却してエラーを返す
                    pItem->last = back->last;
                    m_FreeList.Insert(pItem);
                    FreeSpan(back);
                    return e;
                }

                m_FreeList.Insert(back);
                InsertSpanAfter(pItem, back);
            }
            else
            {
                // 前を切り取ってフリーリストに戻す（外部断片化防止のため？）
                errno_t e;
                Span* back = NewSpan();
                if (!back)
                {
                    m_FreeList.Insert(pItem);
                    return ENOMEM;
                }
                back->first = pItem->last - size;
                back->last = pItem->last;

                if ((e = m_Trie.AddSpan(back)) != 0)
                {
                    m_FreeList.Insert(pItem);
                    FreeSpan(back);
                    return e;
                }

                pItem->last = back->first;
                m_FreeList.Insert(pItem);
                InsertSpanAfter(pItem, back);
                pItem = back;
            }
        }
        else
        {
            errno_t e;
            if ((e = m_Trie.AddSpan(pItem)) != 0)
            {
                m_FreeList.Insert(pItem);
                return e;
            }
        }

        pItem->status = SpanStatus_Allocated;
        pItem->u.UserData.dtor = NULL;
        pItem->u.UserData.pData = NULL;
        *pAllocated = pItem;
        return 0;
    }

    /**
    * @brief       size 分の区間を割り当て、 pAllocated に先頭を入れます。
    * @param[out]  pAllocated  確保された Span*
    * @param[in]   size        確保したいサイズ
    * @param[in]   align       アライメント
    */
    errno_t Alloc(Span** pAllocated, uint32_t size, uint32_t align) NN_NOEXCEPT
    {
        Span* pItem = m_FreeList.FindAndRemove(size, align);
        if (!pItem)
        {
            // 適切な領域が見つけられないなら、コンパクション
            Compaction();
            pItem = m_FreeList.FindAndRemove(size, align);
            if (!pItem)
            {
                return ENOENT;
            }
        }
        bool justfit = (pItem->first + size == pItem->last);
        if (!justfit)
        {
            if (pItem->first % align == 0)
            {
                // 後ろを切り取ってフリーリストに戻す
                errno_t e;
                // フリーリストに返す分の span ポインタ用の領域を確保
                Span* back = NewSpan();
                if (!back)
                {
                    m_FreeList.Insert(pItem);
                    return ENOMEM;
                }
                back->first = pItem->first + size;
                back->last = pItem->last;
                back->status = SpanStatus_OnFreeList;

                pItem->last = back->first;
                if ((e = m_Trie.AddSpan(pItem)) != 0)
                {
                    // trie に追加失敗したならフリーリストに全部返却してエラーを返す
                    pItem->last = back->last;
                    m_FreeList.Insert(pItem);
                    FreeSpan(back);
                    return e;
                }

                m_FreeList.Insert(back);
                InsertSpanAfter(pItem, back);
            }
            else
            {
                // アライメントに合う部分を切り取って残りをフリーリストに戻す
                errno_t e;
                Span* middle = NewSpan();
                if (!middle)
                {
                    m_FreeList.Insert(pItem);
                    return ENOMEM;
                }
                Span* back = NewSpan();
                if (!back)
                {
                    FreeSpan(middle);
                    m_FreeList.Insert(pItem);
                    return ENOMEM;
                }
                // 前方の切り取り
                bool isFound = false;
                for (uint32_t i = pItem->first; pItem->last - i >= size; ++i)
                {
                    if (i % align == 0)
                    {
                        middle->first = i;
                        middle->last = middle->first + size;
                        isFound = true;
                        break;
                    }
                }
                NN_UNUSED(isFound);
                NN_SDK_ASSERT(isFound);
                // 後方の切り取り
                back->first = middle->last;
                back->last = pItem->last;
                back->status = SpanStatus_OnFreeList;
                pItem->last = middle->first;
                if ((e = m_Trie.AddSpan(middle)) != 0)
                {
                    pItem->last = back->last;
                    m_FreeList.Insert(pItem);
                    FreeSpan(middle);
                    FreeSpan(back);
                    return e;
                }

                m_FreeList.Insert(pItem);
                m_FreeList.Insert(back);
                InsertSpanAfter(pItem, middle);
                InsertSpanAfter(middle, back);
                pItem = middle;
            }
        }
        else
        {
            errno_t e;
            if ((e = m_Trie.AddSpan(pItem)) != 0)
            {
                m_FreeList.Insert(pItem);
                return e;
            }
        }

        pItem->status = SpanStatus_Allocated;
        pItem->u.UserData.dtor = NULL;
        pItem->u.UserData.pData = NULL;
        *pAllocated = pItem;
        return 0;
    }

    /**
     * @brief       確保済みの部分数直線を解放します
     * @param[in]   first   確保済みの部分数直線の先頭の数字
     */
    errno_t Free(uint32_t first) NN_NOEXCEPT
    {
        Span* pItem = m_Trie.FindSpan(first, true);
        if (!pItem)
        {
            return ENOENT;
        }
        if (pItem->u.UserData.dtor)
        {
            pItem->u.UserData.dtor(pItem->u.UserData.pData);
        }
        pItem->status = SpanStatus_OnFreeList;
        if (pItem->first + 256 <= pItem->last)
        {
            // 自身のサイズが大きい場合、両隣と統合しようとする
            // Compaction()が呼ばれるリスクを低くするため
            Span* next = pItem->next;
            if (next->status == SpanStatus_OnFreeList)
            {
                m_FreeList.Remove(next);
                RemoveSpan(next);
                pItem->last = next->last;
                FreeSpan(next);
            }
            Span* prev = pItem->prev;
            if (prev->status == SpanStatus_OnFreeList)
            {
                bool prevIsSmall = prev->first + 256 > prev->last;
                prev->last = pItem->last;
                if (prevIsSmall)
                {
                    // 登録しなおす必要がある。
                    m_FreeList.Remove(prev);
                    m_FreeList.Insert(prev);
                }
                RemoveSpan(pItem);
                FreeSpan(pItem);
            }
            else
            {
                m_FreeList.Insert(pItem);
            }
        }
        else
        {
            m_FreeList.Insert(pItem);
        }
        return 0;
    }

    /**
     * @brief   ユーザ固有領域を探します。
     */
    errno_t FindUserData(void** user_data_ptr, uint32_t first) NN_NOEXCEPT
    {
        Span* pItem = m_Trie.FindSpan(first, false);
        if (!pItem)
        {
            return ENOENT;
        }
        *user_data_ptr = pItem->u.UserData.pData;
        return 0;
    }

    /**
     * @brief   first の長さを取得します。
     */
    errno_t GetSizeOf(int* outSize, uint32_t first) NN_NOEXCEPT
    {
        Span* pItem = m_Trie.FindSpan(first, false);
        if (!pItem)
        {
            return ENOENT;
        }
        *outSize = pItem->last - pItem->first;
        return 0;
    }

    /**
     * @brief   [first, last) を確保可能にします。
     * @details 該当の Span の SpanStatus を SpanStatus_OnFreeList にします。
     *          複数の Span をまたいで確保可能にすることはできません。
     */
    errno_t AddRange(uint32_t first, uint32_t last) NN_NOEXCEPT
    {
        Compaction();
        // 自身を含むSpanを探す
        Span* pEnd = &m_SpanList;
        Span* it = pEnd->next;
        // span をたどっていく
        for (;;)
        {
            if (first < it->last)
            {
                if (it->first <= first && it->last >= last)
                {
                    if (it->status == SpanStatus_NotExist)
                    {
                        break;
                    }
                    return EBUSY;
                }
                else
                {
                    return EBUSY;
                }
            }
            it = it->next;
            NN_SDK_ASSERT(it != pEnd);
        }
        if (it->first == first)
        {
            if (it->last == last)
            {
                // 前後ピッタリ合う span なら、そのまま m_FreeList に入れる
                it->status = SpanStatus_OnFreeList;
                m_FreeList.Insert(it);
            }
            else
            {
                // 後ろのみを切り取る
                Span* pBack = NewSpan();
                if (!pBack)
                {
                    return ENOMEM;
                }
                pBack->first = it->first + (last - first);
                pBack->last = it->last;
                pBack->status = SpanStatus_NotExist;

                it->last = pBack->first;
                it->status = SpanStatus_OnFreeList;
                m_FreeList.Insert(it);
                InsertSpanAfter(it, pBack);
            }
        }
        else if (it->last == last)
        {
            // 前のみを切り取る
            Span* pBack = NewSpan();
            if (!pBack)
            {
                return ENOMEM;
            }
            pBack->first = first;
            pBack->last = last;
            pBack->status = SpanStatus_OnFreeList;

            it->last = first;

            InsertSpanAfter(it, pBack);
            m_FreeList.Insert(pBack);
        }
        else
        {
            // 前後を切り取る
            Span* pMiddle = NewSpan();
            if (!pMiddle)
            {
                return ENOMEM;
            }
            Span* pBack = NewSpan();
            if (!pBack)
            {
                FreeSpan(pMiddle);
                return ENOMEM;
            }

            pMiddle->first = first;
            pMiddle->last = last;
            pMiddle->status = SpanStatus_OnFreeList;

            pBack->first = last;
            pBack->last = it->last;
            pBack->status = SpanStatus_NotExist;

            it->last = first;

            InsertSpanAfter(it, pMiddle);
            InsertSpanAfter(pMiddle, pBack);
            m_FreeList.Insert(pMiddle);
        }
        return 0;
    }

    /**
    * @brief   [first, last) を確保不可にします。
    * @details 該当の span の SpanStatus を SpanStatus_NotExist にします。
    *          複数の Span をまたいで確保可能にすることはできません。
    */
    errno_t RemoveRange(uint32_t first, uint32_t last) NN_NOEXCEPT
    {
        Compaction();
        // 自身を含むspanを探す
        Span* pEnd = &m_SpanList;
        Span* it = pEnd->next;
        for (;;)
        {
            if (first < it->last)
            {
                if (it->first <= first && it->last >= last)
                {
                    if (it->status == SpanStatus_OnFreeList)
                    {
                        break;
                    }
                    return EBUSY;
                }
                else
                {
                    return EBUSY;
                }
            }
            it = it->next;
            NN_SDK_ASSERT(it != pEnd);
        }
        m_FreeList.Remove(it);
        if (it->first == first)
        {
            if (it->last == last)
            {
                it->status = SpanStatus_NotExist;
            }
            else
            {
                // 後ろのみを切り取る
                Span* pBack = NewSpan();
                if (!pBack)
                {
                    return ENOMEM;
                }
                pBack->first = it->first + (last - first);
                pBack->last = it->last;
                pBack->status = SpanStatus_OnFreeList;

                it->last = pBack->first;
                it->status = SpanStatus_NotExist;

                InsertSpanAfter(it, pBack);
                m_FreeList.Insert(pBack);
            }
        }
        else if (it->last == last)
        {
            // 前のみを切り取る
            Span* pBack = NewSpan();
            if (!pBack)
            {
                return ENOMEM;
            }
            pBack->first = first;
            pBack->last = last;
            pBack->status = SpanStatus_NotExist;

            it->last = first;

            InsertSpanAfter(it, pBack);
            m_FreeList.Insert(it);
        }
        else
        {
            // 前後を切り取る
            Span* pMiddle = NewSpan();
            if (!pMiddle)
            {
                return ENOMEM;
            }
            Span* pBack = NewSpan();
            if (!pBack)
            {
                FreeSpan(pMiddle);
                return ENOMEM;
            }

            pMiddle->first = first;
            pMiddle->last = last;
            pMiddle->status = SpanStatus_NotExist;

            pBack->first = last;
            pBack->last = it->last;
            pBack->status = SpanStatus_OnFreeList;

            it->last = first;

            InsertSpanAfter(it, pMiddle);
            InsertSpanAfter(pMiddle, pBack);
            m_FreeList.Insert(it);
            m_FreeList.Insert(pBack);
        }
        return 0;
    }

    /**
     * @brief   空き領域の合計を取得します。
     */
    int GetTotalFreeSize() NN_NOEXCEPT
    {
        int allocatableSize, totalFreeSize;
        m_FreeList.GetSizeInfo(&totalFreeSize, &allocatableSize);
        NN_UNUSED(allocatableSize);
        return totalFreeSize;
    }

    /**
     * @brief   確保可能な最大サイズを取得します。
     */
    int GetAllocatableSize() NN_NOEXCEPT
    {
        int allocatableSize, totalFreeSize;
        m_FreeList.GetSizeInfo(&totalFreeSize, &allocatableSize);
        NN_UNUSED(totalFreeSize);
        return allocatableSize;
    }

    /**
     * @brief   アロケータの状態をダンプします。
     */
    void Dump(uint64_t offset, int factor) NN_NOEXCEPT
    {
        int allocatableSize, totalFreeSize;
        m_FreeList.GetSizeInfo(&totalFreeSize, &allocatableSize);
        size_t allocatableSizeForDisp = allocatableSize * static_cast<size_t>(factor);
        size_t totalFreeSizeForDisp = totalFreeSize * static_cast<size_t>(factor);
        size_t allocatedSizeForDisp = m_Trie.GetAllocatedSize() * static_cast<size_t>(factor);
        NN_UNUSED(allocatableSize);
        NN_UNUSED(totalFreeSize);
        NN_UNUSED(allocatableSizeForDisp);
        NN_UNUSED(totalFreeSizeForDisp);
        NN_UNUSED(allocatedSizeForDisp);

        NN_DETAIL_MEM_INFO("<heapinfo>\n");
        NN_DETAIL_MEM_INFO("  <offset>%llu</offset>\n", offset);
        NN_DETAIL_MEM_INFO("  <factor>%d</factor>\n", factor);
        NN_DETAIL_MEM_INFO("  <max_allocatable_size>%llu</max_allocatable_size>\n", allocatableSizeForDisp);
        NN_DETAIL_MEM_INFO("  <free_memory>%llu</free_memory>\n", totalFreeSizeForDisp);
        NN_DETAIL_MEM_INFO("  <allocated_memory>%llu</allocated_memory>\n", allocatedSizeForDisp);
        NN_DETAIL_MEM_INFO("  <freelists>\n");
        m_FreeList.Dump(offset, factor);
        NN_DETAIL_MEM_INFO("  </freelists>\n");
        NN_DETAIL_MEM_INFO("  <allocs>\n");
        m_Trie.Dump(offset, factor);
        NN_DETAIL_MEM_INFO("  </allocs>\n");
        NN_DETAIL_MEM_INFO("</heapinfo>\n");
    }

private:
    /**
     * @brief   領域のコンパクションをします。
     * @details 隣り合う同じ SpanStatus の Span を統合します。
     *          Free 時に両隣と統合されるため、あまり呼ばれないようにはなっている。
     */
    void Compaction() NN_NOEXCEPT
    {
        Span* pEnd = &m_SpanList;
        Span* it = pEnd->next;
        while (it != pEnd)
        {
            switch (it->status)
            {
            case SpanStatus_NotExist:
                while (it->next->status == SpanStatus_NotExist)
                {
                    // SpanStatus_NotExist が続くなら最初の 1 つだけ残して後の Span は解放
                    // 最初の 1 つの last を更新
                    Span* pRemoved = it->next;
                    RemoveSpan(pRemoved);
                    it->last = pRemoved->last;
                    FreeSpan(pRemoved);
                }
                break;
            case SpanStatus_OnFreeList:
                while (it->next->status == SpanStatus_OnFreeList)
                {
                    // 大きなフリーリストならば統合によってスロットの場所はかわらないことを利用
                    // FreeListの実装に依存した最適化
                    bool can_omit_remove_insert = it->first + 256 <= it->last;  // 大きい Span なら true
                    Span* pRemoved = it->next;
                    it->last = pRemoved->last;
                    m_FreeList.Remove(pRemoved);
                    RemoveSpan(pRemoved);
                    FreeSpan(pRemoved);
                    if (!can_omit_remove_insert)
                    {
                        // last を更新したことでサイズが変更されるので、
                        // 適切な freelist に入れなおす
                        m_FreeList.Remove(it);
                        m_FreeList.Insert(it);
                    }
                }
                break;
            default:
                break;
            }
            it = it->next;
        }
    }
    /**
     * @brief   SpanCache から Span ポインタのための領域を確保します。
     * @details m_SpanCacheはCompaction()の動作とそこからの立ち上がりを速くする。
     */
    Span* NewSpan() NN_NOEXCEPT
    {
        return m_SpanCache.Alloc();
    }
    /**
     * @brief   SpanCache へ Span ポインタのための領域を返却します。
     */
    void FreeSpan(Span* ptr) NN_NOEXCEPT
    {
        m_SpanCache.Free(ptr);
    }

private:
    Trie8x3 m_Trie;
    FreeList m_FreeList;
    SpanCache m_SpanCache;
    // 双方向循環リスト
    // 確保済みのものは別途 Trie8x3 にも入れられて管理され、
    // フリーのものは別途 FreeList にも入れられて管理される
    Span m_SpanList;
};

// ------------
// NumberLineAllocator
// ------------

void NumberLineAllocator::Initialize(MallocCallback mallocFunc, void* pMallocParam, FreeCallback freeFunc, void* pFreeParam) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_pNumberLine == NULL);

    void* ptr = mallocFunc(sizeof(NumberLineAllocatorPrv), pMallocParam);
    NumberLineAllocatorPrv* prv = new(ptr)NumberLineAllocatorPrv(mallocFunc, pMallocParam, freeFunc, pFreeParam);
    if (!prv)
    {
        NN_ABORT_UNLESS(prv != NULL);   //ENOMEM
    }
    errno_t e = prv->Init();
    if (e != 0)
    {
        prv->~NumberLineAllocatorPrv();
        m_FreeFunc(prv, pFreeParam);
        NN_ABORT_UNLESS(e == 0);
    }
    m_FreeFunc = freeFunc;
    m_pFreeParam = pFreeParam;
    m_pNumberLine = prv;
}

void NumberLineAllocator::Initialize(MallocCallback mallocFunc, void* pMallocParam, FreeCallback freeFunc, void* pFreeParam, bool isThreadSafe) NN_NOEXCEPT
{
    m_IsThreadSafe = isThreadSafe;
    Initialize(mallocFunc, pMallocParam, freeFunc, pFreeParam);
}

void NumberLineAllocator::Finalize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_pNumberLine);

    m_pNumberLine->~NumberLineAllocatorPrv();
    m_FreeFunc(m_pNumberLine, m_pFreeParam);
    m_pNumberLine = NULL;
}

bool NumberLineAllocator::AddRange(int first, int last) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(first >= 0);              // EINVAL
    NN_SDK_ASSERT(last > 0);                // EINVAL
    NN_SDK_ASSERT(first < 0x01000000);      // EINVAL
    NN_SDK_ASSERT(last <= 0x01000000);      // EINVAL
    NN_SDK_ASSERT(first < last);            // EINVAL

    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->AddRange(first, last);
        if (e != 0)
        {
            return false;
        }
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->AddRange(first, last);
        if (e != 0)
        {
            return false;
        }
        return true;
    }
}

bool NumberLineAllocator::RemoveRange(int first, int last) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(first >= 0);              // EINVAL
    NN_SDK_ASSERT(last > 0);                // EINVAL
    NN_SDK_ASSERT(first < 0x01000000);      // EINVAL
    NN_SDK_ASSERT(last <= 0x01000000);      // EINVAL
    NN_SDK_ASSERT(first < last);            // EINVAL

    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->RemoveRange(first, last);
        if (e != 0)
        {
            return false;
        }
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->RemoveRange(first, last);
        if (e != 0)
        {
            return false;
        }
        return true;
    }
}

bool NumberLineAllocator::Allocate(int* pOutFirst, int size) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(size > 0 && size <= 0x01000000);  // EINVAL

    Span* allocated;
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size));
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size));
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        return true;
    }
}

bool NumberLineAllocator::Allocate(int* pOutFirst, int size, void* userPtr, UserCallback userFunc) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(size > 0 && size <= 0x01000000);  // EINVAL

    Span* allocated;
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size));
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        allocated->u.UserData.pData = userPtr;
        allocated->u.UserData.dtor = userFunc;
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size));
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        allocated->u.UserData.pData = userPtr;
        allocated->u.UserData.dtor = userFunc;
        return true;
    }
}

bool NumberLineAllocator::Allocate(int* pOutFirst, int size, int alignment) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(alignment > 0);
    NN_SDK_ASSERT(size > 0 && size <= 0x01000000);  // EINVAL

    Span* allocated;
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size), alignment);
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        NN_SDK_ASSERT(allocated->first % alignment == 0);
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size), alignment);
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        NN_SDK_ASSERT(allocated->first % alignment == 0);
        return true;
    }
}

bool NumberLineAllocator::Allocate(int* pOutFirst, int size, int alignment, void* userPtr, UserCallback userFunc) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(alignment > 0);
    NN_SDK_ASSERT(size > 0 && size <= 0x01000000);  // EINVAL

    Span* allocated;
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size), alignment);
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        allocated->u.UserData.pData = userPtr;
        allocated->u.UserData.dtor = userFunc;
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->Alloc(&allocated, static_cast<uint32_t>(size), alignment);
        if (e != 0)
        {
            return false;
        }
        *pOutFirst = allocated->first;
        allocated->u.UserData.pData = userPtr;
        allocated->u.UserData.dtor = userFunc;
        return true;
    }
}

void NumberLineAllocator::Free(int first) NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(first < 0x01000000);  // EINVAL
    NN_SDK_ASSERT(first >= 0);          // EINVAL

    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->Free(first);
        NN_UNUSED(e);
        NN_SDK_ASSERT(e == 0);
    }
    else
    {
        errno_t e = m_pNumberLine->Free(first);
        NN_UNUSED(e);
        NN_SDK_ASSERT(e == 0);
    }
}

bool NumberLineAllocator::FindUserData(void** pOutUserPtr, int first) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(first < 0x01000000);  //EINVAL
    NN_SDK_ASSERT(first >= 0);          // EINVAL
    NN_SDK_ASSERT(pOutUserPtr);         // EINVAL

    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->FindUserData(pOutUserPtr, first);
        if (e != 0)
        {
            return false;
        }
        return true;
    }
    else
    {
        errno_t e = m_pNumberLine->FindUserData(pOutUserPtr, first);
        if (e != 0)
        {
            return false;
        }
        return true;
    }
}

int NumberLineAllocator::GetSizeOf(int first) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    NN_SDK_ASSERT(first < 0x01000000);  //EINVAL
    NN_SDK_ASSERT(first >= 0);          // EINVAL

    int outSize = 0;
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        errno_t e = m_pNumberLine->GetSizeOf(&outSize, first);
        NN_UNUSED(e);
        NN_SDK_ASSERT(e == 0);
        return outSize;
    }
    else
    {
        errno_t e = m_pNumberLine->GetSizeOf(&outSize, first);
        NN_UNUSED(e);
        NN_SDK_ASSERT(e == 0);
        return outSize;
    }
}

int NumberLineAllocator::GetTotalFreeSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        return m_pNumberLine->GetTotalFreeSize();
    }
    else
    {
        return m_pNumberLine->GetTotalFreeSize();
    }
}

int NumberLineAllocator::GetAllocatableSize() const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    if (m_IsThreadSafe)
    {
        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        return m_pNumberLine->GetAllocatableSize();
    }
    else
    {
        return m_pNumberLine->GetAllocatableSize();
    }
}

void NumberLineAllocator::Dump(uint64_t offset, int factor) const NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_pNumberLine);
    m_pNumberLine->Dump(offset, factor);
}

}} // namespace nn { namespace mem {
