﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/htcfs/detail/htcfs_ServiceName.h>

#include <mutex>
#include <nn/os/os_Mutex.h>
#include <nn/sf/sf_HipcClient.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include "htcfs_ServiceObjectUtil.h"

namespace nn { namespace htcfs {

namespace {

const int SessionCount = 3;
const int HeapSize = 16 * 1024;

// 拡張ヒープを使用したアロケータを、HIPC プロキシ用に準備
struct CreateFileSystemServiceObjectByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<HeapSize, CreateFileSystemServiceObjectByHipcTag> MyAllocator;

// MyAllocator を静的コンストラクタで初期化するためのヘルパー
class MyAllocatorInitializer
{
public:
    MyAllocatorInitializer() NN_NOEXCEPT
    {
        MyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_MyAllocatorInitializer;

nn::htcfs::IFileSystem* g_pFileSystem;
nn::os::Mutex g_FileSystemMutex(false);

nn::sf::HipcSimpleClientSessionManager& GetClientManager() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(nn::sf::HipcSimpleClientSessionManager, s_ClientManager);
    return s_ClientManager;
}

}

nn::sf::SharedPointer<IFileSystem> CreateFileSystemServiceObjectByHipc() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_FileSystemMutex);

    if (!g_pFileSystem)
    {
        nn::sf::SharedPointer<IFileSystem> fileSystem;
        auto result = GetClientManager().InitializeByName<IFileSystem, MyAllocator::Policy>(&fileSystem, detail::ServiceName);
        NN_ABORT_UNLESS(result.IsSuccess());

        GetClientManager().SetSessionCount(SessionCount);
        g_pFileSystem = fileSystem.Detach();
    }

    return nn::sf::SharedPointer<IFileSystem>(g_pFileSystem, false);
}

void DeleteFileSystemServiceObjectByHipc() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_FileSystemMutex);

    g_pFileSystem = nullptr;
    GetClientManager().Finalize();
}

}}
