﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_StringUtil.h>

#include "htc_TenvAllocator.h"
#include "pugixml.hpp"

namespace nn { namespace htc { namespace tenv { namespace detail {

    namespace {
        void DeallocateForPugiXml(void* addr) NN_NOEXCEPT
        {
            return Deallocate(addr, 0);
        }

        bool g_IsInitialized = false;
        struct StaticMutex
        {
            nn::os::MutexType mutex;
            void lock() NN_NOEXCEPT
            {
                nn::os::LockMutex(&mutex);
            }
            void unlock() NN_NOEXCEPT
            {
                nn::os::UnlockMutex(&mutex);
            }
        } g_InitializeMutex = { NN_OS_MUTEX_INITIALIZER(false) };

        void InitializeAllocatorForPugiXml() NN_NOEXCEPT
        {
            std::lock_guard<decltype(g_InitializeMutex)> scopedLock(g_InitializeMutex);
            if (!g_IsInitialized)
            {
                ::pugi::set_memory_management_functions(Allocate, DeallocateForPugiXml);
                g_IsInitialized = true;
            }
        }

        bool LoadXmlDocument(::pugi::xml_document* pDocument, const char* buffer, size_t bufferSize) NN_NOEXCEPT
        {
            InitializeAllocatorForPugiXml();

            ::pugi::xml_parse_result result = pDocument->load_buffer(buffer, bufferSize);

            if (result.status != ::pugi::xml_parse_status::status_ok)
            {
                return false;
            }

            return true;
        }

        bool GetXmlNodeChild(::pugi::xml_node* pOutValue, const ::pugi::xml_node& node, const char* name) NN_NOEXCEPT
        {
            const ::pugi::xml_node child = node.child(name);

            if (child.empty())
            {
                return false;
            }

            *pOutValue = child;

            return true;
        }

        const char* const XmlNodeTenvVariable = "TargetEnvironmentVariables";
        const char* const XmlNodeSystemVariable = "SystemVariables";
        const char* const XmlNodeUserVariable = "UserVariables";
        const char* const XmlAttributeName = "Name";
        const char* const XmlAttributeValueType = "ValueType";
    }

    const int SearchKeyLengthMax = 64;
    const int ValueLengthMax = 2048;

    bool SearchKey(char* outValue, size_t* outValueLength, const char* searchKey, const char* buffer, size_t bufferSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValueLength);
        NN_SDK_REQUIRES_NOT_NULL(searchKey);
        NN_SDK_REQUIRES_NOT_NULL(buffer);

        auto length = util::Strnlen(searchKey, SearchKeyLengthMax);
        if (length == SearchKeyLengthMax)
        {
            return false;
        }

        ::pugi::xml_document document;
        if (!LoadXmlDocument(&document, buffer, bufferSize))
        {
            return false;
        }

        // rootNode
        auto rootNode = ::pugi::xml_node();
        if (GetXmlNodeChild(&rootNode, document, XmlNodeTenvVariable))
        {
            // systemConfig, userConfig
            const char* const keys[] = { XmlNodeSystemVariable, XmlNodeUserVariable };
            for (auto& key : keys)
            {
                auto node = ::pugi::xml_node();
                if (!GetXmlNodeChild(&node, rootNode, key))
                {
                    continue;
                }

                for (auto& childNode : node.children())
                {
                    if (util::Strncmp(searchKey, childNode.attribute(XmlAttributeName).value(), length) == 0 && util::Strnlen(childNode.attribute(XmlAttributeName).value(), SearchKeyLengthMax) == length)
                    {
                        if (outValue != nullptr)
                        {
                            util::Strlcpy(outValue, childNode.child_value(), ValueLengthMax);
                        }

                        *outValueLength = util::Strnlen(childNode.child_value(), ValueLengthMax - 1);

                        return true;
                    }
                }
            }
        }

        return false;
    }

}}}}
