﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <mutex>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/writer/gc_Writer.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_EmbeddedDataHolder.h>
#include <nn/gc/detail/gc_StateMachine.h>
#include <nn/gc/detail/gc_DeviceDetector.h>
#include <nn/gc/detail/gc_AsicOperation.h>
#include <nn/gc/detail/gc_AsicHandler.h>
#include <nn/gc/detail/gc_AsicHandlerCore.h>

namespace nn { namespace gc { namespace writer {

static const uint32_t ReserveAddrSizeUnit = 8 * 1024 * 1024;

static const uint32_t Crc32Table[256] = {
    0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419,
    0x706af48f, 0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4,
    0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07,
    0x90bf1d91, 0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
    0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7, 0x136c9856,
    0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
    0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4,
    0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
    0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3,
    0x45df5c75, 0xdcd60dcf, 0xabd13d59, 0x26d930ac, 0x51de003a,
    0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599,
    0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
    0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190,
    0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f,
    0x9fbfe4a5, 0xe8b8d433, 0x7807c9a2, 0x0f00f934, 0x9609a88e,
    0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
    0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed,
    0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
    0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3,
    0xfbd44c65, 0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
    0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a,
    0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5,
    0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa, 0xbe0b1010,
    0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
    0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17,
    0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6,
    0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615,
    0x73dc1683, 0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
    0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1, 0xf00f9344,
    0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
    0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a,
    0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
    0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1,
    0xa6bc5767, 0x3fb506dd, 0x48b2364b, 0xd80d2bda, 0xaf0a1b4c,
    0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef,
    0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
    0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe,
    0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31,
    0x2cd99e8b, 0x5bdeae1d, 0x9b64c2b0, 0xec63f226, 0x756aa39c,
    0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
    0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b,
    0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
    0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1,
    0x18b74777, 0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
    0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45, 0xa00ae278,
    0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7,
    0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc, 0x40df0b66,
    0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
    0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605,
    0xcdd70693, 0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8,
    0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b,
    0x2d02ef8d
};

namespace {

uint32_t CalcCrc32(const uint8_t *frame, size_t frame_len)
{
    size_t i;
    uint32_t crc;

    crc = 0xFFFFFFFF;
    for (i = 0; i < frame_len; i++)
    {
        crc = Crc32Table[(crc ^ frame[i]) & 0xff] ^ (crc >> 8);
    }
    return ~crc;
}

}

class GcWriter
{
    NN_DISALLOW_COPY(GcWriter);

private:
    static const int ParameterHeadWritableAreaSize = 73;    // 書き込みパラメタで先頭から見たとき、初めて read only の MEM_CAP にあたるまでの領域のサイズ
    static const size_t   ReserveAreaStartAddrSize = 3;
    static const uint32_t CertAreaEndPageAddress = 0x80;

    detail::DevCardParameter m_DevCardParameter;
    detail::DevCardNandSize m_NandSize;
    uint32_t m_ReservedAreaStartAddr;
    MemorySize m_MemorySize;
    bool m_IsParameterSet;
    bool m_IsVerifyEnable;

public:
    GcWriter() NN_NOEXCEPT
    {
        m_NandSize = detail::DevCardNandSize_32GB;
        m_MemorySize = MemorySize_1GB;
        m_ReservedAreaStartAddr = 0;
        m_IsParameterSet = false;
        m_IsVerifyEnable = false;
        memset(&m_DevCardParameter, 0x00, sizeof(detail::DevCardParameter));
    }

    void ChangeMode(const AsicMode asicMode) NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();

        if(asicMode == AsicMode_WriteMode)
        {
            detail::StateMachine::GetInstanceForAsicHandler().SelectAsicFirmware(detail::AsicFirmwareType_Write);
        }
        else
        {
            // Read 用の fw をセット
            detail::StateMachine::GetInstanceForAsicHandler().SelectAsicFirmware(detail::AsicFirmwareType_Read);
        }
        detail::StateMachine::GetInstanceForAsicHandler().SetForceResetFlag();

        nn::Result resultToBeDiscarded = detail::AsicHandler::GetInstance().OrderWork(detail::AsicWork_Deactivate);
        NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(resultToBeDiscarded, "[writer] Deactivate failed (Module:%d, Description:%d)\n",
            resultToBeDiscarded.GetModule(), resultToBeDiscarded.GetDescription());
        NN_UNUSED(resultToBeDiscarded);
    }

    nn::Result ActivateForWriter() NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();

        detail::AsicHandlerCore& handlerCore = detail::AsicHandlerCore::GetInstance();
        detail::AsicOperation& asicOp = detail::AsicOperation::GetInstance();
        detail::StateMachine::StateMachineForAsicHandler& stateManager = detail::StateMachine::GetInstanceForAsicHandler();
        m_IsParameterSet = false;

        // 初期化の終わりを待つ
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(detail::AsicHandler::GetInstance().OrderWork(detail::AsicWork_None), "[writer] wait for thread failed\n");

        // カードバス有効化
        nn::Result result = handlerCore.EnableCardBusForWriter();
        NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Enable card bus failed\n");
        NN_RESULT_DO( handlerCore.InvalidateAsicIfResultFailure(result) );

        // 状態を更新
        NN_RESULT_DO( stateManager.TransitStateWithActivateGameCard() );

        // ID1, ID2 の read
        detail::CardId1 normalId1;
        detail::CardId2 normalId2;
        result = asicOp.SendCardReadId1(&normalId1);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read Id1 Failure\n");

        result = asicOp.SendCardReadId2(&normalId2);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read Id2 Failure\n");

        // CHG_DBG_MODE 発行
        {
            char resultBuffer[512];
            result = asicOp.ChangeGcModeToDebug(resultBuffer, sizeof(resultBuffer));
            NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] CHG_DBG_MODE Failure\n");
            if(CompareAllMemory(&normalId2, resultBuffer, sizeof(detail::CardId2)) != 0)
            {
                NN_DETAIL_GC_ERR_LOG("[writer] ID2 mismatch in debug mode\n");
                return nn::fs::ResultGameCardDebugCardReceivedIdMismatch();
            }
        }

        // 状態を更新
        NN_RESULT_DO( stateManager.TransitStateWithDebugGameCard() );

        // ID1, ID2 一致確認
        detail::CardId1 debugId1;
        detail::CardId2 debugId2;
        NN_RESULT_DO( asicOp.SendCardReadId1(&debugId1) );
        NN_RESULT_DO( asicOp.SendCardReadId2(&debugId2) );
        if( CompareAllMemory(&normalId1, &debugId1, sizeof(detail::CardId1)) )
        {
            NN_DETAIL_GC_ERR_LOG("[writer] ID1 mismatch\n");
            return nn::fs::ResultGameCardDebugCardId1Mismatch();
        }
        else if( CompareAllMemory(&normalId2, &debugId2, sizeof(detail::CardId2)) )
        {
            NN_DETAIL_GC_ERR_LOG("[writer] ID2 mismatch\n");
            return nn::fs::ResultGameCardDebugCardId2Mismatch();
        }

        // カードの容量を聞かれる事があるので NAND サイズを取得しておく
        detail::DevCardParameter readDevCardParam;
        result = asicOp.SendCardReadParameter(&readDevCardParam);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read Param After Activate Failure\n");
        m_NandSize = readDevCardParam.nandSize;

        return result;
    }

    static MemorySize ConvertNandSizeToMemorySize(const detail::DevCardNandSize nandSize) NN_NOEXCEPT
    {
        MemorySize memorySize = MemorySize_32GB;
        switch(nandSize)
        {
        case detail::DevCardNandSize_16GB: memorySize = MemorySize_16GB; break;
        case detail::DevCardNandSize_32GB: memorySize = MemorySize_32GB; break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        return memorySize;
    }

    nn::Result GetCardAvailableRawSize(int64_t* pOutValue) NN_NOEXCEPT
    {
        MemorySize memorySize = ConvertNandSizeToMemorySize(m_NandSize);
        *pOutValue = static_cast<int64_t>(memorySize) * detail::AvailableSizeBase;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetCardAvailableSize(int64_t* pOutValue) NN_NOEXCEPT
    {
        *pOutValue = static_cast<int64_t>(m_MemorySize) * detail::AvailableSizeBase;
        NN_RESULT_SUCCESS;
    }

    nn::Result EraseAndWriteParameter(const MemorySize memorySize, const uint32_t secureAreaStartPageAddress) NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();
        SetDevCardParameter(memorySize, secureAreaStartPageAddress);

        // Write Parameter
        NN_RESULT_DO(this->WriteDevCardParam(m_DevCardParameter));

        // Read Parameter
        detail::DevCardParameter readDevCardParam;
        NN_RESULT_DO(this->ReadDevCardParam(&readDevCardParam));

        // compare
        if(CompareDevCardParameters(&readDevCardParam) == false)
        {
            NN_DETAIL_GC_ERR_LOG("[writer] Parameter mismatch\n");
            return nn::fs::ResultGameCardDebugParameterMismatch();
        }
        NN_RESULT_SUCCESS;
    }

    nn::Result Write(void* inDataBuffer, const size_t dataBufferSize, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();

        int64_t cardSize = 0;
        GetCardAvailableSize(&cardSize);

        NN_SDK_REQUIRES(pageAddress < (cardSize / GcPageSize));
        NN_SDK_REQUIRES(pageCount <= ((cardSize / GcPageSize) - pageAddress));

        const size_t writeSize = pageCount * GcPageSize;
        NN_SDK_REQUIRES(writeSize <= dataBufferSize);

        // 書き出し領域のチェック
        if(pageCount == 0)
        {
            NN_DETAIL_GC_WARNING_LOG("[writer] 0 size write occurred\n");
            NN_RESULT_SUCCESS;
        }

        if(!m_IsParameterSet)
        {
            ReadAndSetDevCardParameter();
            m_IsParameterSet = true;
        }
        bool isCrcCheckEnable = m_IsVerifyEnable;
        if(pageAddress < CertAreaEndPageAddress || pageAddress >= m_ReservedAreaStartAddr || static_cast<uint64_t>(pageAddress) + pageCount >= m_ReservedAreaStartAddr)
        {
            isCrcCheckEnable = false;
        }
        detail::AsicOperation& asicOp = detail::AsicOperation::GetInstance();
        uint32_t expCrc = 0;
        if(isCrcCheckEnable)
        {
            expCrc = CalcCrc32(reinterpret_cast<uint8_t*>(inDataBuffer), writeSize);
        }
        NN_RESULT_DO(asicOp.SendCardWritePage(reinterpret_cast<char*>(inDataBuffer), writeSize, pageAddress, pageCount));
        if(isCrcCheckEnable)
        {
            uint32_t curCrc;
            NN_RESULT_DO(asicOp.SendCardReadCrc(&curCrc, sizeof(uint32_t), pageAddress, pageCount));
            if(expCrc != curCrc)
            {
                NN_DETAIL_GC_ERR_LOG("[writer] CRC mismatch - addr : %x, expected : %x, actual : %x\n",pageAddress, expCrc, curCrc);
                return nn::fs::ResultGameCardDebugWriteCrcMismatch();
            }
        }
        NN_RESULT_SUCCESS;
    }

    Result WriteDevCardParam(detail::DevCardParameter& devCardParameter) NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();
        detail::AsicOperation& asicOp = detail::AsicOperation::GetInstance();

        // Erase
        nn::Result result = asicOp.SendCardErase();
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Erase Failure\n");

        // Write Parameter
        result = asicOp.SendCardWriteParameter(devCardParameter);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Write Param Failure\n");

        // 必要な設定
        result = ReadAndSetDevCardParameter();
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] ReadAndSetDevCardParameter Failure\n");
        NN_RESULT_SUCCESS;
    }

    Result ReadDevCardParam(detail::DevCardParameter* pOutValue) NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();
        detail::AsicOperation& asicOp = detail::AsicOperation::GetInstance();

        // Read Parameter
        nn::Result result = asicOp.SendCardReadParameter(pOutValue);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read Dev Card Param Failure\n");
        m_NandSize = pOutValue->nandSize;
        NN_RESULT_SUCCESS;
    }

    void SetVerifyEnalbleFlag(bool isVerifyEnable) NN_NOEXCEPT
    {
        m_IsVerifyEnable = isVerifyEnable;
    }

    Result ForceErase() NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();
        detail::AsicOperation& asicOp = detail::AsicOperation::GetInstance();
        detail::AsicHandlerCore& handlerCore = detail::AsicHandlerCore::GetInstance();
        detail::StateMachine::StateMachineForAsicHandler& stateManager = detail::StateMachine::GetInstanceForAsicHandler();

        // 初期化の終わりを待つ
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(detail::AsicHandler::GetInstance().OrderWork(detail::AsicWork_None), "[writer] wait for thread failed\n");

        // カードバス有効化
        nn::Result result = handlerCore.EnableCardBusForWriter();
        NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Enable card bus failed\n");
        NN_RESULT_DO( handlerCore.InvalidateAsicIfResultFailure(result) );
        NN_RESULT_DO( stateManager.TransitStateWithActivateGameCard() );

        // ID1 の read
        detail::CardId1 normalId1;
        result = asicOp.SendCardReadId1(&normalId1);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read Id1 Failure\n");

        NN_RESULT_DO(asicOp.SendCardChangeDebugDirect());

        // 状態を更新
        NN_RESULT_DO( stateManager.TransitStateWithDebugGameCard() );

        // Read Parameter
        detail::DevCardParameter readDevCardParam;
        result = detail::AsicOperation::GetInstance().SendCardReadParameter(&readDevCardParam);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read and Set Param Failure\n");

        memset(&m_DevCardParameter, 0x00, sizeof(m_DevCardParameter));

        InitializeParameter(MemorySize_1GB, 0x1000);

        // Write Parameter
        NN_RESULT_DO(this->WriteDevCardParam(m_DevCardParameter));

        detail::CardId1 debugId1;
        NN_RESULT_DO( asicOp.SendCardReadId1(&debugId1) );

        NN_RESULT_SUCCESS;
    }

private:
    static uint32_t GetReservedAreaStartAddr(const MemorySize memorySize) NN_NOEXCEPT
    {
        return static_cast<uint32_t>(memorySize) * detail::AvailableSizeBase / GcPageSize;
    }

    static void SetMemorySizeToParameter(detail::DevCardParameter* pOutDevCardParam, const MemorySize memorySize) NN_NOEXCEPT
    {
        switch(memorySize)
        {
        case MemorySize_1GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_1GB;
                pOutDevCardParam->romSize                          = detail::DevCardRomSize_1GB;
            }
            break;
        case MemorySize_2GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_2GB;
                pOutDevCardParam->romSize                          = detail::DevCardRomSize_2GB;
            }
            break;
        case MemorySize_4GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_4GB;
                pOutDevCardParam->romSize                          = detail::DevCardRomSize_4GB;
            }
            break;
        case MemorySize_8GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_8GB;
                pOutDevCardParam->romSize                          = detail::DevCardRomSize_8GB;
            }
            break;
        case MemorySize_16GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_16GB;
                pOutDevCardParam->romSize                          = detail::DevCardRomSize_16GB;
            }
            break;
        case MemorySize_32GB:
            {
                pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_32GB;
                pOutDevCardParam->romSize                          = detail::DevCardRomSize_32GB;
            }
            break;
        default:
            pOutDevCardParam->cardId1.memoryCapacity           = detail::MemoryCapacity_1GB;
            pOutDevCardParam->romSize                          = detail::DevCardRomSize_1GB;
        }
    }

    void InitializeParameter(MemorySize memorySize, const uint32_t secureAreaStartPageAddress) NN_NOEXCEPT
    {
        m_DevCardParameter.cardId1.makerCode                = detail::MakerCodeForCardId1_Lapis;
        m_DevCardParameter.cardId1.memoryType               = detail::MemoryType_T1RomFast;
        m_DevCardParameter.cardId2.cardSecurityNumber       = detail::CardSecurityNumber_2;
        m_DevCardParameter.romAreaStartAddr                 = secureAreaStartPageAddress / 64;
        m_DevCardParameter.backupAreaStartAddr              = 0xFFFFFFFF;
        m_DevCardParameter.nandSize                         = detail::DevCardNandSize_32GB; // Read Only Rarameter のはず
        SetMemorySizeToParameter(&m_DevCardParameter, memorySize);

        // Memory Size 及び ReservedArea の開始アドレスを設定
        m_MemorySize = memorySize;
        m_ReservedAreaStartAddr = GetReservedAreaStartAddr(m_MemorySize);
        uint32_t reserveParamValue = m_ReservedAreaStartAddr / (ReserveAddrSizeUnit / GcPageSize);

        // ReservedArea の書き込みパラメータへの反映
        memcpy(m_DevCardParameter.reserveAreaStartAddr, &reserveParamValue, ReserveAreaStartAddrSize);

        m_IsParameterSet = true;

        // Memory Size 及び ReservedArea の開始アドレスを設定
        m_MemorySize = ConvertRomSizeToMemorySize(m_DevCardParameter.romSize);
        m_ReservedAreaStartAddr = GetReservedAreaStartAddr(m_MemorySize);
    }

    void SetDevCardParameter(MemorySize memorySize, const uint32_t secureAreaStartPageAddress) NN_NOEXCEPT
    {
        // 書き込みパラメータのセット
        nn::Result resultToBeDiscarded = ReadAndSetDevCardParameter();
        NN_DETAIL_GC_RESULT_ERR_LOG(resultToBeDiscarded);
        NN_UNUSED(resultToBeDiscarded);
        InitializeParameter(memorySize, secureAreaStartPageAddress);
    }

    nn::Result ReadAndSetDevCardParameter() NN_NOEXCEPT
    {
        NN_DETAIL_GC_THREAD_DEBUG_LOG();

        // Read Parameter
        detail::DevCardParameter readDevCardParam;
        nn::Result result = detail::AsicOperation::GetInstance().SendCardReadParameter(&readDevCardParam);
        NN_DETAIL_GC_RESULT_DO_ERR_LOG_MESSAGE(result, "[writer] Read and Set Param Failure\n");

        // Set Parameter
        memcpy(&m_DevCardParameter, &readDevCardParam, sizeof(detail::DevCardParameter));

        // Memory Size 及び ReservedArea の開始アドレスを設定
        m_MemorySize = ConvertRomSizeToMemorySize(m_DevCardParameter.romSize);
        m_ReservedAreaStartAddr = GetReservedAreaStartAddr(m_MemorySize);

        NN_RESULT_SUCCESS;
    }

    static MemorySize ConvertRomSizeToMemorySize(const detail::DevCardRomSize romsize) NN_NOEXCEPT
    {
        MemorySize memorySize = MemorySize_1GB;
        switch(romsize)
        {
        case detail::DevCardRomSize_1GB  : memorySize = MemorySize_1GB;  break;
        case detail::DevCardRomSize_2GB  : memorySize = MemorySize_2GB;  break;
        case detail::DevCardRomSize_4GB  : memorySize = MemorySize_4GB;  break;
        case detail::DevCardRomSize_8GB  : memorySize = MemorySize_8GB;  break;
        case detail::DevCardRomSize_16GB : memorySize = MemorySize_16GB; break;
        case detail::DevCardRomSize_32GB : memorySize = MemorySize_32GB; break;
        default:
            break;
        }
        return memorySize;
    }

    bool CompareDevCardParameters(detail::DevCardParameter* param) NN_NOEXCEPT
    {
        // write 可能な領域（MEM_CAP まで）を見る
        return CompareAllMemory(&m_DevCardParameter, param, ParameterHeadWritableAreaSize) == 0;

        // read only な MEM CAP は比較を飛ばす
        // 以降の reserved 領域は今は何もないので特に比較しない
    }
};

namespace
{

GcWriter g_GcWriter;

nn::Result GetRmaInformationImpl(RmaInformation* pOutValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT(detail::StateMachine::GetInstance().IsWriterFwSet());
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE( detail::AsicHandler::GetInstance().OrderWork(detail::AsicWork_None), "[writer] wait for thread failed\n" );
    return detail::AsicOperation::GetInstance().ReadRmaInformation(pOutValue);
}

nn::Result HandleWriterResult(const nn::Result result) NN_NOEXCEPT
{
    NN_DETAIL_GC_RESULT_ERR_LOG(result);

    // sdmmc のエラー Result の場合、内部 Result に置換して返す（ResultCommunicationError を一時的に使用しているが、Module だけ取得できればそれでよい）
    if(result.GetModule() == nn::sdmmc::ResultCommunicationError().GetModule())
    {
        NN_DETAIL_GC_ERR_LOG("[writer] sdmmc result (Module:%d, Description:%d) will be replaced to ResultGameCardCardAccessFailure...\n", result.GetModule(), result.GetDescription());
        return nn::fs::ResultGameCardCommunicationFailure();
    }
    return result;
}

}  // namespace

void ChangeMode(const AsicMode asicMode) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    g_GcWriter.ChangeMode(asicMode);
}

nn::Result ActivateForWriter() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result = g_GcWriter.ActivateForWriter();
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

nn::Result GetCardAvailableRawSize(int64_t* pOutValue) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result = g_GcWriter.GetCardAvailableRawSize(pOutValue);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

nn::Result EraseAndWriteParameter(const MemorySize memorySize, const uint32_t secureAreaStartPageAddress) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result = g_GcWriter.EraseAndWriteParameter(memorySize, secureAreaStartPageAddress);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

nn::Result Write(void* inDataBuffer, const size_t dataBufferSize, const uint32_t pageAddress, const uint32_t pageCount) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    nn::Result result = g_GcWriter.Write(inDataBuffer, dataBufferSize, pageAddress, pageCount);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

void SetVerifyEnalbleFlag(bool isVerifyEnable) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    g_GcWriter.SetVerifyEnalbleFlag(isVerifyEnable);
}


// *** 以下 特殊用途向き API ***
void SetUserAsicFirmwareBuffer(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(bufferLength == nn::gc::detail::GcAsicFirmwareSize);
    memcpy(detail::EmbeddedDataHolder::g_FwWriterBuffer, buffer, nn::gc::detail::GcAsicFirmwareSize);
}

nn::Result GetRmaInformation(RmaInformation* pOutValue) NN_NOEXCEPT
{
    nn::Result result = GetRmaInformationImpl(pOutValue);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

Result WriteDevCardParam(detail::DevCardParameter& devCardParameter) NN_NOEXCEPT
{
    nn::Result result = g_GcWriter.WriteDevCardParam(devCardParameter);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

Result ReadDevCardParam(detail::DevCardParameter* pOutValue) NN_NOEXCEPT
{
    nn::Result result = g_GcWriter.ReadDevCardParam(pOutValue);
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

Result ForceErase() NN_NOEXCEPT
{
    nn::Result result = g_GcWriter.ForceErase();
    NN_DETAIL_GC_RESULT_ERR_LOG_MESSAGE(result, "[writer] ForceErase Failure in %s()\n", __FUNCTION__);
    return HandleWriterResult(result);
}

}}} // namespace nn { namespace gc { namespace writer
