﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>

#include <nn/os.h>
#include <nn/gc/detail/gc_Define.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/gc/detail/gc_AsicHandler.h>

namespace nn { namespace gc {
namespace detail {

#if defined(NN_DETAIL_GC_DEBUG_MEMDUMP)
    static char g_QueueBufferCommonDataBuffer[1024];
    char* QueueBuffer::g_CommonDataBuffer = g_QueueBufferCommonDataBuffer;
#else
    char* QueueBuffer::g_CommonDataBuffer = NULL;
#endif

int CompareAllMemory(const void* buffer1, const void* buffer2, const size_t bufferLength) NN_NOEXCEPT
{
    int retValue = 0;
    for(size_t i = 0; i < bufferLength; i++)
    {
        retValue |= reinterpret_cast<const char*>(buffer1)[i] ^ reinterpret_cast<const char*>(buffer2)[i];
    }
    return retValue;
}


// *** QueueBuffer ***

void QueueBuffer::Append(const char* buffer, const size_t bufferLength) NN_NOEXCEPT
{
    memcpy(dataBuffer + length, buffer, bufferLength);
    length += bufferLength;
}

// *** RingBuffer ***

template <typename T>
RingBuffer<T>::RingBuffer(T* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT
{
    Init();
    m_Buffer = dataBuffer;
    m_BufferLength = dataBufferLength;
}

template <typename T>
void RingBuffer<T>::Print() NN_NOEXCEPT
{
    NN_SDK_LOG("[RingBuffer] size: %zu, length: %zu, index: %zu\n", m_BufferLength, m_Length, m_AtIndex);
}

template <typename T>
void RingBuffer<T>::Enqueue(T* dataBuffer, const size_t dataBufferLength) NN_NOEXCEPT
{
    for(size_t i = 0; i < dataBufferLength; i++)
    {
        m_Buffer[m_AtIndex] = dataBuffer[i];
        m_AtIndex = GetIndexWithOffset(1);
        // m_Length は最大でも m_BufferLength の長さ
        if(m_Length < m_BufferLength)
        {
            m_Length++;
        }
    }
}

template <typename T>
size_t RingBuffer<T>::Dequeue(T* outBuffer, const size_t outBufferLength) NN_NOEXCEPT
{
    size_t numDelete = GetMinimumLength(outBufferLength);
    for(size_t i = 0; i < numDelete; i++)
    {
        size_t startIndex = GetStartIndex();
        outBuffer[i] = m_Buffer[startIndex];
        m_Length--;
    }
    return numDelete;
}

template <typename T>
size_t RingBuffer<T>::DeleteHead(const size_t length) NN_NOEXCEPT
{
    size_t numDelete = GetMinimumLength(length);
    m_Length -= numDelete;
    return numDelete;
}

template <typename T>
size_t RingBuffer<T>::DeleteTail(const size_t length) NN_NOEXCEPT
{
    size_t numDelete = GetMinimumLength(length);
    m_AtIndex = GetIndexWithInverseOffset(numDelete);
    m_Length -= numDelete;
    return numDelete;
}

template <typename T>
size_t RingBuffer<T>::GetHead(T* outBuffer, const size_t outBufferLength) NN_NOEXCEPT
{
    size_t numOutput = GetMinimumLength(outBufferLength);
    for(size_t i = 0; i < numOutput; i++)
    {
        size_t index = GetStartIndexWithOffset(i);
        outBuffer[i] = m_Buffer[index];
    }
    return numOutput;
}

template <typename T>
size_t RingBuffer<T>::GetTail(T* outBuffer, const size_t outBufferLength) NN_NOEXCEPT
{
    size_t numOutput = GetMinimumLength(outBufferLength);
    for(size_t i = 0; i < numOutput; i++)
    {
        size_t index = GetIndexWithInverseOffset( numOutput - i );
        outBuffer[i] = m_Buffer[index];
    }
    return numOutput;
}

// このファイル内の関数の各 typename ごとのコードを生成するため必要
template class RingBuffer<char>;
template class RingBuffer<AsicWorkInfo>;


// *** CharRingBuffer

CharRingBuffer::CharRingBuffer() NN_NOEXCEPT
{
}

size_t CharRingBuffer::DequeueLine(char* outBuffer, const size_t outBufferLength) NN_NOEXCEPT
{
    size_t numDelete = GetMinimumLength(outBufferLength);
    for(size_t i = 0; i < numDelete; i++)
    {
        size_t startIndex = GetStartIndex();
        outBuffer[i] = m_Buffer[startIndex];
        m_Length--;
        if(outBuffer[i] == '\0')
        {
            return i + 1;
        }
    }
    if (numDelete < outBufferLength)
    {
        outBuffer[numDelete] = '\0';
        numDelete++;
    }
    else if (outBufferLength > 0)
    {
        outBuffer[outBufferLength - 1] = '\0';
    }
    return numDelete;
}


} } }
