﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>

#include <nn/os.h>
#include <nn/os/os_Thread.h>

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>
#include <nn/gc/detail/gc_AsicHandler.h>
#include <nn/gc/detail/gc_DeviceDetector.h>
#include <nn/gc/detail/gc_GeneralIo.h>

namespace nn { namespace gc {
namespace detail {

// NOTE: コンストラクタ内で GPIO にアクセス
DeviceDetector::DeviceDetector() NN_NOEXCEPT
{
    m_IsInserted = GeneralIo::GetInstance().GetGpioDetectPin();
    m_IsPaused = false;
    m_IsRemovedDuringSleep = false;
    m_pGpioDetectRawEvent = nullptr;
    NN_DETAIL_GC_LOG("DeviceDetector: initial state: %s Inserted\n", m_IsInserted ? " " : "Not");

    std::memset(&m_DeviceDetectorErrorInfo, 0, sizeof(m_DeviceDetectorErrorInfo));
    m_DeviceDetectorErrorInfo.isInserted = m_IsInserted;
    if(m_IsInserted)
    {
        m_DeviceDetectorErrorInfo.insertionCount++;
    }

    InitializeGcCallback(&m_CardRemovalEventCallback);
}

DeviceDetector& DeviceDetector::GetInstance() NN_NOEXCEPT
{
    static DeviceDetector s_Instance;
    return s_Instance;
}

void DeviceDetector::InitializeThread() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    ThreadInterface::InitializeThread(m_ThreadStack, ThreadStackSize, NN_SYSTEM_THREAD_NAME(gc, DeviceDetector), NN_SYSTEM_THREAD_PRIORITY(gc, DeviceDetector));
}

void DeviceDetector::InitializeWaitEvents() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    ThreadInterface::InitializeWaitEvents();

    // イベントの初期化
    m_pGpioDetectRawEvent = GeneralIo::GetInstance().GetDetectEvent();
    nn::os::InitializeEvent(&m_GpioDetectFilteredEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_PauseRequestEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_ResumeRequestEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_ResumeCompleteEvent, false, nn::os::EventClearMode_ManualClear);

    // GPIO Detect 端子
    nn::os::InitializeMultiWaitHolder(&m_GpioDetectRawEventHolder, m_pGpioDetectRawEvent);
    nn::os::InitializeMultiWaitHolder(&m_PauseRequestEventHolder, &m_PauseRequestEvent);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_GpioDetectRawEventHolder);
    nn::os::LinkMultiWaitHolder(&m_Waiter, &m_PauseRequestEventHolder);
}

void DeviceDetector::FinalizeWaitEvents() NN_NOEXCEPT
{
    nn::os::UnlinkMultiWaitHolder(&m_GpioDetectRawEventHolder);
    nn::os::UnlinkMultiWaitHolder(&m_PauseRequestEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_GpioDetectRawEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_PauseRequestEventHolder);

    nn::os::FinalizeEvent(&m_GpioDetectFilteredEvent);
    nn::os::FinalizeEvent(&m_PauseRequestEvent);
    nn::os::FinalizeEvent(&m_ResumeRequestEvent);
    nn::os::FinalizeEvent(&m_ResumeCompleteEvent);

    ThreadInterface::FinalizeWaitEvents();
}

void DeviceDetector::RegisterCardRemovalEventCallback(const GcCallbackStruct* pCardRemovalEventCallback) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    m_CardRemovalEventCallback = *(pCardRemovalEventCallback);
}

void DeviceDetector::UnregisterCardRemovalEventCallback() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();
    InitializeGcCallback(&m_CardRemovalEventCallback);
}

void DeviceDetector::RunThreadFunctionImpl() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    GeneralIo& generalIo = GeneralIo::GetInstance();
    bool isInsertionStateStable = false;

    // DeviceDetector は初期状態はシグナルしない

    // 初回の処理
    {
        // 初回の挿入状態を確定させる
        GetGpioDetectPin();

        // スレッド起動前に GPIO 状態を見られている可能性があるため、状態が変わっている際はシグナルする
        generalIo.ClearGpioDetectEvent();
        CheckDetectStateChangeAndSignalEvent();
    }


    while(NN_STATIC_CONDITION(true))
    {
        nn::os::MultiWaitHolderType* signaledHolder = NULL;

        // イベント待ち
        if(isInsertionStateStable == true)
        {
            // 状態が安定しているならイベントが起きるまで待つ
            signaledHolder = nn::os::WaitAny(&m_Waiter);

            // イベントが起きた直後は状態が安定していない
            isInsertionStateStable = false;
        }
        else
        {
            // 状態が安定していないので、Debounce time だけ待つ
            // （まだ処理していないイベントも溜まっている場合は瞬時に次へ進むが、 continue でループする）
            signaledHolder = nn::os::TimedWaitAny(&m_Waiter, nn::TimeSpan::FromMilliSeconds(GcDetectDebounceTimeMsec));
        }


        // スレッド終了要請であれば終了する
        if(signaledHolder == &m_ThreadEndEventHolder)
        {
            NN_DETAIL_GC_CLEAR_EVENT(&m_ThreadEndEvent);
            break;
        }
        // 一時停止要求イベントだったら Pause させる
        else if(signaledHolder == &m_PauseRequestEventHolder)
        {
            // この関数内でイベントを待ち受けながら一時停止する
            WaitForResume();

            // 一時停止復帰後は状態が不安定とみなし、再開
            isInsertionStateStable = false;
            continue;
        }
        // wait 中に GPIO イベントが起こった場合
        else if(signaledHolder == &m_GpioDetectRawEventHolder)
        {
            NN_DETAIL_GC_LOG("gpio event detected!\n");

            // イベントがあったのならクリアする（イベントを受け取っていない時にクリアしない）
            generalIo.ClearGpioDetectEvent();

            // ここがイベント検知直後の処理を行う場所であり、ここまでに挿入状態だったら「抜け」イベントを発行して「抜け」状態へ
            //（前の状態が「挿入」なら、挿 → 挿, 挿 → 抜 のどちらの場合も「抜け」処理が必要）
            if(m_IsInserted == true)
            {
                SetDetectStateAndSignalEvent(false);
            }

            // 波形が安定するまで繰り返し wait event
            continue;
        }


        // *** Debounce time 内にイベントがなかった場合はここまで到達し、状態が確定している ***
        if(signaledHolder == NULL)
        {
            isInsertionStateStable = true;
            bool isGpioInserted = generalIo.GetGpioDetectPin();

            // イベントがなかったと判定した後にイベントが起きて、確定的な DetectPin 状態ではない可能性がある
            if(nn::os::TryWaitSystemEvent(m_pGpioDetectRawEvent))
            {
                // 実はイベントが起きていたら、イベントをクリアせずにやりなおし
                continue;
            }

            // 「挿入」確定状態
            if(isGpioInserted == true)
            {
                // 抜 → 挿 のケース
                if(m_IsInserted == false)
                {
                    // ここが「挿入」確定直後の処理を行う場所であり、「挿入」イベントを発行して「挿入」状態へ
                    SetDetectStateAndSignalEvent(true);
                }
                //（挿 → 挿 のケースは、挿入確定前にイベントが起きて必ず抜去イベントを処理しているはず）
            }
            // 「抜け」確定状態
            else
            {
                // 挿 → 抜 のケース
                if(m_IsInserted == true)
                {
                    // ここのフローは抜去の検知が GcDetectDebounceTimeMsec だけ遅れるため通らないようにするべきで、
                    // 初回シーケンスについてもこの関数の初めに処理を行っているので、実際この箇所を通ることはないはず
                    NN_DETAIL_GC_ERR_LOG("DeviceDetector: Unhandled event: insert -> removed\n");
                }
            }
        }
    }
}

void DeviceDetector::WaitForResume() NN_NOEXCEPT
{
    GeneralIo& generalIo = GeneralIo::GetInstance();

    // 一時停止遷移要求イベントのクリア
    nn::os::ClearEvent(&m_PauseRequestEvent);

    // 一時停止準備完了イベントのシグナル
    nn::os::SignalEvent(&m_ResumeCompleteEvent);

    // ** 一時停止復帰待ち ***
    nn::os::WaitEvent(&m_ResumeRequestEvent);
    nn::os::ClearEvent(&m_ResumeRequestEvent);
    m_IsRemovedDuringSleep = false;

    // gpio wake のデバッグ用ログ
    if(generalIo.IsGpioDetectWakeEventActive())
    {
        NN_DETAIL_GC_LOG("gpio wake: card %s\n", m_IsInserted ? "removed" : "inserted");
    }

    // スリープ直前まで挿入されていた場合の抜去チェック
    if(m_IsInserted)
    {
        // カードが現在抜けているか、gpio イベントが起きているか、gpio wake フラグが立っているか
        //（gpio wake フラグはここに来るときには必ず更新されている）
        bool isGpioRemoved = generalIo.GetGpioDetectPin() == false;
        bool isGpioEventOccurred = nn::os::TryWaitSystemEvent(m_pGpioDetectRawEvent);
        if(isGpioRemoved || isGpioEventOccurred || generalIo.IsGpioDetectWakeEventActive())
        {
            NN_DETAIL_GC_LOG("removed cause: isGpioRemoved: %d, isGpioEventOCcurred: %d, isGpioDetectWakeEventActive: %d\n",
                isGpioRemoved, isGpioEventOccurred, generalIo.IsGpioDetectWakeEventActive());
            SetDetectStateAndSignalEvent(false);
            m_IsRemovedDuringSleep = true;
        }
        // カードの挿抜イベント（m_pGpioDetectRawEvent）はここではクリアせず残しておき、後段で適切にシグナルしてもらう
        // （ここで抜去処理をしているため後段でも別途同じ処理される可能性があるが、 抜 → 抜 は無視するので問題ない）
    }
    // 挿入の場合は安定待ちに突入

    // 一時停止復帰終了イベントのシグナル
    nn::os::SignalEvent(&m_ResumeCompleteEvent);
}

bool DeviceDetector::GetGpioDetectPin() NN_NOEXCEPT
{
    return m_IsInserted;
}

void DeviceDetector::ClearGpioDetectEvent() NN_NOEXCEPT
{
    nn::os::ClearEvent(&m_GpioDetectFilteredEvent);
}

void DeviceDetector::Pause() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    // 念のため確認
    if(m_IsPaused == true)
    {
        return;
    }
    m_IsPaused = true;
    nn::os::ClearEvent(&m_ResumeCompleteEvent);
    nn::os::SignalEvent(&m_PauseRequestEvent);
    nn::os::WaitEvent(&m_ResumeCompleteEvent);
}

bool DeviceDetector::Resume() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    // 念のため確認
    if(m_IsPaused == false)
    {
        return false;
    }
    m_IsPaused = false;
    nn::os::ClearEvent(&m_ResumeCompleteEvent);
    nn::os::SignalEvent(&m_ResumeRequestEvent);
    nn::os::WaitEvent(&m_ResumeCompleteEvent);

    return m_IsRemovedDuringSleep;
}

void DeviceDetector::ResetPauseState() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    // とりあえず起こしておけば問題ない
    this->Resume();
}

void DeviceDetector::ManualSignalGpioDetectEventForGcCore() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;

    // マニュアルでのシグナルは AsicHandler が自身を抜去状態に遷移させるために利用する
    // 本当に抜去があった場合は DeviceDetector スレッドが別に通知してくれるので転送中断に関しては問題ない
    nn::os::SignalEvent(&m_GpioDetectFilteredEvent);
    m_DeviceDetectorErrorInfo.removalCount++;
}

void DeviceDetector::SetDetectStateAndSignalEvent(bool isInserted) NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG();

    m_IsInserted = isInserted;
    m_DeviceDetectorErrorInfo.isInserted = m_IsInserted;
    nn::os::SignalEvent(&m_GpioDetectFilteredEvent);
    if(m_IsInserted)
    {
        m_DeviceDetectorErrorInfo.insertionCount++;
    }
    else
    {
        m_DeviceDetectorErrorInfo.removalCount++;
    }
    NN_DETAIL_GC_LOG("Card %s\n", m_IsInserted ? "Inserted!" : "Removed!");

    // カード抜去の際は登録されたコールバックを呼ぶ（sdmmc 転送を中断させたいため）
    if(m_IsInserted == false)
    {
        CallGcCallback(&m_CardRemovalEventCallback);
    }
}

void DeviceDetector::CheckDetectStateChangeAndSignalEvent() NN_NOEXCEPT
{
    bool isGpioInserted = GeneralIo::GetInstance().GetGpioDetectPin();
    // 挿 → 挿, 抜 → 抜 : 何もしない
    if(isGpioInserted != m_IsInserted)
    {
        // 挿 → 抜 : 「抜け」イベント
        if(m_IsInserted == true)
        {
            SetDetectStateAndSignalEvent(false);
        }
        // 抜 → 挿 : 「挿入」安定待ち（何もしない）
    }
}

void DeviceDetector::GetDeviceDetectorErrorInfo(DeviceDetectorErrorInfo* pOutDeviceDetectorErrorInfo) NN_NOEXCEPT
{
    *pOutDeviceDetectorErrorInfo = m_DeviceDetectorErrorInfo;
    // 本体電源投入時からのトータルカウントとする為、リセットはしない
}


} } }
