﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Log.h>

#include <nn/gc/detail/gc_DataIo.h>
#include <nn/gc/detail/gc_GcCrypto.h>
#include <nn/gc/detail/gc_StateMachine.h>

namespace nn { namespace gc {
namespace detail {

DataIo::DataIo() NN_NOEXCEPT
{
    m_MaxRetryCount = 0;
    m_IsDeviceVirtualAddressRegistered = false;
}

DataIo& DataIo::GetInstance() NN_NOEXCEPT
{
    static DataIo s_Instance;
    return s_Instance;
}

void DataIo::Initialize() NN_NOEXCEPT
{
    NN_DETAIL_GC_LOG("[GC] DataIo Initialization\n");

    // * SDMMC ライブラリの初期化
    nn::sdmmc::Initialize(m_GcPort);
}

void DataIo::RegisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    nn::sdmmc::RegisterDeviceVirtualAddress(m_GcPort, bufferAddress, bufferSize, bufferDeviceVirtualAddress);
    m_IsDeviceVirtualAddressRegistered = true;
}

nn::Result DataIo::Activate() NN_NOEXCEPT
{
    nn::Result result = nn::sdmmc::Activate(m_GcPort);
    if (result.IsFailure())
    {
        NN_DETAIL_GC_ERR_LOG("nn::sdmmc::Activate failed. Module:%d, Description:%d\n", result.GetModule(), result.GetDescription());
        return result;
    }
    NN_RESULT_SUCCESS;
}

void DataIo::Deactivate() NN_NOEXCEPT
{
    nn::sdmmc::Deactivate(m_GcPort);
}

void DataIo::UnregisterDeviceVirtualAddress(uintptr_t bufferAddress, size_t bufferSize, nn::dd::DeviceVirtualAddress bufferDeviceVirtualAddress) NN_NOEXCEPT
{
    if(m_IsDeviceVirtualAddressRegistered)
    {
        nn::sdmmc::UnregisterDeviceVirtualAddress(m_GcPort, bufferAddress, bufferSize, bufferDeviceVirtualAddress);
    }
}

nn::Result DataIo::Finalize() NN_NOEXCEPT
{
    nn::sdmmc::Deactivate(m_GcPort);
    nn::sdmmc::Finalize(m_GcPort);

    return nn::ResultSuccess();
}

nn::Result DataIo::SendCommandDataRead(char* outAlignedBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();
    NN_DETAIL_GC_IO_LOG("Send command data read: %zu byte(s)\n", bufferLength);
    if ( bufferLength % SectorSize != 0 )
    {
        NN_DETAIL_GC_ERR_LOG("Warning: read data rounded down from %zu to %zu\n", bufferLength, bufferLength - (bufferLength % SectorSize));
    }

    // セクタ番号はゲームカードの場合無視される
    result = nn::sdmmc::Read(outAlignedBuffer, bufferLength, m_GcPort, 0, (bufferLength / SectorSize));

    // Read の復号化は上のレイヤで行う

    return result;
}

nn::Result DataIo::SendCommandDataWrite(char* workAlignedDataBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_IO_LOG("Send command data write: %zu byte(s)\n", bufferLength);
        if ( bufferLength % SectorSize != 0 )
    {
        NN_DETAIL_GC_ERR_LOG("Warning: write data rounded down from %zu to %zu\n", bufferLength, bufferLength - (bufferLength % SectorSize));
    }

    StateMachine& gcState = StateMachine::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();

    // データを暗号化する
    if ( gcState.IsAsicSecure() )
    {
        // セキュアモードなら AES-CTR で暗号化
        gcCrypto.EncryptWithAesCtr(workAlignedDataBuffer, bufferLength, workAlignedDataBuffer, bufferLength);
    }
    else
    {
        if (gcState.IsCommonKeyReady())
        {
            // セキュアモードでなくて、共通鍵があるなら AES-CBC で暗号化
            gcCrypto.EncryptWithAesCbc(workAlignedDataBuffer, bufferLength, workAlignedDataBuffer, bufferLength);
        }
    }
    // セキュアモードでなくて、共通鍵もないならデータを横流しする（この場合 workDataBuffer は変更されない）

    // デバッグ出力
    // NN_DETAIL_GC_DEBUG_MEMDUMP(workAlignedDataBuffer, bufferLength, "Possibly encrypted data to send\n");

    // セクタ番号はゲームカードの場合無視される
    return nn::sdmmc::Write(m_GcPort, 0, (bufferLength / SectorSize), workAlignedDataBuffer, bufferLength);
}


nn::Result DataIo::GetDeviceStatus(BitPack32 *outStatus) NN_NOEXCEPT
{
    u32 outStatusValue;
    nn::Result result = nn::sdmmc::GetDeviceStatus(&outStatusValue, m_GcPort);
    outStatus->storage = outStatusValue;

    if ( result.IsFailure() )
    {
        // NOTE: カード抜けの際に必ずログ出力されてしまうので、ERR_LOG にしない
        NN_DETAIL_GC_LOG("Failed to get device status (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
    }
    return result;
}



void DataIo::PrintDeviceStatus() NN_NOEXCEPT
{
    BitPack32 response;
    this->GetDeviceStatus(&response);
    NN_DETAIL_GC_IO_LOG("[GC] Get device status : 0x%04x\n", response.storage);
}


nn::Result DataIo::SendOperationStart(char* workAlignedOperationBuffer, const size_t bufferLength) NN_NOEXCEPT
{
    NN_DETAIL_GC_SDK_REQUIRES(bufferLength == GcMmcCmd60DataSize, "Invalid buffer length for asic operation.");
    NN_DETAIL_GC_IO_LOG("Send operation start : 0x%02x\n", workAlignedOperationBuffer[0]);

    StateMachine& gcState = StateMachine::GetInstance();
    GcCrypto& gcCrypto = GcCrypto::GetInstance();

    // 必要ならCV値をつける
    if ( gcState.IsAsicSecure() )
    {
        gcCrypto.AddCommandVerificationValue(workAlignedOperationBuffer, bufferLength);
    }

    // セキュアモードなら AES-CTR で暗号化する
    if ( gcState.IsAsicSecure() )
    {
        gcCrypto.EncryptWithAesCtr(workAlignedOperationBuffer, bufferLength,
            workAlignedOperationBuffer, bufferLength);
    }
    // セキュアモードでないなら、オペレーション64byteの暗号化はされない

    // SDMMC コマンドの発行
    nn::Result result = SendSdmmcCommandWithRetry(nn::sdmmc::WriteGcAsicOperation, workAlignedOperationBuffer, bufferLength);

    return result;
}

nn::Result DataIo::FinishOperation() NN_NOEXCEPT
{
    NN_DETAIL_GC_IO_LOG("Finish operation\n");
    return SendSdmmcCommandWithRetry(nn::sdmmc::FinishGcAsicOperation);
}

nn::Result DataIo::AbortGcAsicOperation() NN_NOEXCEPT
{
    NN_DETAIL_GC_IO_LOG("Abort GcAsic Operation (CMD12)\n");
    return SendSdmmcCommandWithRetry(nn::sdmmc::AbortGcAsicOperation);
}

nn::Result DataIo::PutGcAsicToSleep() NN_NOEXCEPT
{
    NN_DETAIL_GC_IO_LOG("Go to sleep\n");
    return SendSdmmcCommandWithRetry(nn::sdmmc::SleepGcAsic);
}

nn::Result DataIo::AwakenGcAsic() NN_NOEXCEPT
{
    NN_DETAIL_GC_IO_LOG("Wake up\n");
    // スリープ復帰は FinishOperation
    return SendSdmmcCommandWithRetry(nn::sdmmc::FinishGcAsicOperation);
}

// 続けて FinishOperation を呼ぶ必要がある
nn::Result DataIo::UpdateKey() NN_NOEXCEPT
{
    NN_DETAIL_GC_IO_LOG("Update key\n");
    return SendSdmmcCommandWithRetry(nn::sdmmc::UpdateGcAsicKey);
}

void DataIo::SignalRemovedEvent() NN_NOEXCEPT
{
    nn::sdmmc::SignalGcRemovedEvent(m_GcPort);
}

void DataIo::ClearRemovedEvent() NN_NOEXCEPT
{
    nn::sdmmc::ClearGcRemovedEvent(m_GcPort);
}

void DataIo::PutSdmmcToSleep() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    nn::sdmmc::PutGcAsicToSleep(m_GcPort);
}

Result DataIo::AwakenSdmmc() NN_NOEXCEPT
{
    NN_DETAIL_GC_THREAD_DEBUG_LOG_LINE;
    return nn::sdmmc::AwakenGcAsic(m_GcPort);
}

nn::Result DataIo::SendSdmmcCommandWithRetry(sdmmcCommandFunctionPointer pFunction) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    for(int i=0; i<=m_MaxRetryCount; i++)
    {
        result = pFunction(m_GcPort);
        if ( result.IsSuccess() )
        {
            return nn::ResultSuccess();
        }
        NN_DETAIL_GC_LOG("Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        this->PrintDeviceStatus();
    }

    return result;
}

nn::Result DataIo::SendSdmmcCommandWithRetry(sdmmcCommandFunctionWithBufferPointer pFunction, const void* operationBuffer, const size_t operationBufferSize) NN_NOEXCEPT
{
    nn::Result result = nn::ResultSuccess();

    for(int i=0; i<=m_MaxRetryCount; i++)
    {
        result = pFunction(m_GcPort, operationBuffer, operationBufferSize);
        if ( result.IsSuccess() )
        {
            return nn::ResultSuccess();
        }
        NN_DETAIL_GC_LOG("Failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        this->PrintDeviceStatus();
    }

    return result;
}

void DataIo::SetTransferTimeout(uint32_t milliSeconds)
{
    nn::sdmmc::ChangeCheckTransferInterval(m_GcPort, milliSeconds);
}
void DataIo::SetDefaultTransferTimeout()
{
    nn::sdmmc::SetDefaultCheckTransferInterval(m_GcPort);
}
} } }
