﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_SdkLog.h>
#include <nn/gc/detail/gc_Asn1Parser.h>

namespace nn { namespace gc { namespace detail {

namespace
{
bool CheckLengthAndCopy(char* pOutBuffer, const size_t outBufferLength, const size_t copyLength, const size_t index, const char* pInData, const size_t pInDataLength) NN_NOEXCEPT
{
    if( copyLength > outBufferLength)
    {
        return false;
    }
    else if( index + copyLength > pInDataLength)
    {
        return false;
    }
    memset(pOutBuffer, 0, outBufferLength);
    memcpy(pOutBuffer + outBufferLength - copyLength, pInData + index, copyLength);
    return true;
}

}

/**
    @brief GC ASIC の証明書をパースします。
 */
bool Asn1Parser::ParseGcAsicCertificate (const char* pInData, const size_t length) NN_NOEXCEPT
{
    size_t index = 0;
    size_t nextIndex = 0;
    size_t valueIndex = 0;
    size_t valueLength = 0;
    m_IsParseFinished = false;
    // ブロックのオフセットを固定してパース
    for(size_t counter = 0; counter < LoopNum; counter++)
    {
        AnalyzeBlock(&nextIndex, &valueIndex, &valueLength, pInData, index, (counter == ExponentBlockNum));
        if(counter == TbsCertBlockNum)
        {
            m_TbsCertificateIndex     = index;
            m_TbsCertificateLength    = valueLength + (valueIndex - index);
            if(m_TbsCertificateLength > length || m_TbsCertificateIndex + m_TbsCertificateLength > length)
            {
                return false;
            }
        }
        else if(counter == SerialNumberBlockNum)
        {
            if(!CheckLengthAndCopy(m_SerialNumber, sizeof(m_SerialNumber), valueLength, valueIndex, pInData, length))
            {
                return false;
            }
        }
        else if(counter == ModulusBlockNum)
        {
            if(!CheckLengthAndCopy(m_Modulus, sizeof(m_Modulus), valueLength, valueIndex, pInData, length))
            {
                return false;
            }
        }
        else if(counter == ExponentBlockNum)
        {
            if(!CheckLengthAndCopy(m_Exponent, sizeof(m_Exponent), valueLength, valueIndex, pInData, length))
            {
                return false;
            }
        }
        else if(counter == SignatureAlgorithmBlockNum)
        {
            m_SignatureAlgorithmIndex  = valueIndex;
            m_SignatureAlgorithmLength = valueLength;
            if(m_SignatureAlgorithmLength > length || m_SignatureAlgorithmIndex + m_SignatureAlgorithmLength > length)
            {
                return false;
            }
        }
        else if(counter == SignatureBlockNum)
        {
            if((*(pInData + valueIndex) == 0x00) && (valueLength > 1))
            {
                valueIndex  += 1;
                valueLength -= 1;
            }
            m_SignatureIndex          = valueIndex;
            m_SignatureLength         = valueLength;
            if(m_SignatureLength > length || m_SignatureIndex + m_SignatureLength > length)
            {
                return false;
            }
        }
        index = nextIndex;
        if(index > length)
        {
            return false;
        }
    }
    m_IsParseFinished = true;
    return true;
}

void Asn1Parser::AnalyzeBlock(size_t* pOutNextIndex, size_t* pOutValueIndex, size_t* pOutValueLength, const char* pInData, const size_t index, const bool isSkipEnabled) NN_NOEXCEPT
{
    size_t tag = *(pInData + index);
    size_t currentIndex = index + 1;
    if(*(pInData + currentIndex) == Asn1LongLengthBitFlag) // この bit だけが立っていることはない
    {
        *pOutNextIndex = currentIndex + 1;
        *pOutValueIndex   = 0;
        *pOutValueLength = 0;
        return;
    }
    size_t length = *(pInData + currentIndex) & (Asn1LongLengthBitFlag - 1);
    // length の表記が 1 Byte で収まらない場合
    if(*(pInData + currentIndex++) & Asn1LongLengthBitFlag)
    {
        // length の表記が 2 Byte より大きいことは有り得ない
        if(length > 2)
        {
            *pOutNextIndex = currentIndex + 1;
            *pOutValueIndex   = 0;
            *pOutValueLength = 0;
            return;
        }
        size_t valueLength = 0;
        for(size_t i = 0; i < length; i++)
        {
            valueLength <<= 8L;
            valueLength |= *(pInData + currentIndex + i);
        }
        *pOutValueIndex   = currentIndex + length;
        *pOutValueLength  = valueLength;
        *pOutNextIndex    = *pOutValueIndex;
        // 構造化されている場合
        if((tag & Asn1TagConstructed) || (tag == Asn1TagBitString))
        {
            if((tag == Asn1TagBitString) && (*(pInData + currentIndex + length + 1) == (Asn1TagSequence | Asn1TagConstructed)) && !isSkipEnabled)
            {
                *pOutNextIndex += 1;
            }
        }
        // 構造化されていない場合
        else
        {
            *pOutNextIndex    += *pOutValueLength;
        }
    }
    else // length の表記が 1 Byte である場合
    {
        *pOutValueIndex   = currentIndex;
        *pOutValueLength  = length;
        *pOutNextIndex    = *pOutValueIndex;
        if(!(tag & Asn1TagConstructed))
        {
            *pOutNextIndex += *pOutValueLength;
        }
    }
    if(tag == Asn1TagInteger)
    {
        // Asn1TagInteger の場合のみの特殊対応(先頭が 0x00 かつ length が 1より大きければ読み飛ばす)
        if((*(pInData + *pOutValueIndex) == 0x00) && (*pOutValueLength > 1))
        {
            *pOutValueIndex  += 1;
            *pOutValueLength -= 1;
        }
    }
    return;
}

} } }

