﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_ScopeExit.h>

#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Log.h>

#include <nn/gc/detail/gc_AsicRegister.h>
#include <nn/gc/detail/gc_AsicOperation.h>

namespace nn { namespace gc {
namespace detail {

// レジスタの読み出しは AsicRegister クラスのみが行うため、ここでビット定義
// レジスタのビットフィールド定義（C++ の bit field はプラットフォーム依存の為、使用不可）
struct AsicRegisterMap_AsicStatus
{
    typedef BitPack32::Field<0, 8, u8> cdst;
    typedef BitPack32::Field<8, 2, u8> crcrslt;
    typedef BitPack32::Field<12, 1, bool> toerr;
    typedef BitPack32::Field<15, 1, bool> ltc;
    typedef BitPack32::Field<16, 8, u8> asicst;
    typedef BitPack32::Field<24, 2, u8> asicmode;
};
struct AsicRegisterMap_AccessControl1
{
    typedef BitPack32::Field<0, 3, u8> rdclk;
    typedef BitPack32::Field<4, 3, u8> wrclk;
    typedef BitPack32::Field<16, 1, u8> toset;
    typedef BitPack32::Field<20, 4, u8> to;
    typedef BitPack32::Field<31, 1, u8> clksel;
};
struct AsicRegisterMap_ReadWait1Time
{
    typedef BitPack32::Field<0, 26, u32> wait1;
};
struct AsicRegisterMap_ReadWait2Time
{
    typedef BitPack32::Field<0, 26, u32> wait2;
};
struct AsicRegisterMap_WriteWait1Time
{
    typedef BitPack32::Field<0, 26, u32> wait1;
};
struct AsicRegisterMap_WriteWait2Time
{
    typedef BitPack32::Field<0, 26, u32> wait2;
};
struct AsicRegisterMap_PageRemainder
{
    typedef BitPack32::Field<0, 32, u32> pr;
};
struct AsicRegisterMap_LatencyTime
{
    typedef BitPack32::Field<0, 9, u16> ltct;
};
struct AsicRegisterMap_LimitedArea
{
    typedef BitPack32::Field<0, 32, u32> lra;
};
struct AsicRegisterMap_CupVersion
{
    typedef BitPack32::Field<0, 32, u32> cver;
};
struct AsicRegisterMap_AsicVersion
{
    typedef BitPack32::Field<0, 32, u32> ver;
};
struct AsicRegisterMap_Standby2Control
{
    typedef BitPack32::Field<0, 1, bool> stb;
};

AsicRegister::AsicRegister() NN_NOEXCEPT : registerArray(m_RegisterArray), m_FirstRegisterAddress(&(m_RegisterMap.asicStatus))
{
    std::memset(&m_AsicRegisterErrorInfo, 0, sizeof(m_AsicRegisterErrorInfo));
    ResetRegister();
}

AsicRegister& AsicRegister::GetInstance() NN_NOEXCEPT
{
    static AsicRegister s_Instance;
    return s_Instance;
}

nn::Result AsicRegister::WriteRegister(const BitPack32 *registerValueAddress) NN_NOEXCEPT
{
    u32 address = ConvertRegisterValueAddressToRegisterAddress(registerValueAddress);
    u32 value = registerValueAddress->storage;
    return AsicOperation::GetInstance().WriteRegister(address, value);
}

u32 AsicRegister::ConvertRegisterValueAddressToRegisterAddress(const BitPack32 *registerValueAddress) NN_NOEXCEPT
{
    NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES( ( m_FirstRegisterAddress <= registerValueAddress ) &&
    ( static_cast<u32>(registerValueAddress - m_FirstRegisterAddress) < (sizeof(m_RegisterMap) / sizeof(BitPack32)) ) );
    return static_cast<u32>(registerValueAddress - m_FirstRegisterAddress) * sizeof(BitPack32);
}

// *** public
nn::Result AsicRegister::ClearRegisterErrorStatus() NN_NOEXCEPT
{
    // レジスタのクリア
    m_RegisterMap.asicStatus.Clear();
    return WriteRegister(&m_RegisterMap.asicStatus);
}

Result AsicRegister::ReadAndCheckRegisterErrorStatus(nn::Result* pOutRegisterResult) NN_NOEXCEPT
{
    // 読み込み
    NN_RESULT_DO(AsicOperation::GetInstance().ReadRegister());

    *pOutRegisterResult = ReadAndCheckRegisterErrorStatusImpl();

    NN_RESULT_SUCCESS;
}

Result AsicRegister::SetLatencyCheckValue(const u16 value) NN_NOEXCEPT
{
    m_RegisterMap.latencyTime.Set<AsicRegisterMap_LatencyTime::ltct>(value);
    return WriteRegister(&m_RegisterMap.latencyTime);
}


// *** private
void AsicRegister::ResetRegister() NN_NOEXCEPT
{
    std::memset(m_RegisterArray, 0, sizeof(m_RegisterArray));

    // 初期値のあるものに代入
    m_RegisterMap.accessControl1.storage = 0x11;
    m_RegisterMap.latencyTime.storage = 0x64;
    m_RegisterMap.limitedArea.storage = 0x01;
}

Result AsicRegister::ReadAndCheckRegisterErrorStatusImpl() NN_NOEXCEPT
{
    nn::Result result = CheckAsicErrorStatus();

    // ResultAsicStatusError（意味：エラー内容が特定できなかった）の場合のみ後段でチェックを行う
    if(result.IsFailure() && nn::fs::ResultGameCardAsicStatusError::Includes(result) == false)
    {
        return result;
    }
    NN_RESULT_DO(CheckTimoutError());
    NN_RESULT_DO(CheckCardStatus());
    NN_RESULT_DO(CheckCrcResult());
    return result;
}

Result AsicRegister::CheckAsicErrorStatus()
{
    const u8 asicst = m_RegisterMap.asicStatus.Get<AsicRegisterMap_AsicStatus::asicst>();
    const u8 sessionErrorMask = (1 << 6) - 1; // bit0 - bit5 いずれかが立っている場合
    const u8 updateKeyErrorMask = 1 << 6;     // bit6

    if( (asicst & sessionErrorMask) != 0 )
    {
        NN_DETAIL_GC_ERR_LOG("Invalid card (%d)\n", (asicst & sessionErrorMask));
        return nn::fs::ResultGameCardWrongCard();
    }
    else if( (asicst & updateKeyErrorMask) != 0 )
    {
        NN_DETAIL_GC_ERR_LOG("Session failed to update\n");
        return nn::fs::ResultGameCardUpdateKeyFailure();
    }
    else if( asicst > 0 ) // bit7 が 1 のとき
    {
        // その他のエラー
        NN_DETAIL_GC_WARNING_LOG("Card failure (%d); detail follows...\n", asicst);
        return nn::fs::ResultGameCardAsicStatusError();
    }
    NN_RESULT_SUCCESS;
}

Result AsicRegister::CheckTimoutError()
{
    const bool toerr = m_RegisterMap.asicStatus.Get<AsicRegisterMap_AsicStatus::toerr>();
    if(toerr)
    {
        NN_DETAIL_GC_WARNING_LOG("Card access timeout\n");
        m_AsicRegisterErrorInfo.gameCardTimeoutErrorNum++;
        return nn::fs::ResultGameCardCardAccessTimeout();
    }
    NN_RESULT_SUCCESS;
}

Result AsicRegister::CheckCardStatus()
{
    const u8 cdst = m_RegisterMap.asicStatus.Get<AsicRegisterMap_AsicStatus::cdst>();
    const u8 crcErrorMask   = 1 << 0;
    const u8 writeErrorMask = 1 << 1;
    const u8 fatalErrorMask = 1 << 2;
    const u8 refreshReqMask = 1 << 3;
    const u8 reservedMask   = ((1 << 4) - 1) << 4;// 上位 4 ビット

    // Fatal/WriteError/reserve どれかが 1 の時に ASIC reboot のパスへ
    // TODO : 今後ユーザーの手元で書き込み可能なカードが出てきたらライトエラーのチェックは外す必要あり
    if( (cdst & fatalErrorMask) != 0 || (cdst & writeErrorMask) != 0 || (cdst & reservedMask) != 0)
    {
        // TODO: 受け手で強制リブート
        NN_DETAIL_GC_ERR_LOG("Card fatal error\n");
        return nn::fs::ResultGameCardCardNeedRetryAfterAsicReinitialize();
    }

    // リトライ要否判定
    bool needRetryFlag = (cdst & (crcErrorMask | writeErrorMask)) != 0;

    // refresh 要求が来ているなら先に refresh しないと次のコマンドを受け付けてくれない
    if( (cdst & refreshReqMask) != 0 )
    {
        if(needRetryFlag)
        {
            // GC CRC エラー発生場所
            NN_DETAIL_GC_WARNING_LOG("CRC error (%d) and Refresh request\n", needRetryFlag);
            m_AsicRegisterErrorInfo.gameCardCrcErrorNum++;
            m_AsicRegisterErrorInfo.refreshNum++;
            return nn::fs::ResultGameCardNeedRefreshAndCardNeedRetry();
        }
        // リトライ要求発生場所
        NN_DETAIL_GC_WARNING_LOG("Refresh request\n");
        m_AsicRegisterErrorInfo.refreshNum++;
        return nn::fs::ResultGameCardNeedRefresh();
    }
    else if( needRetryFlag )
    {
        // ASIC CRC エラー発生場所
        m_AsicRegisterErrorInfo.gameCardCrcErrorNum++;
        NN_DETAIL_GC_WARNING_LOG("Card CRC error (%d)\n", needRetryFlag);
        return nn::fs::ResultGameCardCardNeedRetry();
    }
    NN_RESULT_SUCCESS;
}

Result AsicRegister::CheckCrcResult()
{
    const u8 crcrslt = m_RegisterMap.asicStatus.Get<AsicRegisterMap_AsicStatus::crcrslt>();
    // crcrslt = 2'b00 : エラーなし
    // crcrslt = 2'b01 : コマンド CRC エラー発生 (CardStatus の CRC エラーが立っているはず)
    // crcrslt = 2'b10 : ページデータ CRC エラー発生
    // crcrslt = 2'b11 : 未定義

    switch (crcrslt)
    {
    case 0:
        {
            NN_RESULT_SUCCESS;
        }
    case 1:
        {
            m_AsicRegisterErrorInfo.gameCardCrcErrorNum++;
            NN_DETAIL_GC_WARNING_LOG("Card CRC error\n");
            return nn::fs::ResultGameCardCardNeedRetry();
        }
    case 2:
        {
            // ASIC CRC エラー発生場所
            m_AsicRegisterErrorInfo.asicCrcErrorNum++;
            NN_DETAIL_GC_WARNING_LOG("ASIC CRC error\n", crcrslt);
            return nn::fs::ResultGameCardCardNeedRetry();
        }
    default:
        {
            NN_DETAIL_GC_ERR_LOG("Card fatal error in CheckCrcResult\n");
            return nn::fs::ResultGameCardCardNeedRetryAfterAsicReinitialize();
        }
    }
}

void AsicRegister::GetRegisterErrorInfo(AsicRegisterErrorInfo* pOutAsicRegisterErrorInfo) NN_NOEXCEPT
{
    *pOutAsicRegisterErrorInfo = m_AsicRegisterErrorInfo;
}

void AsicRegister::ClearErrorInfo() NN_NOEXCEPT
{
    memset(&m_AsicRegisterErrorInfo, 0, sizeof(m_AsicRegisterErrorInfo));
}

} } }
