﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/save/fs_UnionStorage.h>

namespace nn { namespace fssystem { namespace save {

    namespace {
        const int64_t Sentinel = -1; // ログの終わりを示すオフセット値
        const int64_t OffsetLogBegin = sizeof(int64_t); // 先頭ログのオフセット

        // ログデータのサイズを取得する
        int64_t GetLogSize(int64_t sizeBlock) NN_NOEXCEPT
        {
            return sizeof(int64_t) + sizeBlock;
        }

        // 実データのオフセットを取得する
        int64_t GetDataOffset(int64_t offsetLog) NN_NOEXCEPT
        {
            return offsetLog + sizeof(int64_t);
        }

        // ログ末尾のオフセットを取得する
        int64_t GetLogTailOffset(int64_t sizeBlock, int countLogs) NN_NOEXCEPT
        {
            return OffsetLogBegin + countLogs * GetLogSize(sizeBlock);
        }
    }

    Result UnionStorage::Format(
               fs::SubStorage storageLog,
               int64_t sizeBlock
           ) NN_NOEXCEPT
    {
        // 先頭にブロックサイズ (int64_t)
        // 次に 0 個以上のログデータ
        // ログデータは、先頭にオフセット (int64_t)、その次に実データ (char[sizeBlock])
        // 最後に Sentinel (int64_t)

        NN_SDK_REQUIRES_GREATER(sizeBlock, 1);
        NN_SDK_REQUIRES(nn::util::ispow2(sizeBlock));

        const int64_t buffer[2] = { sizeBlock, Sentinel };

        return storageLog.Write(0, buffer, sizeof(buffer));
    }

    Result UnionStorage::Initialize(
               fs::SubStorage storageOriginal,
               fs::SubStorage storageLog,
               int64_t sizeBlock
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_Buffer == nullptr);

        // ログを読み込み、壊れていないか検証する
        NN_RESULT_DO(storageLog.Read(0, &m_SizeBlock, sizeof(int64_t)));

        if( sizeBlock <= 1 || (sizeBlock & (sizeBlock - 1)) != 0 || sizeBlock != m_SizeBlock )
        {
            return nn::fs::ResultInvalidLogBlockSize();
        }

        for( auto offset = OffsetLogBegin; ; offset += GetLogSize(m_SizeBlock) )
        {
            int64_t offsetOriginal = 0;

            NN_RESULT_DO(storageLog.Read(offset, &offsetOriginal, sizeof(int64_t)));

            if( offsetOriginal == Sentinel )
            {
                break;
            }
            else if( offsetOriginal % m_SizeBlock != 0 )
            {
                return nn::fs::ResultInvalidLogOffset();
            }

            ++m_CountLogs;
        }

        // 検証に成功したら初期化する
        m_StorageOriginal = storageOriginal;
        m_StorageLog = storageLog;
        m_Buffer = nn::fs::detail::MakeUnique<char[]>(static_cast<size_t>(m_SizeBlock));
        NN_RESULT_THROW_UNLESS(m_Buffer != nullptr, nn::fs::ResultAllocationMemoryFailedMakeUnique());
        NN_RESULT_SUCCESS;
    }

    Result UnionStorage::Freeze() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);
        const auto offsetLogTail = GetLogTailOffset(m_SizeBlock, m_CountLogs);

        NN_RESULT_DO(m_StorageLog.Write(offsetLogTail, &Sentinel, sizeof(int64_t)));
        NN_RESULT_DO(m_StorageLog.Flush());
        NN_RESULT_SUCCESS;
    }

    Result UnionStorage::Commit() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        const auto offsetLogTail = GetLogTailOffset(m_SizeBlock, m_CountLogs);
        const auto logSize = GetLogSize(m_SizeBlock);

        for( auto offsetLog = OffsetLogBegin; offsetLog < offsetLogTail; offsetLog += logSize )
        {
            int64_t offsetOriginal = 0;

            NN_RESULT_DO(m_StorageLog.Read(offsetLog, &offsetOriginal, sizeof(int64_t)));

            if( offsetOriginal != Sentinel )
            {
                const auto size = static_cast<size_t>(m_SizeBlock);

                NN_RESULT_DO(m_StorageLog.Read(GetDataOffset(offsetLog), m_Buffer.get(), size));
                NN_RESULT_DO(m_StorageOriginal.Write(offsetOriginal, m_Buffer.get(), size));
            }
            else
            {
                return nn::fs::ResultUnexpectedEndOfLog();
            }
        }

        return m_StorageOriginal.Flush();
    }

    Result UnionStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

        size_t sizeReadSum = 0;
        auto offsetOriginal = offset & ~(m_SizeBlock - 1);
        auto offsetBlock = offset - offsetOriginal;

        while( sizeReadSum < size )
        {
            auto found = false;
            int64_t offsetLog = 0;
            const auto sizeReadBlock = static_cast<size_t>(m_SizeBlock - offsetBlock);
            const auto sizeRemain = size - sizeReadSum;
            const auto sizeRead = std::min(sizeReadBlock, sizeRemain);
            const auto bufferRead = reinterpret_cast<char*>(buffer) + sizeReadSum;

            NN_RESULT_DO(FindLog(&found, &offsetLog, offsetOriginal));

            if( found )
            {
                // ログに書かれているならそこから読み込む
                const auto offsetRead = GetDataOffset(offsetLog) + offsetBlock;

                NN_RESULT_DO(m_StorageLog.Read(offsetRead, bufferRead, sizeRead));
            }
            else
            {
                // ログに書かれていないならオリジナルを読み込む
                const auto offsetRead = offsetOriginal + offsetBlock;

                NN_RESULT_DO(m_StorageOriginal.Read(offsetRead, bufferRead, sizeRead));
            }

            sizeReadSum += sizeRead;
            offsetOriginal += m_SizeBlock;
            offsetBlock = 0;
        }

        NN_RESULT_SUCCESS;
    }

    Result UnionStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

        std::lock_guard<nn::os::Mutex> lock(m_Mutex);

        size_t sizeWriteSum = 0;
        auto offsetOriginal = offset & ~(m_SizeBlock - 1);
        auto offsetBlock = offset - offsetOriginal;

        while( sizeWriteSum < size )
        {
            NN_SDK_ASSERT_NOT_EQUAL(Sentinel, offsetOriginal);

            auto found = false;
            int64_t offsetLog = 0;
            const auto sizeWriteBlock = static_cast<size_t>(m_SizeBlock - offsetBlock);
            const auto sizeRemain = size - sizeWriteSum;
            const auto sizeWrite = std::min(sizeWriteBlock, sizeRemain);
            const auto bufferWrite = reinterpret_cast<const char*>(buffer) + sizeWriteSum;

            NN_RESULT_DO(FindLog(&found, &offsetLog, offsetOriginal));

            if( found )
            {
                // ログに書かれているなら上書きする
                const auto offsetWrite = GetDataOffset(offsetLog) + offsetBlock;

                NN_RESULT_DO(m_StorageLog.Write(offsetWrite, bufferWrite, sizeWrite));
            }
            else
            {
                NN_RESULT_DO(m_StorageLog.Write(offsetLog, &offsetOriginal, sizeof(int64_t)));

                if( sizeWrite == static_cast<size_t>(m_SizeBlock) )
                {
                    // ブロック全体をそのまま書き込む
                    const auto offsetWrite = GetDataOffset(offsetLog) + offsetBlock;

                    NN_RESULT_DO(m_StorageLog.Write(offsetWrite, bufferWrite, sizeWrite));
                }
                else
                {
                    // ブロックの一部を書き込むなら、一度読み込んだデータを上書きしてから書き込む
                    const auto offsetWrite = GetDataOffset(offsetLog);
                    const auto bufferCopy = m_Buffer.get() + offsetBlock;
                    const auto sizeBlock = static_cast<size_t>(m_SizeBlock);

                    NN_RESULT_DO(m_StorageOriginal.Read(
                        offsetOriginal,
                        m_Buffer.get(),
                        sizeBlock
                    ));
                    std::memcpy(bufferCopy, bufferWrite, sizeWrite);
                    NN_RESULT_DO(m_StorageLog.Write(offsetWrite, m_Buffer.get(), sizeBlock));
                }

                ++m_CountLogs;
            }

            sizeWriteSum += sizeWrite;
            offsetOriginal += m_SizeBlock;
            offsetBlock = 0;
        }

        NN_RESULT_SUCCESS;
    }

    Result UnionStorage::Flush() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

        NN_RESULT_DO(m_StorageOriginal.Flush());
        NN_RESULT_DO(m_StorageLog.Flush());
        NN_RESULT_SUCCESS;
    }

    Result UnionStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);

        return m_StorageOriginal.GetSize(outValue);
    }

    Result UnionStorage::OperateRange(
               void* outBuffer,
               size_t outBufferSize,
               fs::OperationId operationId,
               int64_t offset,
               int64_t size,
               const void* inBuffer,
               size_t inBufferSize
           ) NN_NOEXCEPT
    {
        for( auto offsetOriginal = offset & ~(m_SizeBlock - 1);
            offsetOriginal < offset + size;
            offsetOriginal += m_SizeBlock )
        {
            auto found = false;
            int64_t offsetLog = 0;
            NN_RESULT_DO(FindLog(&found, &offsetLog, offsetOriginal));
            if( found )
            {
                NN_RESULT_DO(m_StorageLog.OperateRange(
                    outBuffer,
                    outBufferSize,
                    operationId,
                    offsetLog,
                    m_SizeBlock,
                    inBuffer,
                    inBufferSize));
            }
        }
        return m_StorageOriginal.OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

    Result UnionStorage::FindLog(
               bool* outFound,
               int64_t* outOffsetLog,
               int64_t offsetOriginal
           ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_Buffer);
        NN_SDK_REQUIRES_NOT_NULL(outFound);
        NN_SDK_REQUIRES_NOT_NULL(outOffsetLog);

        *outOffsetLog = GetLogTailOffset(m_SizeBlock, m_CountLogs);
        *outFound = false;

        const auto offsetLogTail = GetLogTailOffset(m_SizeBlock, m_CountLogs);
        const auto sizeLog = GetLogSize(m_SizeBlock);

        for( auto offsetLog = OffsetLogBegin; offsetLog < offsetLogTail; offsetLog += sizeLog )
        {
            int64_t offset = 0;

            NN_RESULT_DO(m_StorageLog.Read(offsetLog, &offset, sizeof(int64_t)));

            if( offset == Sentinel )
            {
                return nn::fs::ResultLogNotFound();
            }
            else if( offset == offsetOriginal )
            {
                *outOffsetLog = offsetLog;
                *outFound = true;
                break;
            }
        }

        NN_RESULT_SUCCESS;
    }

}}}
