﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/util/util_IntUtil.h>
#include <nn/util/util_BinTypes.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_Assert.h>
#include <nn/fssystem/save/fs_SavePath.h>
#include <nn/fssystem/save/fs_IInternalStorageFileSystemVisitor.h>
#include <nn/fssystem/save/fs_IntegrityVerificationStorage.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.impl.h>
#include <nn/fssystem/dbm/fs_FileSystemTemplate.impl.h>
#include <nn/fssystem/dbm/fs_FileObjectTemplate.impl.h>
#include <nn/fssystem/dbm/fs_DirectoryObjectTemplate.impl.h>

namespace nn { namespace fssystem { namespace save {

namespace
{
    //!< セーブデータアーカイブの初期化済みコード 'SAVE'
    static const uint32_t MagicCode = NN_UTIL_CREATE_SIGNATURE_4('S','A','V','E');

    //!< セーブデータアーカイブのバージョン
    static const uint32_t VersionCode = 0x00060000;
    static const uint32_t VersionCodeMask = 0xFFFF0000;
}

/**
* @brief        セーブデータ領域をフォーマットします。
*
* @param[in]    storageControlArea  管理領域として使用するストレージ
* @param[in]    storageMeta         メタデータ領域として使用するストレージ
* @param[in]    storageData         実データ領域として使用するストレージ
* @param[in]    sizeBlock           ブロックサイズ
* @param[in]    pBufferManager      バッファマネージャ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::Format(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageMeta,
           fs::SubStorage storageData,
           uint32_t sizeBlock,
           fssystem::IBufferManager* pBufferManager
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(sizeBlock != 0);

    int64_t sizeData = 0;
    int64_t sizeMeta = 0;
    int64_t sizeControlArea = 0;
    NN_RESULT_DO(storageControlArea.GetSize(&sizeControlArea));
    NN_RESULT_DO(storageMeta.GetSize(&sizeMeta));
    NN_RESULT_DO(storageData.GetSize(&sizeData));

    NN_SDK_REQUIRES_GREATER_EQUAL(sizeControlArea, static_cast<int64_t>(sizeof(FileSystemHeader)));

    // メタデータ領域の先頭にアーカイブヘッダを作成します。
    FileSystemHeader header;
    std::memset(&header, 0, sizeof(header));

    // データサイズから初期化パラメーターを決定します。
    header.magic = MagicCode;
    header.version = VersionCode;
    header.countTotalBlocks = sizeData / sizeBlock;
    header.sizeBlocks = sizeBlock;

    // データ領域が予約領域サイズ以上かチェックします。
    if( header.countTotalBlocks < 2 )
    {
        // 容量不足のためファイルシステムを構築できません。
        return nn::fs::ResultOutOfResource();
    }

    // メタデータ領域のサイズをチェックします。
    int64_t totalMetaSize = QueryAllocationTableStorageSize(
                                static_cast<uint32_t>(header.countTotalBlocks)
                            );
    if( totalMetaSize > sizeMeta )
    {
        // 容量不足のためファイルシステムを構築できません。
        return nn::fs::ResultOutOfResource();
    }

    // エラー発生時の途中抜けを考慮し、マジックコードをつぶしておきます。
    uint32_t dmy = 0;
    NN_RESULT_DO(storageControlArea.Write(0, &dmy, sizeof(dmy)));
    NN_RESULT_DO(storageControlArea.Flush());

    // dbm レイヤーをマウントし、フォーマットします。
    size_t offsetControlArea = offsetof(FileSystemHeader, fileSystemControlArea);
    NN_RESULT_DO(
        FormatDbmLayer(
            fs::SubStorage(
                &storageControlArea,
                offsetControlArea,
                sizeControlArea - offsetControlArea
            ),
            storageMeta,
            storageData,
            sizeBlock,
            static_cast<uint32_t>(header.countTotalBlocks),
            pBufferManager
        )
    );

    // 正しくフォーマットできたので、ヘッダの先頭部分を書き込みます。
    NN_RESULT_DO(
        storageControlArea.Write(
            0,
            &header,
            offsetControlArea
        )
    );
    NN_RESULT_DO(storageControlArea.Flush());

    NN_RESULT_SUCCESS;
}

/**
* @brief        管理領域を拡張します。
*
* @param[in]    storageControlArea  管理領域として使用するストレージ
* @param[in]    sizeDataNew         新しい実データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::ExpandControlArea(
            fs::SubStorage storageControlArea,
            int64_t sizeDataNew
        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(sizeDataNew, 0);

    // データが設定されているかチェックします
    FileSystemHeader header = {};
    NN_RESULT_DO(storageControlArea.Read(0, &header, sizeof(header)));

    if( header.magic != MagicCode )
    {
        // フォーマットされていません。
        return nn::fs::ResultIncorrectSaveDataFileSystemMagicCode();
    }
    if( (header.version & VersionCodeMask) != (VersionCode & VersionCodeMask) )
    {
        // データのバージョン互換性エラーです。
        return nn::fs::ResultUnsupportedVersion();
    }

    int64_t newBlockCount = sizeDataNew / header.sizeBlocks;

    if( !nn::util::IsIntValueRepresentable<uint32_t>(newBlockCount) )
    {
        // sizeDataNew が大きすぎる
        return nn::fs::ResultInvalidSize();
    }

    if( header.countTotalBlocks != newBlockCount )
    {
        if( newBlockCount < header.countTotalBlocks )
        {
            // sizeDataNew が小さすぎる
            return nn::fs::ResultInvalidSize();
        }

        const auto offsetControlArea = offsetof(FileSystemHeader, fileSystemControlArea);

        int64_t sizeControlArea = 0;
        NN_RESULT_DO(storageControlArea.GetSize(&sizeControlArea));

        FileSystemControlArea controlArea;
        ScopedFinalizer<FileSystemControlArea> finalizeControlArea(&controlArea);
        controlArea.Initialize(
            fs::SubStorage(
                &storageControlArea,
                offsetControlArea,
                sizeControlArea - offsetControlArea
            )
        );

        return controlArea.ExpandAllocationTableInfo(static_cast<uint32_t>(newBlockCount));
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        可変長メタデータ領域を拡張します。
*
* @param[in]    storageMeta     メタデータ領域として使用するストレージ
* @param[in]    sizeBlock       ブロックサイズ
* @param[in]    sizeDataOld     元々の実データ領域のサイズ
* @param[in]    sizeDataNew     新しい実データ領域のサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::ExpandMeta(
            fs::SubStorage storageMeta,
            uint32_t sizeBlock,
            int64_t sizeDataOld,
            int64_t sizeDataNew
        ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(sizeBlock, static_cast<uint32_t>(0));
    NN_SDK_REQUIRES_GREATER_EQUAL(sizeDataOld, 0);
    NN_SDK_REQUIRES_GREATER_EQUAL(sizeDataNew, sizeDataOld);
    NN_SDK_REQUIRES(nn::util::IsIntValueRepresentable<uint32_t>(sizeDataOld / sizeBlock));
    NN_SDK_REQUIRES(nn::util::IsIntValueRepresentable<uint32_t>(sizeDataNew / sizeBlock));

    const uint32_t oldBlockCount = static_cast<uint32_t>(sizeDataOld / sizeBlock);
    const uint32_t newBlockCount = static_cast<uint32_t>(sizeDataNew / sizeBlock);

    if( oldBlockCount < newBlockCount )
    {
        return AllocationTable::Expand(storageMeta, oldBlockCount, newBlockCount);
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        コンストラクタ
*/
SaveDataFileSystemCore::SaveDataFileSystemCore() NN_NOEXCEPT
    : m_OffsetBody(0),
      m_CountBody(0),
      m_SizeBlock(0)
{
}

/**
* @brief        デストラクタ
*/
SaveDataFileSystemCore::~SaveDataFileSystemCore() NN_NOEXCEPT
{
}

/**
* @brief        セーブデータアーカイブをマウントします。
*
* @param[in]    storageControlArea  管理領域として使用するストレージ
* @param[in]    storageMeta         メタデータ領域として使用するストレージ
* @param[in]    storageData         実データ領域として使用するストレージ
* @param[in]    pBufferManager      バッファマネージャ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::Initialize(
           fs::SubStorage storageControlArea,
           fs::SubStorage storageMeta,
           fs::SubStorage storageData,
           fssystem::IBufferManager* pBufferManager
       ) NN_NOEXCEPT
{
    int64_t sizeControlArea = 0;
    NN_RESULT_DO(storageControlArea.GetSize(&sizeControlArea));
    if( sizeControlArea < static_cast<int64_t>(sizeof(FileSystemHeader)) )
    {
        return nn::fs::ResultInvalidSize();
    }

    int64_t sizeData = 0;
    NN_RESULT_DO(storageData.GetSize(&sizeData));

    // 仮のサイズで初期化します。
    m_StorageControlArea = storageControlArea;
    m_StorageMeta = storageMeta;

    // 初期化チェックします。
    FileSystemHeader arch;
    std::memset(&arch, 0xff, sizeof(FileSystemHeader));
    NN_RESULT_DO(m_StorageControlArea.Read(0, &arch, sizeof(arch)));

    if( arch.magic != MagicCode )
    {
        // フォーマットされていません。
        return nn::fs::ResultIncorrectSaveDataFileSystemMagicCode();
    }
    if( (arch.version & VersionCodeMask) != (VersionCode & VersionCodeMask) )
    {
        // データのバージョン互換性エラーです。
        return nn::fs::ResultUnsupportedVersion();
    }

    // 実データ領域をマウントします。
    m_StorageData = storageData;

    size_t offsetControlArea = offsetof(FileSystemHeader, fileSystemControlArea);
    m_ControlArea.Initialize(
        fs::SubStorage(
            &m_StorageControlArea,
            offsetControlArea,
            sizeof(FileSystemControlArea::ControlArea)
        )
    );
    ScopedFinalizer<FileSystemControlArea> finalizeControlArea(&m_ControlArea);

    // 管理領域から各データを取得します。
    int64_t sizeBlock;
    NN_RESULT_DO(m_ControlArea.ReadBlockSize(&sizeBlock));

    uint32_t countAllocationTable;
    int64_t offsetAllocationTable;
    NN_RESULT_DO(m_ControlArea.ReadAllocationTableInfo(&offsetAllocationTable, &countAllocationTable));
    NN_UNUSED(offsetAllocationTable);

    int64_t offsetBody;
    uint32_t countAllocBlocksBody;
    NN_RESULT_DO(m_ControlArea.ReadDataBodyInfo(&offsetBody, &countAllocBlocksBody));

    uint32_t indexDirectoryEntry;
    NN_RESULT_DO(m_ControlArea.ReadDirectoryEntryInfo(&indexDirectoryEntry));

    uint32_t indexFileEntry;
    NN_RESULT_DO(m_ControlArea.ReadFileEntryInfo(&indexFileEntry));

    // 実データ領域のサイズを取得します。
    int64_t sizeBody = static_cast<int64_t>(sizeBlock) * countAllocBlocksBody;

    ScopedFinalizer<AllocationTable> finalizeAllocationTable(&m_AllocationTable);
    m_AllocationTable.Initialize(fs::SubStorage(m_StorageMeta), countAllocationTable);

    // アロケーションテーブル上のデータに対してマウントを行います。
    NN_RESULT_DO(
        m_StorageDirectoryEntry.InitializeBuffered(
            &m_AllocationTable,
            indexDirectoryEntry,
            static_cast<uint32_t>(sizeBlock),
            fs::SubStorage(&m_StorageData, offsetBody, sizeBody),
            pBufferManager
        )
    );
    NN_RESULT_DO(
        m_StorageFileEntry.InitializeBuffered(
            &m_AllocationTable,
            indexFileEntry,
            static_cast<uint32_t>(sizeBlock),
            fs::SubStorage(&m_StorageData, offsetBody, sizeBody),
            pBufferManager
        )
    );

    // ファイルシステムをマウントします。
    ScopedFinalizer<FileSystemObject> finalizeFileSystem(&m_FileSystem);
    m_FileSystem.Initialize(
        &m_AllocationTable,
        static_cast<uint32_t>(sizeBlock),
        &m_StorageDirectoryEntry,
        &m_StorageFileEntry
    );

    m_OffsetBody = offsetBody;
    m_CountBody = countAllocBlocksBody;
    m_SizeBlock = sizeBlock;

    finalizeFileSystem.Release();
    finalizeAllocationTable.Release();

    NN_RESULT_SUCCESS;
}

/**
* @brief セーブデータアーカイブをアンマウントします。
*
* @return 関数の処理結果を返します。
*/
void SaveDataFileSystemCore::Finalize() NN_NOEXCEPT
{
    if( m_CountBody == 0 )
    {
        return;
    }

    m_FileSystem.Finalize();
    m_AllocationTable.Finalize();
    m_OffsetBody = 0;
    m_CountBody = 0;
    m_SizeBlock = 0;
}

/**
* @brief ファイルを開きます。
*
* @param[out] outValue ファイルオブジェクト
* @param[in] path ファイルパス
* @param[in] mode オープンモード
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::OpenFileImpl(
           ExclusiveFile** outValue,
           const Path& path,
           int mode
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CountBody != 0);

    // 許可は
    //  Read
    //  Write
    //  Write|Read
    //  Write|AllowAppend
    //  Read|AllowAppend
    //  Write|Read|AllowAppend
    // 不許可は
    //  0
    //  AllowAppend
    static const int MaskSupported
        = nn::fs::OpenMode_Read | nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend;
    int modeMasked = mode & MaskSupported;
    if( modeMasked == 0 || modeMasked == nn::fs::OpenMode_AllowAppend )
    {
        return nn::fs::ResultInvalidOperationForOpenMode();
    }

    std::unique_ptr<SaveDataFile> pFile(new SaveDataFile(*GetFileOperationLocker()));
    if( pFile.get() == nullptr )
    {
        return nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCoreA();
    }

    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    // ファイルを開きます。
    NN_RESULT_DO(m_FileSystem.OpenFile(pFile->GetFileObject(), bufPath));

    // ファイルオブジェクトを初期化します。
    NN_RESULT_DO(
        pFile->Initialize(
            mode,
            &m_AllocationTable,
            static_cast<uint32_t>(m_SizeBlock),
            m_StorageData,
            m_OffsetBody,
            static_cast<int64_t>(m_SizeBlock) * m_CountBody
        )
    );

    *outValue = pFile.release();

    NN_RESULT_SUCCESS;
}

/**
* @brief ディレクトリを開きます。
*
* @param[out] outValue ディレクトリオブジェクト
* @param[in] path ディレクトリパス
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::OpenDirectoryImpl(
           ExclusiveDirectory** outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    NN_SDK_REQUIRES(m_CountBody != 0);

    // ディレクトリオブジェクト用にメモリを確保します。
    std::unique_ptr<SaveDataDirectory> pDirectory(new SaveDataDirectory(*GetFileOperationLocker()));
    if( pDirectory.get() == nullptr )
    {
        return nn::fs::ResultAllocationMemoryFailedInSaveDataFileSystemCoreB();
    }

    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    // ディレクトリを開きます。
    NN_RESULT_DO(m_FileSystem.OpenDirectory(pDirectory->GetDirectoryObject(), bufPath));

    // ディレクトリオブジェクトを初期化します。
    NN_RESULT_DO(pDirectory->Initialize());

    *outValue = pDirectory.release();

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルが存在しているかどうかを取得します。
*
* @param[out]   outValue    ファイルが存在しているかどうか
* @param[in]    path        ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::HasFileImpl(
           bool* outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    FileObject file;
    StorageIndex index;
    Result result = m_FileSystem.OpenFile(&index, &file, bufPath);
    if( nn::fs::ResultNotFound::Includes(result)
     || nn::fs::ResultPermissionDenied::Includes(result)
     || nn::fs::ResultUnsupportedOperation::Includes(result)
     || nn::fs::ResultIncompatiblePath::Includes(result)
    )
    {
        *outValue = false;
    }
    else if( result.IsFailure() )
    {
        return result;
    }
    else
    {
        *outValue = true;
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したディレクトリが存在しているかどうかを取得します。
*
* @param[out]   outValue    ディレクトリが存在しているかどうか
* @param[in]    path        ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::HasDirectoryImpl(
           bool* outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    DirectoryObject dir;
    Result result = m_FileSystem.OpenDirectory(&dir, bufPath);
    if( nn::fs::ResultNotFound::Includes(result)
     || nn::fs::ResultPermissionDenied::Includes(result)
     || nn::fs::ResultUnsupportedOperation::Includes(result)
     || nn::fs::ResultIncompatiblePath::Includes(result)
    )
    {
        *outValue = false;
    }
    else if( result.IsFailure() )
    {
        return result;
    }
    else
    {
        *outValue = true;
    }
    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルオブジェクト用リソースを解放します。
*
* @param[in]    pFile ファイル
*
* @return       関数の処理結果を返します。
*
* @details      ファイルオブジェクト用にメモリが確保されます。
*               確保されたメモリは @ref CloseFile を用いて解放する必要があります。
*/
void SaveDataFileSystemCore::CloseFileImpl(ExclusiveFile* pFile) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFile);
    if( pFile != nullptr )
    {
        delete pFile;
    }
}

/**
* @brief        ディレクトリオブジェクト用リソースを解放します。
*
* @param[in]    pDirectory ディレクトリオブジェクト
*
* @return       関数の処理結果を返します。
*
* @details      ディレクトリオブジェクト用にメモリが確保されます。
*               確保されたメモリは @ref CloseDirectory を用いて解放する必要があります。
*/
void SaveDataFileSystemCore::CloseDirectoryImpl(ExclusiveDirectory* pDirectory) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDirectory);
    if( pDirectory != nullptr )
    {
        delete pDirectory;
    }
}

/**
* @brief ファイルを削除します。
*
* @param[in] path ファイルパス
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::DeleteFileImpl(const Path& path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CountBody != 0);

    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    NN_RESULT_DO(m_FileSystem.DeleteFile(bufPath));

    NN_RESULT_SUCCESS;
}

/**
* @brief ディレクトリを削除します。
*
* @param[in] path ディレクトリパス
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::DeleteDirectoryImpl(const Path& path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CountBody != 0);

    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    NN_RESULT_DO(m_FileSystem.DeleteDirectory(bufPath));

    NN_RESULT_SUCCESS;
}

/**
* @brief ファイルを作成します。
*
* @param[in] path ファイルパス
* @param[in] size ファイルサイズ
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::CreateFileImpl(
           const Path& path,
           int64_t size
       ) NN_NOEXCEPT
{
    NN_FSP_REQUIRES(size >= 0, nn::fs::ResultInvalidSize());

    Result result;

    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    StorageIndex index;
    dbm::FileOptionalInfo fileInfo = {};
    NN_RESULT_DO(m_FileSystem.CreateFile(&index, bufPath, fileInfo));

    // ファイルオブジェクト内部のサイズを変更します。
    FileObject file;
    result = m_FileSystem.OpenFile(&file, bufPath);
    if( result.IsFailure() )
    {
        // エラー発生時は作成したファイルを削除します。
        (void)m_FileSystem.DeleteFile(bufPath);
        return result;
    }
    result = file.Resize(size);
    if( result.IsFailure() )
    {
        // エラー発生時は作成したファイルを削除します。
        (void)m_FileSystem.DeleteFile(bufPath);
        return result;
    }

    // 新規確保した領域をゼロ埋めします。
    if( 0 < size )
    {
        AllocationTableStorage storageTable;
        uint32_t indexTop = 0;
        auto finished = false;
        NN_RESULT_DO(file.IterateBegin(&indexTop, &finished, 0));
        NN_RESULT_DO(
            storageTable.Initialize(
                &m_AllocationTable,
                indexTop,
                static_cast<uint32_t>(m_SizeBlock),
                fs::SubStorage(&m_StorageData, m_OffsetBody, m_CountBody * m_SizeBlock)
            )
        );
        NN_UTIL_SCOPE_EXIT
        {
            storageTable.Finalize();
        };

        NN_RESULT_DO(storageTable.OperateRange(fs::OperationId::FillZero, 0, size));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief ディレクトリを作成します。
*
* @param[in] path ディレクトリパス
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::CreateDirectoryImpl(const Path& path) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CountBody != 0);

    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    NN_RESULT_DO(m_FileSystem.CreateDirectory(bufPath));

    NN_RESULT_SUCCESS;
}

/**
* @brief ファイルをリネームします。
*        ファイル操作用ロックを取得した状態で呼び出してください。
*
* @param[in] oldPath 変更対象のファイルパス
* @param[in] newPath 変更後のファイルパス
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::RenameFileImpl(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CountBody != 0);

    // パスの正当性のみチェックします。
    // dbm レイヤーには正規化してないパスを渡します。
    Path::Char buf[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(buf, oldPath));
    NN_RESULT_DO(GetStringPath(buf, newPath));

    bool isFile;
    bool isSameEntry;
    Result result = m_FileSystem.RenameFile(
                        &isFile,
                        &isSameEntry,
                        newPath.GetString(),
                        oldPath.GetString()
                    );
// 上書きリネームは今は対応しない
#if 0
    if( result.IsFailure() )
    {
        if( nn::fs::ResultAlreadyExists::Includes(result) && isFile && !isSameEntry )
        {
            // 上書きリネームを行ないます。
            NN_RESULT_DO(DeleteFileImpl(newPath));
            NN_RESULT_DO(
                m_FileSystem.RenameFile(
                    newPath.GetString(),
                    oldPath.GetString()
                )
            );
        }
        else
        {
            return result;
        }
    }
    NN_RESULT_SUCCESS;
#else
    return result;
#endif
}

/**
* @brief ディレクトリをリネームします。
*
* @param[in] oldPath 変更対象のディレクトリパス
* @param[in] newPath 変更後のディレクトリパス
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::RenameDirectoryImpl(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_CountBody != 0);

    // パスの正当性のみチェックします。
    // dbm レイヤーには正規化してないパスを渡します。
    Path::Char buf[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(buf, oldPath));
    NN_RESULT_DO(GetStringPath(buf, newPath));

    bool isFile;
    bool isSameEntry;
    bool isParentEntry;
    Result result = m_FileSystem.RenameDirectory(
                        &isFile,
                        &isSameEntry,
                        &isParentEntry,
                        newPath.GetString(),
                        oldPath.GetString()
                    );
// 上書きリネームは今は対応しない
#if 0
    if( nn::fs::ResultDbmAlreadyExists::Includes(result) && !isFile && !isSameEntry && !isParentEntry )
    {
        // 上書きリネームを行ないます。
        NN_RESULT_DO(
            DeleteDirectoryRecursively(newPath)
        );
        NN_RESULT_DO(
            m_FileSystem.RenameDirectory(
                &isFile,
                &isSameEntry,
                &isParentEntry,
                newPath.GetString(),
                oldPath.GetString()
            )
        );
    }
    else if( result.IsFailure() )
    {
        return result;
    }

    NN_RESULT_SUCCESS;
#else
    return result;
#endif
}

/**
* @brief        コンストラクタ
*/
SaveDataFile::SaveDataFile(os::Mutex& locker) NN_NOEXCEPT
    : m_LockObject(locker),
      m_pAllocationTable(nullptr),
      m_Storage(),
      m_SizeBlock(0),
      m_OffsetStorage(0),
      m_SizeStorage(0)
{
}

/**
* @brief        デストラクタ
*/
SaveDataFile::~SaveDataFile() NN_NOEXCEPT
{
}

/**
* @brief        ファイルデータアクセス用ユーティリティを初期化します。
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::InitializeTable() NN_NOEXCEPT
{
    // ファイルデータの先頭インデックスを取得します。
    uint32_t indexTop = 0;
    bool isFinished;
    NN_RESULT_DO(m_FileObject.IterateBegin(&indexTop, &isFinished, 0));

    // ファイルデータアクセス用ユーティリティを初期化します。
    return m_Table.Initialize(
               m_pAllocationTable,
               indexTop,
               static_cast<uint32_t>(m_SizeBlock),
               fs::SubStorage(&m_Storage, m_OffsetStorage, m_SizeStorage)
           );
}

/**
* @brief        ファイルオブジェクトを初期化します。
*
* @param[in]    mode                ファイルオープンモード
* @param[in]    pAllocationTable    実データ領域でのデータ管理用アロケーションテーブル
* @param[in]    sizeBlock           ブロックサイズ
* @param[in]    pStorage            実データ用ストレージ
* @param[in]    offsetStorage       実データ用ストレージの開始オフセット
* @param[in]    sizeStorage         実データ用ストレージのサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::Initialize(
           int mode,
           AllocationTable* pAllocationTable,
           uint32_t sizeBlock,
           fs::SubStorage storage,
           int64_t offsetStorage,
           int64_t sizeStorage
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pAllocationTable);

    NN_UNUSED(mode);

    m_pAllocationTable = pAllocationTable;
    m_SizeBlock = sizeBlock;
    m_Storage = storage;
    m_OffsetStorage = offsetStorage;
    m_SizeStorage = sizeStorage;

    // ファイルサイズが 0 以上であればアロケーションテーブルに
    // 領域を保持しているのでリードライト用に初期化を行います。
    if( m_FileObject.GetSize() > 0 )
    {
        NN_RESULT_DO(InitializeTable());
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルの内容をバッファに読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::ReadBytes(
           int64_t offset,
           void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    // 読み込み属性があるかどうかをチェックします。
    if( (GetMode() & nn::fs::OpenMode_Read) == 0 )
    {
        return nn::fs::ResultReadUnpermitted();
    }

    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    int64_t maxSize = m_FileObject.GetSize();
    // 読み込み開始位置をチェックします。
    if( offset > maxSize )
    {
        return nn::fs::ResultInvalidOffset();
    }

    // 読み込むサイズを調整します。
    if( static_cast<int64_t>(offset + size) > maxSize )
    {
        size = static_cast<size_t>(maxSize - offset);
    }

    // データの読み込みを実行します。
    if( size > 0 )
    {
        Result result = m_Table.Read(offset, buffer, size);
        NN_RESULT_DO(result);
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        バッファの内容をファイルに書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::WriteBytes(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buffer);

    // 書き込み属性があるかどうかをチェックします。
    if( (GetMode() & nn::fs::OpenMode_Write) == 0 )
    {
        return nn::fs::ResultWriteUnpermitted();
    }

    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    int64_t maxSize = m_FileObject.GetSize();

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    // 書き込み開始位置をチェックします。
    // offset がファイルサイズより大きく、且つ自動拡張が不許可ならエラーを返します
    if( offset > maxSize )
    {
        if( (GetMode() & nn::fs::OpenMode_AllowAppend) == 0 )
        {
            return nn::fs::ResultInvalidOffset();
        }
    }

    // 書き込むサイズを調整します。
    if( static_cast<int64_t>(offset + size) > maxSize )
    {
        if( (GetMode() & nn::fs::OpenMode_AllowAppend) == 0 )
        {
            // 自動伸長が有効になっていません。
            return nn::fs::ResultFileExtensionWithoutOpenModeAllowAppend();
        }
        else
        {
            NN_RESULT_DO(SetSize(offset + size));
        }
    }

    // データの書き込みを実行します。
    NN_RESULT_DO(m_Table.Write(offset, buffer, size));

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルサイズを取得します。
*
* @param[out]   outValue    ファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    *outValue = m_FileObject.GetSize();

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルサイズを変更します。
*
* @param[in]    size    変更後のファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::SetSize(int64_t size) NN_NOEXCEPT
{
    if( size < 0 )
    {
        return nn::fs::ResultInvalidSize();
    }
    if( (GetMode() & nn::fs::OpenMode_Write) == 0 )
    {
        return nn::fs::ResultWriteUnpermitted();
    }

    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    // ファイルオブジェクト内部のサイズを変更します。
    const auto previousSize = m_FileObject.GetSize();
    NN_RESULT_DO(m_FileObject.Resize(size));

    // ファイルサイズが 0 以上であればアロケーションテーブルに
    // 領域を保持しているのでリードライト用に初期化を行います。
    // 逆に、ファイルサイズが 0 になるのであればリードライト用
    // ユーティリティをリセットします。
    if( size > 0 )
    {
        NN_RESULT_DO(InitializeTable());
    }
    else
    {
        m_Table.Finalize();
    }

    // 新規確保した領域をゼロ埋めします。
    if( previousSize < size )
    {
        NN_RESULT_DO(m_Table.OperateRange(
                         fs::OperationId::FillZero,
                         previousSize,
                         size - previousSize
                     ));
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result SaveDataFile::Flush() NN_NOEXCEPT
{
    if( (GetMode() & fs::OpenMode_Write) == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    if( 0 < m_FileObject.GetSize() )
    {
        return m_Table.Flush();
    }
    else
    {
        NN_RESULT_SUCCESS;
    }
}

/**
* @brief       範囲指定処理を行います。
*
* @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
* @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
* @param[in]   operationId      範囲指定処理の種類
* @param[in]   offset           範囲指定処理開始位置
* @param[in]   size             範囲指定処理を行うデータサイズ
* @param[in]   inBuffer         範囲指定処理に渡すバッファ
* @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
*
* @return      関数の処理結果を返します。
*/
Result SaveDataFile::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    // 操作を実行します。
    return m_Table.OperateRange(
        outBuffer,
        outBufferSize,
        operationId,
        offset,
        size,
        inBuffer,
        inBufferSize);
}

/**
* @brief        コンストラクタ
*/
SaveDataDirectory::SaveDataDirectory(os::Mutex& locker) NN_NOEXCEPT
    : m_LockObject(locker)
{
}

/**
* @brief        デストラクタ
*/
SaveDataDirectory::~SaveDataDirectory() NN_NOEXCEPT
{
}

/**
* @brief        ディレクトリオブジェクトを初期化します。
*
* @return       関数の処理結果を返します。
*/
Result SaveDataDirectory::Initialize() NN_NOEXCEPT
{
    // イテレータを初期化します。
    return m_DirectoryObject.FindOpen();
}

/**
* @brief        ディレクトリ内の子ディレクトリエントリーを取得します。
*
* @param[out]   outNumEntries   取得したエントリー数
* @param[out]   outEntries      取得したエントリー情報
*                               配列のサイズは numEntries 以上である必要があります。
* @param[in]    numEntries      取得するエントリー数
*
* @return       関数の処理結果を返します。
*/
Result SaveDataDirectory::Read(
           int32_t* outNumEntries,
           nn::fs::DirectoryEntry outEntries[],
           int32_t numEntries
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outNumEntries);
    NN_SDK_ASSERT_NOT_NULL(outEntries);

    Result result;
    int32_t entryCount = 0;

    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    // 子ディレクトリを取得します。
    for( ; ; )
    {
        dbm::DirectoryName dirName;
        bool isFindFinished;
        NN_RESULT_DO(m_DirectoryObject.FindNextDirectory(&dirName, &isFindFinished));
        if( isFindFinished )
        {
            break;
        }

        // 情報を設定します。
        std::memset(&outEntries[entryCount], 0, sizeof(nn::fs::DirectoryEntry));
        std::memcpy(outEntries[entryCount].name, dirName.name, sizeof(dirName.name));
        outEntries[entryCount].name[sizeof(dirName.name)] = 0;
        outEntries[entryCount].directoryEntryType = nn::fs::DirectoryEntryType_Directory;
        outEntries[entryCount].fileSize = 0;

        if( ++entryCount >= numEntries )
        {
            *outNumEntries = numEntries;
            NN_RESULT_SUCCESS;
        }
    }

    // 子ファイルを取得します。
    for( ; ; )
    {
        dbm::FileName fileName;
        int64_t sizeFile;
        bool isFindFinished;
        NN_RESULT_DO(m_DirectoryObject.FindNextFile(&sizeFile, &fileName, &isFindFinished));
        if( isFindFinished )
        {
            break;
        }

        // 情報を設定します。
        std::memset(&outEntries[entryCount], 0, sizeof(nn::fs::DirectoryEntry));
        std::memcpy(outEntries[entryCount].name, fileName.name, sizeof(fileName.name));
        outEntries[entryCount].name[dbm::MaxFileNameLength] = 0;
        outEntries[entryCount].directoryEntryType = nn::fs::DirectoryEntryType_File;
        outEntries[entryCount].fileSize = sizeFile;

        if( ++entryCount >= numEntries )
        {
            *outNumEntries = numEntries;
            NN_RESULT_SUCCESS;
        }
    }

    *outNumEntries = entryCount;

    NN_RESULT_SUCCESS;
}

/**
* @brief        削除の通知を送ります。列挙が破綻しないように調整します。
*
* @param[in]    path    削除するファイルパス
*
* @return       関数の処理結果を返します。
*/
Result SaveDataDirectory::NotifyDelete(int64_t id, bool isFile) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(&m_LockObject);

    bool isFinished;
    return m_DirectoryObject.Notify(&isFinished, id, isFile);
}

/**
* @brief        管理領域をフォーマットします。
*
* @param[in]    pFileSystemHeader   管理領域
* @param[in]    storageControlArea  管理領域として使用するストレージ
* @param[in]    storageMeta         メタデータ領域として使用するストレージ
* @param[in]    storageData         実データ領域として使用するストレージ
* @param[in]    sizeBlock           実データ領域のブロックサイズ
* @param[in]    countBlocks         実データ領域に割り当てるブロック数
* @param[in]    pBufferManager      バッファマネージャ
*
* @return       関数の処理結果を返します。
*
* @details      SaveDataFileSystemCore::Format からのみ呼びだされます。
*/
Result SaveDataFileSystemCore::FormatDbmLayer(
           fs::SubStorage storageDbmControlArea,
           fs::SubStorage storageMeta,
           fs::SubStorage storageData,
           uint32_t sizeBlock,
           uint32_t countBlocks,
           fssystem::IBufferManager* pBufferManager
       ) NN_NOEXCEPT
{
    ScopedFinalizeObject<FileSystemControlArea> controlArea;
    controlArea->Initialize(storageDbmControlArea);

    // 初期化パラメーターを設定します。
    NN_RESULT_DO(controlArea->WriteBlockSize(sizeBlock));
    NN_RESULT_DO(controlArea->WriteAllocationTableInfo(0, countBlocks));
    NN_RESULT_DO(controlArea->WriteDataBodyInfo(0, countBlocks));

    // アローケーションテーブル(可変長 DB)を生成します。
    NN_RESULT_DO(AllocationTable::Format(storageMeta, countBlocks));
    AllocationTable allocationTable;
    allocationTable.Initialize(storageMeta, countBlocks);

    // ディレクトリエントリー用領域を可変長 DB 上に確保します。
    uint32_t indexDirEntry = static_cast<uint32_t>(-1);
    size_t sizeDirectoryEntry = FileSystemObject::QueryDirectoryEntryStorageSize(0);
    size_t blockCountDirectoryEntry = (sizeDirectoryEntry + sizeBlock - 1) / sizeBlock;
    NN_RESULT_DO(allocationTable.Allocate(&indexDirEntry, static_cast<uint32_t>(blockCountDirectoryEntry)));
    NN_RESULT_DO(controlArea->WriteDirectoryEntryInfo(indexDirEntry));

    // ファイルエントリー用領域を可変長 DB 上に確保します。
    uint32_t indexFileEntry = static_cast<uint32_t>(-1);
    size_t sizeFileEntry = FileSystemObject::QueryFileEntryStorageSize( 0 );
    size_t blockCountFileEntry = (sizeFileEntry + sizeBlock - 1) / sizeBlock;
    NN_RESULT_DO(allocationTable.Allocate(&indexFileEntry, static_cast<uint32_t>(blockCountFileEntry)));
    NN_RESULT_DO(controlArea->WriteFileEntryInfo(indexFileEntry));

    // 管理領域に書き込んだ値を使用して各オブジェクトをフォーマットします。
    BufferedAllocationTableStorage storageDirectoryEntry;
    NN_RESULT_DO(
        storageDirectoryEntry.InitializeBuffered(
            &allocationTable,
            indexDirEntry,
            sizeBlock,
            fs::SubStorage(
                &storageData,
                0,
                static_cast<int64_t>(countBlocks) * sizeBlock
            ),
            pBufferManager
        )
    );

    BufferedAllocationTableStorage storageFileEntry;
    NN_RESULT_DO(
        storageFileEntry.InitializeBuffered(
            &allocationTable,
            indexFileEntry,
            sizeBlock,
            fs::SubStorage(
                &storageData,
                0,
                static_cast<int64_t>(countBlocks) * sizeBlock
            ),
            pBufferManager
        )
    );

    FileSystemObject fileSystemObject;
    NN_RESULT_DO(
        fileSystemObject.Format(
            &storageDirectoryEntry,
            &storageFileEntry
        )
    );

    NN_RESULT_SUCCESS;
}

/**
* @brief パスの正規化を行います。
*
* @param[out] pCount 正規化後のパス
* @param[in] path パスオブジェクト
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::GetStringPath(
           Path::Char* outBuf,
           const Path& path
       ) NN_NOEXCEPT
{
    size_t normalizedLength;
    NN_RESULT_TRY(
        PathTool::Normalize(
            outBuf,
            &normalizedLength,
            path.GetString(),
            nn::fs::EntryNameLengthMax
        )
    )
    NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
    {
        return nn::fs::ResultInvalidPath();
    }
    NN_RESULT_END_TRY;

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルパスからIDを取得します。
*
* @param[out]   outId   ID
* @param[in]    path    フルパス
* @param[in]    isFile  ファイルであれば true、ディレクトリであれば false
*
* @return 関数の処理結果を返します。
*/
Result SaveDataFileSystemCore::GetFileIdFromPath(
           int64_t *outId,
           const Path& path,
           bool isFile
       ) NN_NOEXCEPT
{
    Path::Char bufPath[nn::fs::EntryNameLengthMax + 1];
    NN_RESULT_DO(GetStringPath(bufPath, path));

    NN_SDK_ASSERT_NOT_NULL(outId);
    if( isFile )
    {
        FileObject file;
        StorageIndex index;
        NN_RESULT_TRY(m_FileSystem.OpenFile(&index, &file, bufPath))
            NN_RESULT_CATCH(fs::ResultBufferAllocationFailed)
            {
                NN_RESULT_RETHROW;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(fs::ResultFileNotFound());
            }
        NN_RESULT_END_TRY;

        *outId = index;
        NN_RESULT_SUCCESS;
    }
    else
    {
        DirectoryObject dir;
        StorageIndex index = 0;
        NN_RESULT_TRY(m_FileSystem.OpenDirectory(&index, &dir, bufPath))
            NN_RESULT_CATCH(fs::ResultBufferAllocationFailed)
            {
                NN_RESULT_RETHROW;
            }
            NN_RESULT_CATCH_ALL
            {
                NN_RESULT_THROW(fs::ResultFileNotFound());
            }
        NN_RESULT_END_TRY;

        *outId = index;
        NN_RESULT_SUCCESS;
    }
}

/**
* @brief        ID からエントリとディレクトリが親子関係にあるか調べる
*
* @param[out]   outIsSubEntry       checkEntryId が baseDirectoryId の子か否か
* @param[in]    baseDirectoryId     ディレクトリの ID
* @param[in]    checkEntryId        エントリの ID
* @param[in]    isFile              ファイルであれば true、ディレクトリであれば false
*
* @return       関数の処理結果を返します。
*
* @details      ID からエントリとディレクトリが親子関係にあるか調べる
*/
Result SaveDataFileSystemCore::CheckSubEntry(
        bool* outIsSubEntry,
        StorageIndex baseDirectoryId,
        StorageIndex checkEntryId,
        bool isFile
    ) const NN_NOEXCEPT
{
    return m_FileSystem.CheckSubEntry(
                outIsSubEntry,
                baseDirectoryId,
                checkEntryId,
                isFile
            );
}

// セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
nn::Result SaveDataFileSystemCore::AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pVisitor);
    NN_RESULT_DO(pVisitor->VisitStorage(InternalStorageFileNameAllocationTableControlArea, &m_StorageControlArea));
    NN_RESULT_DO(pVisitor->VisitStorage(InternalStorageFileNameAllocationTableMeta,        &m_StorageMeta));
    NN_RESULT_DO(pVisitor->VisitStorage(InternalStorageFileNameAllocationTableData,        &m_StorageData));

    NN_RESULT_SUCCESS;
}

const char SaveDataFileSystemCore::InternalStorageFileNameAllocationTableControlArea[]      = "AllocationTableControlArea";
const char SaveDataFileSystemCore::InternalStorageFileNameAllocationTableMeta[]             = "AllocationTableMeta";
const char SaveDataFileSystemCore::InternalStorageFileNameAllocationTableData[]             = "AllocationTableData";
const char SaveDataFileSystemCore::InternalStorageFileNameAllocationTableDataWithZeroFree[] = "AllocationTableDataWithZeroFree";

}}}
