﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fssystem/save/fs_ISaveFile.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>
#include <nn/fssystem/save/fs_IntegrityVerificationStorage.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief        コンストラクタ
*/
IntegrityVerificationStorage::IntegrityVerificationStorage() NN_NOEXCEPT
    : m_SizeVerificationBlock(0),
      m_OrderVerificationBlock(0),
      m_SizeUpperLayerVerificationBlock(0),
      m_OrderUpperLayerVerificationBlock(0),
      m_pBuffer(nullptr)
{
}

/**
* @brief        デストラクタ
*/
IntegrityVerificationStorage::~IntegrityVerificationStorage() NN_NOEXCEPT
{
    Finalize();
}

/**
* @brief        完全性検証ストレージレイヤーをマウントします。
*
* @param[in]    storageHash                     ハッシュデータストレージ
* @param[in]    storageData                     ハッシュ検証対象ストレージ
* @param[in]    sizeVerificationBlock           ハッシュブロックサイズ
* @param[in]    sizeUpperLayerVerificationBlock 上位レイヤーのハッシュブロックサイズ
* @param[in]    pBuf                            検証済みハッシュを一時的においておくためのバッファ
* @param[in]    salt                            ハッシュ計算に使用するソルト
* @param[in]    isRealData                      実データに対する完全性検証ストレージかどうか
* @param[in]    storageType                     使用するストレージのタイプ
*
* @return       関数の処理結果を返します。
*/
void IntegrityVerificationStorage::Initialize(
         fs::SubStorage storageHash,
         fs::SubStorage storageData,
         int64_t sizeVerificationBlock,
         int64_t sizeUpperLayerVerificationBlock,
         IBufferManager* pBuf,
         const fs::HashSalt& salt,
         bool isRealDataSign,
         fs::StorageType storageType
     ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(sizeVerificationBlock >= 32);
    NN_SDK_ASSERT_NOT_NULL(pBuf);

    m_StorageHash = storageHash;
    m_StorageData = storageData;

    // ブロックサイズからシフト値を求めておきます。
    m_SizeVerificationBlock = sizeVerificationBlock;
    m_OrderVerificationBlock = ILog2(static_cast<uint32_t>(sizeVerificationBlock));
    NN_SDK_ASSERT((1LL << m_OrderVerificationBlock) == m_SizeVerificationBlock);
    m_pBuffer = pBuf;

    // 上位レイヤーでのハッシュブロックサイズを取得します。
    if( sizeUpperLayerVerificationBlock < HashSize )
    {
        sizeUpperLayerVerificationBlock = HashSize;
    }
    m_SizeUpperLayerVerificationBlock = sizeUpperLayerVerificationBlock;
    m_OrderUpperLayerVerificationBlock
        = ILog2(static_cast<uint32_t>(sizeUpperLayerVerificationBlock));
    NN_SDK_ASSERT((1LL << m_OrderUpperLayerVerificationBlock)
        == m_SizeUpperLayerVerificationBlock);

    // 初期化パラメーターを確認します。
    {
        int64_t sizeHash = 0;
        int64_t sizeData = 0;
        NN_SDK_ASSERT(storageHash.GetSize(&sizeHash).IsSuccess());
        NN_SDK_ASSERT(storageData.GetSize(&sizeData).IsSuccess());
        NN_UNUSED(sizeHash);
        NN_UNUSED(sizeData);
        NN_SDK_ASSERT(((sizeHash / HashSize) * m_SizeVerificationBlock) >= sizeData);
    }

    std::memcpy(m_Salt.value, salt.value, fs::HashSalt::Size);

    m_IsRealDataSign = isRealDataSign;
    m_StorageType = storageType;
}

/**
* @brief        完全性検証ストレージレイヤーをアンマウントします。
*/
void IntegrityVerificationStorage::Finalize() NN_NOEXCEPT
{
    if( m_pBuffer != nullptr )
    {
        m_StorageHash = fs::SubStorage();
        m_StorageData = fs::SubStorage();
        m_pBuffer = nullptr;
    }
}

/**
* @brief        ハッシュ対象ストレージのサイズを取得します。
*
* @param[out]   outValue    ハッシュ対象ストレージのサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegrityVerificationStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    return m_StorageData.GetSize(outValue);
}

/**
* @brief        完全性検証ストレージレイヤーからデータを読み込みます。
*
* @param[in]    offset  読み込み開始位置
* @param[out]   buffer  読み込んだ内容をコピーするバッファ
* @param[in]    size    読み込むデータサイズ
*
* @return       関数の処理結果を返します。
*
* @details      読み込もうとしたデータに対するハッシュが不正だった場合には、0 フィルされたデータが
*               読込先バッファに書き込まれ、nn::fs::ResultIntegrityVerificationStorageCorrupted を返します。
*/
Result IntegrityVerificationStorage::Read(
           int64_t offset,
           void* buffer,
           const size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT(size != 0);

    // 検証レイヤーへのアクセスは、必ず検証ブロック単位で要求するものとします。
    NN_SDK_ASSERT(0 == (offset & (m_SizeVerificationBlock - 1)));
    NN_SDK_ASSERT(0 == (size & (m_SizeVerificationBlock - 1)));

    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());

    int64_t sizeData;
    NN_RESULT_DO(m_StorageData.GetSize(&sizeData));

    NN_RESULT_THROW_UNLESS(sizeData >= offset, nn::fs::ResultInvalidOffset());

    // 極端に大きい場合はパラメーターエラー
    // このような値は指定できないと割り切ります。
    NN_RESULT_THROW_UNLESS(
        nn::fs::IStorage::CheckAccessRange(
            offset,
            size,
            nn::util::align_up(sizeData, static_cast<size_t>(m_SizeVerificationBlock))
        ),
        nn::fs::ResultOutOfRange()
    );

    // 長さの調整
    size_t sizeReadRequest = size;
    if( static_cast<int64_t>(offset + sizeReadRequest) > sizeData )
    {
        // データ末尾パディング
        int64_t offsetPadding = sizeData - offset;
        size_t sizePadding = static_cast<size_t>(m_SizeVerificationBlock
            - (offsetPadding & (m_SizeVerificationBlock - 1)));
        NN_SDK_ASSERT(static_cast<int64_t>(sizePadding) < m_SizeVerificationBlock);

        // データがない部分を0クリアします。
        std::memset(static_cast<uint8_t *>(buffer) + offsetPadding, 0, sizePadding);

        // サイズを末尾に収まるように調整します。
        sizeReadRequest = static_cast<size_t>(sizeData - offset);
    }

    // バッファに検証対象のデータを読み込みます。
    {
        Result resultRead = m_StorageData.Read(offset, buffer, sizeReadRequest);
        if( resultRead.IsFailure() )
        {
            std::memset(buffer, 0, size);
            return resultRead;
        }
    }

    nn::Result resultVerifyHash = nn::ResultSuccess();

    // 検証単位に区切りつつ、完全性検証を行います。
    const auto countSignature = size >> m_OrderVerificationBlock;
    PooledBuffer signatureBuffer(countSignature * sizeof(BlockHash), sizeof(BlockHash));
    const auto countBuffer = std::min(
        countSignature,
        signatureBuffer.GetSize() / sizeof(BlockHash)
    );
    size_t countVerified = 0;
    while( countVerified < countSignature )
    {
        const auto countVerify = std::min(countBuffer, countSignature - countVerified);

        auto resultCur = ReadBlockSignature(
            signatureBuffer.GetBuffer(),
            signatureBuffer.GetSize(),
            offset + (countVerified << m_OrderVerificationBlock),
            countVerify << m_OrderVerificationBlock
        );

        ScopedThreadPriorityChanger changePriority(+1, ScopedThreadPriorityChanger::Mode::Relative);

        for( size_t indexVerify = 0;
            indexVerify < countVerify && resultCur.IsSuccess();
            ++indexVerify )
        {
            const auto sizeVerified = (indexVerify + countVerified) << m_OrderVerificationBlock;
            resultCur = VerifyHash(
                static_cast<char*>(buffer) + sizeVerified,
                reinterpret_cast<BlockHash*>(signatureBuffer.GetBuffer()) + indexVerify
            );

            if( nn::fs::ResultIntegrityVerificationStorageCorrupted::Includes(resultCur) )
            {
                // ハッシュ検証失敗または対応する署名がオールゼロである領域
                //
                // ダーティなデータを誤って使わないよう、バッファをクリアします。
                // また、ハッシュがオールゼロの場合、データもゼロクリアされているものとして扱います。
                //
                // この過程で、
                //  ・完全性検証に失敗した箇所は0クリア
                //  ・完全性検証に成功した箇所は値が入っている
                // 配列が出来上がります。
                //
                // 戻り値で"ResultFailedVerifySign"を返すときは
                // 戻り値の一部、もしくは全てがベリファイエラーでゼロクリアされていることを示すものとします。
                std::memset(
                    static_cast<char*>(buffer) + sizeVerified,
                    0,
                    static_cast<size_t>(m_SizeVerificationBlock)
                );

                if( !nn::fs::ResultClearedRealDataVerificationFailed::Includes(resultCur) &&
                    m_StorageType != fs::StorageType_Authoring )
                {
                    resultVerifyHash = resultCur;
                }

                resultCur = ResultSuccess();
            }
        }

        if( resultCur.IsFailure() )
        {
            // 致命的なエラー
            std::memset(buffer, 0, size);
            return resultCur;
        }

        countVerified += countVerify;
    }

    return resultVerifyHash;
}

/**
* @brief        完全性検証ストレージレイヤーにデータを書き込みます。
*
* @param[in]    offset  書き込み開始位置
* @param[in]    buffer  書き込むデータ
* @param[in]    size    書き込むデータサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegrityVerificationStorage::Write(
           int64_t offset,
           const void* buffer,
           size_t size
       ) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());
    NN_RESULT_THROW_UNLESS(nn::fs::IStorage::CheckOffsetAndSize(offset, size), nn::fs::ResultInvalidOffset());

    Result result = nn::ResultSuccess();
    int64_t sizeData;
    NN_RESULT_DO(m_StorageData.GetSize(&sizeData));

    if( offset >= sizeData )
    {
        return nn::fs::ResultInvalidOffset();
    }
    NN_RESULT_THROW_UNLESS(
        CheckAccessRange(offset, size,
        nn::util::align_up(sizeData, static_cast<size_t>(m_SizeVerificationBlock))),
        nn::fs::ResultOutOfRange()
    );

    // 検証レイヤーへのアクセスは、下に挟まるBufferdFile層がすべてのパラメーターを適切にケアするものとします。
    //
    // * 必ず検証ブロック単位で要求するものとします。
    // * サイズ外の要求ははじかれるものとします。
    //
    // したがって、この層が不適切な値で呼び出された場合は
    //
    // * BufferdFile層の実装が不正ということを意味します。
    // * ResultInvalidPosition, ResultInvalidParametor等は返しません。
    // * 基本的にASSERT止めで通知。
    //
    // というルールにします。

    NN_SDK_ASSERT(0 == (offset & (m_SizeVerificationBlock - 1)));
    NN_SDK_ASSERT(0 == (size & (m_SizeVerificationBlock - 1)));
    NN_SDK_ASSERT(offset <= sizeData);
    NN_SDK_ASSERT(static_cast<int64_t>(offset + size) < sizeData + m_SizeVerificationBlock);

    // ストレージサイズを超える、パディング部分は必ず0クリアされていなければなりません
    // そうしなければデータの完全性検証が出来ません
    // size == 0 の場合は下記の判定は必ず false
    if( static_cast<int64_t>(offset + size) > sizeData )
    {
        const uint8_t* p = static_cast<const uint8_t*>(buffer) + sizeData - offset;
        const uint8_t* pe = p + offset + size - sizeData;

        while( p < pe )
        {
            NN_SDK_ASSERT(!*p);
            p++;
        }
    }

    // ストレージサイズに切り詰めます。
    auto sizeWriteRequest = size;
    if( static_cast<int64_t>(offset + sizeWriteRequest) > sizeData )
    {
        sizeWriteRequest = static_cast<size_t>(sizeData - offset);

        // 丸められた結果 size == 0 となった場合には即時復帰(明示的な size == 0 の場合と切り分け)
        if( sizeWriteRequest == 0 )
        {
            NN_RESULT_SUCCESS;
        }
    }
    const auto sizeAlignedWriteRequest
        = static_cast<size_t>(
              (sizeWriteRequest + (m_SizeVerificationBlock - 1)) & ~(m_SizeVerificationBlock - 1)
          );

    // 検証単位に区切りつつ、ハッシュ検証を行います。
    size_t countUpdated = 0;

    {
        const auto countSignature = sizeAlignedWriteRequest >> m_OrderVerificationBlock;
        PooledBuffer signatureBuffer(countSignature * sizeof(BlockHash), sizeof(BlockHash));
        const auto countBuffer = std::min(
                                     countSignature,
                                     signatureBuffer.GetSize() / sizeof(BlockHash)
                                 );

        while( countUpdated < countSignature )
        {
            const auto countUpdate = std::min(countBuffer, countSignature - countUpdated);

            {
                ScopedThreadPriorityChanger changePriority(
                    +1,
                    ScopedThreadPriorityChanger::Mode::Relative);

                for( size_t indexUpdate = 0; indexUpdate < countUpdate; ++indexUpdate )
                {
                    const auto sizeUpdated = (indexUpdate + countUpdated) << m_OrderVerificationBlock;
                    CalcBlockHash(
                        reinterpret_cast<BlockHash*>(signatureBuffer.GetBuffer()) + indexUpdate,
                        reinterpret_cast<const char*>(buffer) + sizeUpdated
                    );
                }
            }

            result = WriteBlockSignature(
                signatureBuffer.GetBuffer(),
                signatureBuffer.GetSize(),
                offset + (countUpdated << m_OrderVerificationBlock),
                countUpdate << m_OrderVerificationBlock
            );
            if( result.IsFailure() )
            {
                break;
            }

            countUpdated += countUpdate;
        }
    }

    // バッファにデータを書き込みます。
    NN_RESULT_DO(
        m_StorageData.Write(
            offset,
            buffer,
            std::min(sizeWriteRequest, countUpdated << m_OrderVerificationBlock)
        )
    );

    return result;
} // NOLINT(impl/function_size)

/**
* @brief        フラッシュします。
*
* @return       関数の処理結果を返します。
*/
Result IntegrityVerificationStorage::Flush() NN_NOEXCEPT
{
    NN_RESULT_DO(m_StorageHash.Flush());
    NN_RESULT_DO(m_StorageData.Flush());
    NN_RESULT_SUCCESS;
}

/**
* @brief        範囲を指定して操作を実行します。
*
* @param[out]   outBuffer       範囲指定処理の結果を格納するバッファ
* @param[in]    outBufferSize   範囲指定処理の結果を格納するバッファのサイズ
* @param[in]    operationId     実行する操作
* @param[in]    offset          操作を実行する範囲の先頭オフセット
* @param[in]    size            操作を実行する範囲のサイズ
* @param[in]    inBuffer        範囲指定処理に渡すバッファ
* @param[in]    inBufferSize    範囲指定処理に渡すバッファのサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegrityVerificationStorage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_ALIGNED(offset, static_cast<size_t>(m_SizeVerificationBlock));
    NN_SDK_REQUIRES_ALIGNED(size, static_cast<size_t>(m_SizeVerificationBlock));

    switch( operationId )
    {
    case fs::OperationId::FillZero:
        {
            NN_SDK_ASSERT(m_StorageType == fs::StorageType_SaveData);
            int64_t sizeStorageData = 0;
            NN_RESULT_DO(m_StorageData.GetSize(&sizeStorageData));
            if( (offset < 0) || (sizeStorageData < static_cast<int64_t>(offset)) )
            {
                return nn::fs::ResultInvalidOffset();
            }

            const auto offsetSign = (offset >> m_OrderVerificationBlock) * HashSize;
            const auto sizeSign
                = (std::min(size, sizeStorageData - offset) >> m_OrderVerificationBlock)
                * HashSize;

            // 最大 4 ブロック分のバッファを確保しゼロクリアします。
            const auto sizeBuf = static_cast<size_t>(std::min(
                sizeSign, static_cast<int64_t>(1) << (m_OrderUpperLayerVerificationBlock + 2)));
            std::unique_ptr<char[], nn::fs::detail::Deleter> buf
                = nn::fs::detail::MakeUnique<char[]>(sizeBuf);
            if( buf == nullptr )
            {
                return nn::fs::ResultAllocationMemoryFailedInIntegrityVerificationStorageA();
            }
            std::memset(buf.get(), 0, sizeBuf);

            auto sizeRemain = sizeSign;
            while( 0 < sizeRemain )
            {
                const auto sizeFill = static_cast<size_t>(
                    std::min(sizeRemain, static_cast<int64_t>(sizeBuf)));
                NN_RESULT_DO(
                    m_StorageHash.Write(
                        offsetSign + sizeSign - sizeRemain,
                        buf.get(),
                        sizeFill
                    )
                );
                sizeRemain -= sizeFill;
            }
            NN_RESULT_SUCCESS;
        }

    case fs::OperationId::DestroySignature:
        {
            NN_SDK_ASSERT(m_StorageType == fs::StorageType_SaveData);
            int64_t sizeStorageData = 0;
            NN_RESULT_DO(m_StorageData.GetSize(&sizeStorageData));
            if( (offset < 0) || (sizeStorageData < static_cast<int64_t>(offset)) )
            {
                return nn::fs::ResultInvalidOffset();
            }

            const auto offsetSign = (offset >> m_OrderVerificationBlock) * HashSize;
            const auto sizeSign = static_cast<size_t>((std::min(size, sizeStorageData - offset)
                >> m_OrderVerificationBlock) * HashSize);
            std::unique_ptr<char[], nn::fs::detail::Deleter> buf
                = nn::fs::detail::MakeUnique<char[]>(sizeSign);
            NN_RESULT_THROW_UNLESS(buf != nullptr, nn::fs::ResultAllocationMemoryFailedInIntegrityVerificationStorageB());
            NN_RESULT_DO(m_StorageHash.Read(offsetSign, buf.get(), sizeSign));
            for( auto index = 0U; index < sizeSign; ++index )
            {
                buf[index] ^= ((index + 1) % HashSize == 0 ? 0x7f : 0xff);
            }
            return m_StorageHash.Write(offsetSign, buf.get(), sizeSign);
        }

    case fs::OperationId::Invalidate:
        {
            if( m_StorageType == fs::StorageType_SaveData )
            {
                return nn::fs::ResultUnsupportedOperation();
            }
            int64_t sizeStorageData = 0;
            NN_RESULT_DO(m_StorageData.GetSize(&sizeStorageData));
            if( (offset < 0) || (sizeStorageData < offset) )
            {
                return nn::fs::ResultInvalidOffset();
            }

            const auto accessibleSize = std::min(size, sizeStorageData - offset);
            const auto offsetSign = (offset >> m_OrderVerificationBlock) * HashSize;
            const auto sizeSign = (accessibleSize >> m_OrderVerificationBlock) * HashSize;
            NN_RESULT_DO(m_StorageHash.OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offsetSign,
                sizeSign,
                inBuffer,
                inBufferSize));
            NN_RESULT_DO(m_StorageData.OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize));

            NN_RESULT_SUCCESS;
        }

    case nn::fs::OperationId::QueryRange:
        {
            int64_t sizeStorageData = 0;
            NN_RESULT_DO(m_StorageData.GetSize(&sizeStorageData));
            if( (offset < 0) || (sizeStorageData < offset) )
            {
                return nn::fs::ResultInvalidOffset();
            }

            const auto accessibleSize = std::min(size, sizeStorageData - offset);

            NN_RESULT_DO(m_StorageData.OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                accessibleSize,
                inBuffer,
                inBufferSize));

            NN_RESULT_SUCCESS;
        }

    default:
        return nn::fs::ResultUnsupportedOperation();
    }
} // NOLINT(impl/function_size)

/**
* @brief        バッファに対するハッシュを計算します。
*
* @param[out]   outHash     ハッシュブロック
* @param[in]    buffer      ハッシュを計算するデータ
* @param[in]    sizeBlock   ハッシュを計算するブロックサイズ
*/
void IntegrityVerificationStorage::CalcBlockHash(
         BlockHash* outHash,
         const void* buffer,
         size_t sizeBlock
     ) const NN_NOEXCEPT
{
    // ソフトウェアでハッシュ計算します。
    nn::crypto::Sha256Generator generator;
    generator.Initialize();
    if( m_StorageType == fs::StorageType_SaveData )
    {
        generator.Update(m_Salt.value, sizeof(m_Salt));
    }
    generator.Update(buffer, sizeBlock);
    generator.GetHash(outHash, sizeof(BlockHash));

    if( m_StorageType == fs::StorageType_SaveData )
    {
        // セーブデータの場合に最上位ビットを立てます。
        SetValidationBit(outHash);
    }
}

/**
* @brief        指定したオフセットでのハッシュブロックを読み込みます。
*
* @param[out]   outBuf  ハッシュブロック
* @param[in]    bufSize ハッシュサイズ
* @param[in]    offset  ハッシュ検証対象データのオフセット
* @param[in]    size    ハッシュ検証対象データサイズ
*
* @return       関数の処理結果を返します。
* @retval       nn::fs::ResultIntegrityVerificationStorageCorrupted 検証の読み込みに失敗した場合に返します。
*/
Result IntegrityVerificationStorage::ReadBlockSignature(
           void* outBuf,
           size_t bufSize,
           int64_t offset,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outBuf);
    NN_SDK_ASSERT_ALIGNED(offset, static_cast<size_t>(m_SizeVerificationBlock));
    NN_SDK_ASSERT_ALIGNED(size, static_cast<size_t>(m_SizeVerificationBlock));

    NN_UNUSED(bufSize);

    // ハッシュの格納場所を求めます。
    int64_t offsetSignData = (offset >> m_OrderVerificationBlock) * HashSize;
    const auto sizeSignData = static_cast<size_t>((size >> m_OrderVerificationBlock) * HashSize);
    NN_SDK_ASSERT_GREATER_EQUAL(bufSize, sizeSignData);

    // サイズオーバーの確認
    int64_t sizeHash;
    NN_RESULT_DO(m_StorageHash.GetSize(&sizeHash));
    NN_SDK_ASSERT(static_cast<int64_t>(offsetSignData + sizeSignData) <= sizeHash);
    if( static_cast<int64_t>(offsetSignData + sizeSignData) > sizeHash )
    {
        std::memset(outBuf, 0, sizeSignData);
        return nn::fs::ResultOutOfRange();
    }

    // 上位レイヤーからハッシュを読み込みます。
    Result result = m_StorageHash.Read(offsetSignData, outBuf, sizeSignData);
    if( result.IsFailure() )
    {
        std::memset(outBuf, 0, sizeSignData);
        return result;
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したオフセットでのハッシュ検証ブロックを更新します。
*
* @param[in]    buf     ハッシュブロック
* @param[in]    bufSize ハッシュサイズ
* @param[in]    offset  ハッシュ検証対象データのオフセット
* @param[in]    size    ハッシュ検証対象データサイズ
*
* @return       関数の処理結果を返します。
*/
Result IntegrityVerificationStorage::WriteBlockSignature(
           const void* buf,
           size_t bufSize,
           int64_t offset,
           size_t size
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(buf);
    NN_SDK_ASSERT(0 == (offset & (m_SizeVerificationBlock - 1)));

    NN_UNUSED(bufSize);

    // ハッシュの格納場所を求めます。
    const auto offsetSignData = (offset >> m_OrderVerificationBlock) * HashSize;
    const auto sizeSignData = static_cast<size_t>((size >> m_OrderVerificationBlock) * HashSize);
    NN_SDK_ASSERT_GREATER_EQUAL(bufSize, sizeSignData);

    // 上位レイヤーでのハッシュを書き換えます。
    NN_RESULT_DO(m_StorageHash.Write(offsetSignData, buf, sizeSignData));

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したオフセットでのハッシュとデータから計算したハッシュが等しいかどうか検証します。
*
* @param[in]    buf             ハッシュを計算するデータ
* @param[in]    pHash           ハッシュ
*
* @return       関数の処理結果を返します。
* @retval       nn::fs::ResultIntegrityVerificationStorageCorrupted 検証に失敗した場合に返します。
*/
Result IntegrityVerificationStorage::VerifyHash(
           const void* buf,
           BlockHash* pHash
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buf);
    NN_SDK_REQUIRES_NOT_NULL(pHash);

    auto& s1 = *pHash;

    // 未書き込み領域かどうか判定します。
    if( m_StorageType == fs::StorageType_SaveData )
    {
        bool isCleared = false;
        NN_RESULT_DO(IsCleared(&isCleared, s1));
        if( isCleared )
        {
            return nn::fs::ResultClearedRealDataVerificationFailed();
        }
    }

    BlockHash s2;
    CalcBlockHash(&s2, buf);

    // ハッシュが正しいかどうか比較を行います。
    if( !nn::crypto::IsSameBytes(&s1, &s2, sizeof(s1)) )
    {
        // 検証失敗
        // スタック攻撃避けのため、正解値はつぶしておきます。
        std::memset(&s1, 0, sizeof(s1));

        // ユーザーから預かったバッファ上にて検証失敗が発生したか、
        // FileServer内のヒープ上で発生したかが区別できるように
        // 実データに対するハッシュ検証エラーは読み替えます。
        if( m_IsRealDataSign )
        {
            return nn::fs::ResultUnclearedRealDataVerificationFailed();
        }
        else
        {
            return nn::fs::ResultNonRealDataVerificationFailed();
        }
    }

    NN_RESULT_SUCCESS;
}

/**
* @brief        ハッシュがゼロクリアされているかどうか判定します。
*               有効ビットを使用しない場合には常に成功し false が格納されます。
*
* @param[out]   outCleared      ゼロクリアされていれば true、
*                               有効ビットが立っていれば false が格納されます。
* @param[in]    hash            判定するハッシュを指定します。
*
* @return       関数の処理結果を返します。
* @retval       ResultSuccess ハッシュはゼロクリアされているか、または有効ビットが立っています。
* @retval       ResultInvalidZeroHash
*                   ハッシュは有効ビットが立っていないにもかかわらずゼロクリアされていません。
*/
Result IntegrityVerificationStorage::IsCleared(
           bool *isCleared,
           const BlockHash& hash
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(isCleared);
    NN_SDK_ASSERT(m_StorageType == fs::StorageType_SaveData);

    *isCleared = false;

    if( IsValidationBit(&hash) )
    {
        NN_RESULT_SUCCESS;
    }
    else
    {
        // 最上位ビットがゼロなら他のビットもゼロのはず
        // さもなければ検証エラー
        for( size_t hashIndex = 0; hashIndex < sizeof(hash.hash); ++hashIndex )
        {
            if( hash.hash[hashIndex] != 0 )
            {
                return nn::fs::ResultInvalidZeroHash();
            }
        }

        *isCleared = true;
        NN_RESULT_SUCCESS;
    }
}

}}}

