﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_IExclusiveFileSystem.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief        ファイルを作成します。
*
* @param[in]    path    ファイルパス
* @param[in]    size    ファイルサイズ
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::CreateFile(
           const Path& path,
           int64_t size
       ) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認は不要です。

    // 実際にファイルを作成します。
    return CreateFileImpl(path, size);
}

/**
* @brief        ディレクトリを作成します。
*
* @param[in]    path ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::CreateDirectory(
           const Path& path
       ) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認は不要です。

    // 実際にディレクトリを作成します。
    return CreateDirectoryImpl(path);
}

/**
* @brief        ファイルを開きます。
*
* @param[out]   outValue    ファイルオブジェクト
* @param[in]    path        ファイルパス
* @param[in]    mode        オープンモード
*
* @return       関数の処理結果を返します。
*
* @details      ファイルオブジェクト用にメモリが確保されます。
*               確保されたメモリは @ref IExclusiveFileSystem::CloseFile
*               を用いて解放する必要があります。
*/
Result IExclusiveFileSystem::OpenFile(
           IFile** outValue,
           const Path& path,
           int mode
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    Result result;

    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認
    FileOrDirectoryId id;
    NN_RESULT_DO(
        CheckAccessPolicy(
            &id,
            path,
            mode,
            true,           // ファイル
            false           // 非削除
        )
    );

    // 実際にファイルを開きます。
    ExclusiveFile *pFile;
    NN_RESULT_DO(OpenFileImpl(&pFile, path, mode));
    *outValue = pFile;

    if( id < 0 )
    {
        result = GetFileIdFromPath(&id, path, true);
        NN_SDK_ASSERT(result.IsSuccess() || fs::ResultBufferAllocationFailed::Includes(result));
        if( result.IsFailure() )
        {
            CloseFileImpl(pFile);
            NN_RESULT_THROW(result);
        }
    }

    // ファイル情報を設定して、制御リストに追加します。
    pFile->SetMode(mode);
    NN_SDK_ASSERT(id > 0);
    pFile->SetId(id);
    LinkObject(pFile);
    NN_SDK_ASSERT(pFile->IsFile());

    NN_RESULT_SUCCESS;
}

/**
* @brief        ディレクトリを開きます。
*
* @param[out]   outValue    ディレクトリオブジェクト
* @param[in]    path        ディレクトリパス
*
* @return       関数の処理結果を返します。
*
* @details      ディレクトリオブジェクト用にメモリが確保されます。
*               確保されたメモリは @ref IExclusiveFileSystem::CloseDirectory
*               を用いて解放する必要があります。
*/
Result IExclusiveFileSystem::OpenDirectory(
           IDirectory** outValue,
           const Path& path
       ) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outValue);

    Result result;

    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御は不要。

    // 実際にディレクトリを開きます。
    ExclusiveDirectory *pDirectory;
    NN_RESULT_DO(OpenDirectoryImpl(&pDirectory, path));

    // ファイル情報を設定して、排他制御リストに追加します。
    *outValue = pDirectory;
    FileOrDirectoryId id;
    result = GetFileIdFromPath(&id, path, false);
    NN_SDK_ASSERT(result.IsSuccess() || fs::ResultBufferAllocationFailed::Includes(result));
    if( result.IsFailure() )
    {
        CloseDirectoryImpl(pDirectory);
        NN_RESULT_THROW(result);
    }

    NN_SDK_ASSERT(id > 0);
    pDirectory->SetId(id);
    LinkObject(pDirectory);
    NN_SDK_ASSERT(!pDirectory->IsFile());

    NN_RESULT_SUCCESS;
}

/**
* @brief        ファイルオブジェクト用リソースを解放します。
*
* @param[in]    pFile   ファイルオブジェクト
*/
void IExclusiveFileSystem::CloseFile(IFile* pFile) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pFile);

    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御を解除します。
    UnlinkObject(static_cast<ExclusiveFile*>(pFile));
    CloseFileImpl(static_cast<ExclusiveFile*>(pFile));
}

/**
* @brief        ディレクトリオブジェクト用リソースを解放します。
*
* @param[in]    pDirectory  ディレクトリオブジェクト
*/
void IExclusiveFileSystem::CloseDirectory(IDirectory* pDirectory) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDirectory);

    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御を解除します。
    UnlinkObject(static_cast<ExclusiveDirectory*>(pDirectory));
    CloseDirectoryImpl(static_cast<ExclusiveDirectory*>(pDirectory));
}

/**
* @brief        ファイルを削除します。
*
* @param[in]    path    ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::DeleteFile(const Path& path) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認
    FileOrDirectoryId id;
    NN_RESULT_DO(
        CheckAccessPolicy(
            &id,
            path,
            nn::fs::OpenMode_Write, // 書き込みモード
            true,                    // ファイル
            true                     // 削除
        )
    );

    // 実際にファイルを削除します。
    return DeleteFileImpl(path);
}

/**
* @brief        ディレクトリを削除します。
*
* @param[in]    path    ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::DeleteDirectory(const Path& path) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認
    FileOrDirectoryId id;
    auto opened = false;
    const auto result = CheckAccessPolicy(
                            &id,
                            &opened,
                            path,
                            nn::fs::OpenMode_Write,   // 書き込みモード
                            false,                    // 非ファイル
                            true                      // 削除
                        );
    NN_RESULT_TRY(result)
        NN_RESULT_CATCH(nn::fs::ResultPermissionDenied)
        {
            if( !opened )
            {
                return nn::fs::ResultDirectoryNotEmpty();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    // 実際にディレクトリを削除します。
    return DeleteDirectoryImpl(path);
}

/**
* @brief        ファイルをリネームします。
*
* @param[in]    oldPath 変更対象のファイルパス
* @param[in]    newPath 変更後のファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::RenameFile(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認
    FileOrDirectoryId id;

    const auto oldPathResult = CheckAccessPolicy(
                                   &id,
                                   oldPath,
                                   nn::fs::OpenMode_Write,
                                   true,    // ファイル
                                   true     // 削除
                               );
    NN_RESULT_TRY(oldPathResult)
        NN_RESULT_CATCH(nn::fs::ResultPermissionDenied)
        {
            if( Path::IsEqualPath(oldPath, newPath) )
            {
                NN_RESULT_SUCCESS;
            }
            else if( CheckEntryExistence(newPath) )
            {
                return nn::fs::ResultPathAlreadyExists();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    const auto newPathResult = CheckAccessPolicy(
                                   &id,
                                   newPath,
                                   nn::fs::OpenMode_Write,
                                   true,    // ファイル
                                   false    // 削除
                               );
    NN_RESULT_TRY(newPathResult)
        NN_RESULT_CATCH(nn::fs::ResultPermissionDenied)
        {
            if( CheckEntryExistence(newPath) )
            {
                return nn::fs::ResultPathAlreadyExists();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    // 実際にリネームを行います。
    NN_RESULT_TRY(RenameFileImpl(oldPath, newPath))
        NN_RESULT_CATCH(nn::fs::ResultAlreadyExists)
        {
            if( Path::IsEqualPath(oldPath, newPath) )
            {
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

/**
* @brief        ディレクトリをリネームします。
*
* @param[in]    oldPath 変更対象のディレクトリパス
* @param[in]    newPath 変更後のディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::RenameDirectory(
           const Path& oldPath,
           const Path& newPath
       ) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 排他制御の確認
    FileOrDirectoryId id;

    const auto oldPathResult = CheckAccessPolicy(
                                   &id,
                                   oldPath,
                                   nn::fs::OpenMode_Write,
                                   false,      // 非ファイル
                                   true        // 削除
                               );
    NN_RESULT_TRY(oldPathResult)
        NN_RESULT_CATCH(nn::fs::ResultPermissionDenied)
        {
            if( Path::IsEqualPath(oldPath, newPath) )
            {
                NN_RESULT_SUCCESS;
            }
            else if( CheckEntryExistence(newPath) )
            {
                return nn::fs::ResultPathAlreadyExists();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    const auto newPathResult = CheckAccessPolicy(
                                   &id,
                                   newPath,
                                   nn::fs::OpenMode_Write,
                                   false,        // 非ファイル
                                   false         // 削除
                               );
    NN_RESULT_TRY(newPathResult)
        NN_RESULT_CATCH(nn::fs::ResultPermissionDenied)
        {
            if( CheckEntryExistence(newPath) )
            {
                return nn::fs::ResultPathAlreadyExists();
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    // 実際にリネームを行います。
    NN_RESULT_TRY(RenameDirectoryImpl(oldPath, newPath))
        NN_RESULT_CATCH(nn::fs::ResultAlreadyExists)
        {
            if( Path::IsEqualPath(oldPath, newPath) )
            {
                NN_RESULT_SUCCESS;
            }
            else
            {
                NN_RESULT_RETHROW;
            }
        }
    NN_RESULT_END_TRY

    NN_RESULT_SUCCESS;
}

/**
* @brief        指定したファイルが存在しているかどうかを取得します。
*
* @param[out]   outValue    ファイルが存在しているかどうか
* @param[in]    path        ファイルパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::HasFile(bool* outValue, const Path& path) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 実際に取得します。
    return HasFileImpl(outValue, path);
}

/**
* @brief        指定したディレクトリが存在しているかどうかを取得します。
*
* @param[out]   outValue    ディレクトリが存在しているかどうか
* @param[in]    path        ディレクトリパス
*
* @return       関数の処理結果を返します。
*/
Result IExclusiveFileSystem::HasDirectory(bool* outValue, const Path& path) NN_NOEXCEPT
{
    // ファイル操作中はロックします。
    ScopedFSLock lock(GetFileOperationLocker());

    // 実際に取得します。
    return HasDirectoryImpl(outValue, path);
}

/**
* @brief        このモードでアクセス可能かどうかを診断します。
*
* @param[out]   outId               ID
* @param[out]   outDirectoryOpened  オープンされているかどうか
* @param[in]    path                パス
* @param[in]    mode                オープンモード
* @param[in]    isFile              ファイルかどうか
* @param[in]    isDelete            ファイル削除かどうか
*
* @return       関数の処理結果を返します。
*
* @details      outDirectoryOpened は対象がディレクトリの場合のみ有効な値が格納されます。
*/
Result IExclusiveFileSystem::CheckAccessPolicy(
           int64_t *outId,
           bool* outDirectoryOpened,
           const Path& path,
           int mode,
           bool isFile,
           bool isDelete
       ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outId);
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryOpened);

    int64_t id;
    {
        Result result = GetFileIdFromPath(&id, path, isFile);
        if( result.IsFailure() )
        {
            // "見つからない" idに書き換えます
            *outId = -1;
            return ResultSuccess();
        }
    }
    *outId = id;

    *outDirectoryOpened = false;

    nn::Result result = nn::ResultSuccess();
    IExclusiveObject* pObj = m_pOpened;
    while( pObj != nullptr )
    {
        if( isFile )
        {
            if( pObj->IsFile() )
            {
                // 既に開いているファイル対し、何かしようとした場合、
                // 組み合わせに応じてアクセス拒否を発行します。
                if( pObj->GetId() == id )
                {
                    if( (pObj->GetMode() & nn::fs::OpenMode_Write) != 0 )
                    {
                        return nn::fs::ResultPermissionDenied();
                    }
                    if( (mode & nn::fs::OpenMode_Write) != 0 )
                    {
                        return nn::fs::ResultPermissionDenied();
                    }
                }
            }
            else
            {
                NN_SDK_ASSERT(!pObj->IsFile());
                if( isDelete )
                {
                    NN_RESULT_DO(pObj->NotifyDelete(id, isFile));
                }
            }
        }
        else
        {
            // このディレクトリをリネーム、削除しようとしている場合
            if( (mode & nn::fs::OpenMode_Write) != 0 )
            {
                // 子ファイルが存在している場合削除できません。
                if( !pObj->IsFile() )
                {
                    if( pObj->GetId() == id )
                    {
                        // deleteのみ対象とします。
                        *outDirectoryOpened = true;
                        return nn::fs::ResultPermissionDenied();
                    }
                }
            }
            if( isDelete )
            {
                // 開いているエントリがディレクトリの子でないかチェック。
                {
                    bool isSubEntry = false;
                    NN_RESULT_DO(
                        CheckSubEntry(
                            &isSubEntry,
                            static_cast<StorageIndex>(id),
                            static_cast<StorageIndex>(pObj->GetId()),
                            pObj->IsFile()
                        )
                    );
                    if( isSubEntry )
                    {
                        result = nn::fs::ResultPermissionDenied();
                    }
                }
                // ディレクトリを削除しようとしました。
                if( result.IsSuccess() && !pObj->IsFile() )
                {
                    // 列挙の整合性を調整します。
                    NN_RESULT_DO(pObj->NotifyDelete(id, isFile));
                }
            }
        }
        pObj = pObj->GetNext();
    }

    return result;
}

/**
* @brief        エントリが存在するかどうかをチェックします。
*
* @param[in]    path    パス
*
* @return       エントリが存在するかどうかを返します。
*/
bool IExclusiveFileSystem::CheckEntryExistence(
         const Path& path
     ) NN_NOEXCEPT
{
    {
        auto exists = false;
        const auto result = HasFile(&exists, path);
        if( result.IsSuccess() && exists )
        {
            return true;
        }
    }
    {
        auto exists = false;
        const auto result = HasDirectory(&exists, path);
        if( result.IsSuccess() && exists )
        {
            return true;
        }
    }
    return false;
}

/**
* @brief        開いているオブジェクトとしてリンクします。
*
* @param[in]    pObj    オブジェクト
*/
void IExclusiveFileSystem::LinkObject(IExclusiveObject* pObj) NN_NOEXCEPT
{
    NN_SDK_ASSERT(nullptr == pObj->GetNext());
    pObj->SetNext(m_pOpened);
    m_pOpened = pObj;
}

/**
* @brief 開いていたオブジェクトへのリンクを解除します。
*
* @param[in] pObj オブジェクト
*/
void IExclusiveFileSystem::UnlinkObject(IExclusiveObject* pObj) NN_NOEXCEPT
{
    IExclusiveObject* pOpenedObj = m_pOpened;
    IExclusiveObject* pPrevObj = nullptr;
    while( pOpenedObj != nullptr )
    {
        if( pOpenedObj == pObj )
        {
            NN_SDK_ASSERT((pOpenedObj->IsFile() && pObj->IsFile()) || ((!pOpenedObj->IsFile()) && (!pObj->IsFile())));
            if( pPrevObj == nullptr )
            {
                m_pOpened = pObj->GetNext();
            }
            else
            {
                pPrevObj->SetNext(pObj->GetNext());
            }

            return;
        }
        pPrevObj = pOpenedObj;
        pOpenedObj = pOpenedObj->GetNext();
    }

    // このアーカイブに関連付けされていません。
    NN_SDK_ASSERT(false);
}

}}}

