﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fs/fs_Result.h>
#include <nn/fssystem/save/fs_DuplexBitmapHolder.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief        コンストラクタ
*/
DuplexBitmapHolder::DuplexBitmapHolder() NN_NOEXCEPT
{
}

/**
* @brief        二重化選択ビットマップをフォーマットします。
*
* @param[in]    blockCount          総ブロック数
* @param[in]    pTable              メタデータ用ストレージ
* @param[in]    offsetBytes         メタデータ用ストレージに割り当てる領域のオフセット
* @param[in]    pOriginalTable      オリジナルメタデータ用ストレージ
* @param[in]    originalOffsetBytes オリジナルメタデータ用ストレージに割り当てる領域のオフセット
*
* @return       関数の処理結果を返します。
*/
Result DuplexBitmapHolder::Format(
           const uint32_t blockCount,
           fs::SubStorage storage,
           fs::SubStorage storageOriginal
       ) NN_NOEXCEPT
{
    int64_t sizeBytes = QuerySize(blockCount);
    fs::SubStorage subStorage(&storage, 0, sizeBytes);
    fs::SubStorage subStorageOriginal(&storageOriginal, 0, sizeBytes);
    return DuplexBitmap::Format(blockCount, subStorage, subStorageOriginal);
}

/**
* @brief        二重化選択ビットマップを拡張します。
*
* @param[in]    blockCountOld       拡張前の総ブロック数
* @param[in]    blockCountNew       拡張後の総ブロック数
* @param[in]    storage             メタデータ用ストレージ
* @param[in]    storageOriginal     オリジナルメタデータ用ストレージ
*
* @return       関数の処理結果を返します。
*/
Result DuplexBitmapHolder::Expand(
           uint32_t blockCountOld,
           uint32_t blockCountNew,
           fs::SubStorage storage,
           fs::SubStorage storageOriginal
       ) NN_NOEXCEPT
{
    const int64_t sizeBytes = QuerySize(blockCountNew);
    fs::SubStorage subStorage(&storage, 0, sizeBytes);
    fs::SubStorage subStorageOriginal(&storageOriginal, 0, sizeBytes);
    return DuplexBitmap::Expand(blockCountOld, blockCountNew, subStorage, subStorageOriginal);
}

/**
* @brief        二重化選択ビットマップをマウントします。
*
* @param[in]    blockCount          総ブロック数
* @param[in]    pFile               メタデータ用ファイル
* @param[in]    offsetBytes         メタデータ用ファイルに割り当てる領域のオフセット
* @param[in]    pOriginalFile       オリジナルメタデータ用ファイル
* @param[in]    originalOffsetBytes オリジナルメタデータ用ファイルに割り当てる領域のオフセット
*
* @return       関数の処理結果を返します。
*/
void DuplexBitmapHolder::Initialize(
         const uint32_t blockCount,
         fs::SubStorage storage,
         fs::SubStorage storageOriginal
     ) NN_NOEXCEPT
{
    int64_t sizeBytes = QuerySize(blockCount);
    m_CountBlock = blockCount;
    m_Storage = fs::SubStorage(&storage, 0, sizeBytes);
    m_StorageOriginal = fs::SubStorage(&storageOriginal, 0, sizeBytes);
    DuplexBitmap::Initialize(blockCount, m_Storage, m_StorageOriginal);
}

/**
* @brief        二重化選択ビットマップを読み込み扱いでマウントします。
*
* @param[in]    blockCount          総ブロック数
* @param[in]    pFile               メタデータ用ファイル
* @param[in]    offsetBytes         メタデータ用ファイルに割り当てる領域のオフセット
* @param[in]    pOriginalFile       オリジナルメタデータ用ファイル
* @param[in]    originalOffsetBytes オリジナルメタデータ用ファイルに割り当てる領域のオフセット
*
* @return 関数の処理結果を返します。
*/
void DuplexBitmapHolder::InitializeForRead(
         uint32_t blockCount,
         fs::SubStorage storage,
         fs::SubStorage storageOriginal
     ) NN_NOEXCEPT
{
    m_CountBlock = blockCount;
    int64_t sizeBytes = QuerySize(m_CountBlock);
    m_Storage = fs::SubStorage(&storage, 0, sizeBytes);
    m_StorageOriginal = fs::SubStorage(&storageOriginal, 0, sizeBytes);
    DuplexBitmap::Initialize(m_CountBlock, m_StorageOriginal, m_StorageOriginal);
}

/**
* @brief        二重化選択ビットマップの内部データを交換します。
*
* @param[out]   pOutBitmap  "更新系"のビットマップ
*
* @details      これは更新系ビットマップと、参照系ビットマップを入れ替えする際に使用します。
*               HierarchicalDuplexFile の実装に強く依存するため、他からこの関数を呼ぶべきではありません。
*/
void DuplexBitmapHolder::SwapDuplexBitmapForHierarchicalDuplexStorage(
         DuplexBitmapHolder* pOutBitmap
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutBitmap);

    uint32_t countBlock = GetBlockCount();
    NN_SDK_ASSERT(this->GetBlockCount() == pOutBitmap->GetBlockCount());

    fs::SubStorage storageUpdate = pOutBitmap->GetUpdateStorage();
    fs::SubStorage storageOriginal = pOutBitmap->GetOriginalStorage();

    // 入れ替えつつ再度初期化します
    // (更新系)
    pOutBitmap->Finalize();
    pOutBitmap->InitializeForRead(countBlock, storageOriginal, storageUpdate);

    // (参照系)
    this->Finalize();
    this->Initialize(countBlock, storageUpdate, storageUpdate);
}

}}}

