﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkLog.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include "fs_LruListCache.h"

namespace nn { namespace fssystem {

// ReadOnly な簡易キャッシュレイヤ
class ReadOnlyBlockCacheStorage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(ReadOnlyBlockCacheStorage);

public:

    ReadOnlyBlockCacheStorage(IStorage* pBaseStorage, int blockSize, char* buffer, size_t bufferSize, int cacheBlockCount) NN_NOEXCEPT
        : m_pBaseStorage(pBaseStorage),
          m_BlockSize(blockSize),
          m_Mutex(false)
    {
        NN_SDK_REQUIRES(bufferSize >= static_cast<size_t>(m_BlockSize));
        NN_SDK_REQUIRES((blockSize & (blockSize - 1)) == 0, "blockSize must be power of 2.");
        NN_SDK_REQUIRES(cacheBlockCount > 0);
        NN_SDK_REQUIRES(bufferSize >= static_cast<size_t>(blockSize * cacheBlockCount));
        NN_UNUSED(bufferSize);

        for(int i = 0; i < cacheBlockCount; ++i)
        {
            std::unique_ptr<BlockCache::Node> pNode(new BlockCache::Node(buffer + m_BlockSize * i));

            // メモリ不足で pNode が作成できなかった場合は何もしない
            NN_SDK_ASSERT_NOT_NULL(pNode);
            if(pNode != nullptr)
            {
                m_BlockCache.PushMruNode(std::move(pNode), -1);
            }
        }
    }

    ~ReadOnlyBlockCacheStorage() NN_NOEXCEPT
    {
        m_BlockCache.DeleteAllNodes();
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        // m_BlockSize 単位アクセスしかこない前提
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, m_BlockSize));
        NN_SDK_REQUIRES(nn::util::is_aligned(size,   m_BlockSize));

        // ブロック単体読み込み時のみ、キャッシュ参照・更新を行う
        // (BufferAlignmentMatchingStorage の head/tail を意図)
        if( size == static_cast<size_t>(m_BlockSize) )
        {
            char* cachedBuffer = nullptr;

            {
                std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
                bool found = m_BlockCache.FindValueAndUpdateMru(&cachedBuffer, offset / m_BlockSize);
                if( found )
                {
                    // hit
                    std::memcpy(buffer, cachedBuffer, size);
                    NN_RESULT_SUCCESS;
                }
            }

            // mishit
            NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, size));

            // add as mru
            {
                std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

                auto pLru = m_BlockCache.PopLruNode();

                std::memcpy(pLru->value, buffer, m_BlockSize);

                m_BlockCache.PushMruNode(std::move(pLru), offset / m_BlockSize);
            }
            NN_RESULT_SUCCESS;
        }
        else
        {
            return m_pBaseStorage->Read(offset, buffer, size);
        }
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_LOG("ReadOnlyBlockCacheStorage doesn't support Write().");
        NN_UNUSED(offset);
        NN_UNUSED(buffer);
        NN_UNUSED(size);
        return nn::fs::ResultUnsupportedOperation();
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->GetSize(outValue);
    }

    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        // m_BlockSize 単位アクセスしかこない前提
        NN_SDK_REQUIRES(nn::util::is_aligned(offset, m_BlockSize));
        NN_SDK_REQUIRES(nn::util::is_aligned(size,   m_BlockSize));

        if( operationId == fs::OperationId::Invalidate )
        {
            NN_RESULT_THROW_UNLESS(0 <= offset, nn::fs::ResultInvalidOffset());

            std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

            const auto cacheBlockCount = m_BlockCache.GetSize();
            BlockCache validCache;

            for( auto count = 0; count < cacheBlockCount; ++count )
            {
                auto pLru = m_BlockCache.PopLruNode();
                if( offset <= pLru->key && pLru->key < offset + size )
                {
                    m_BlockCache.PushMruNode(std::move(pLru), -1);
                }
                else
                {
                    validCache.PushMruNode(std::move(pLru), pLru->key);
                }
            }

            while( !validCache.IsEmpty() )
            {
                auto pLru = validCache.PopLruNode();
                m_BlockCache.PushMruNode(std::move(pLru), pLru->key);
            }
        }

        return m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize);
    }

private:
    IStorage* const m_pBaseStorage;
    int m_BlockSize;

    typedef LruListCache<int64_t, char*> BlockCache;
    BlockCache m_BlockCache;

    os::Mutex m_Mutex; // キャッシュへのアクセス排他

};


}}
