﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <algorithm>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fs_Base.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_Exchange.h>
#include <nn/crypto/crypto_Compare.h>

#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IDirectory.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fssystem/fs_PartitionFileSystemMeta.h>

#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_Assert.h>

using namespace nn::fs;

namespace nn { namespace fssystem {

namespace {
    class DefaultAllocatorForPartitionFileSystem : public MemoryResource
    {
    private:
        void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(alignment);
            return nn::fs::detail::Allocate(bytes);
        }

        void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT
        {
            NN_UNUSED(alignment);
            nn::fs::detail::Deallocate(p, bytes);
        }

        virtual bool do_is_equal(const MemoryResource& other) const NN_NOEXCEPT
        {
            return this == &other;
        }
    };

    DefaultAllocatorForPartitionFileSystem g_DefaultAllocatorForPartitionFileSystem;
}


    template <typename MetaData>
    class PartitionFileSystemCore<MetaData>::PartitionFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
    {
    protected:
        const typename MetaData::PartitionEntry* m_pEntry;
        const PartitionFileSystemCore<MetaData>* m_pParent;
        const nn::fs::OpenMode m_Mode;

    public:
        PartitionFile(const PartitionFileSystemCore<MetaData>* pParent, const typename MetaData::PartitionEntry* pEntry, nn::fs::OpenMode mode) NN_NOEXCEPT
            : m_pEntry(pEntry)
            , m_pParent(pParent)
            , m_Mode(mode)
        {
        }

        virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;

        virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(option);
            NN_FSP_REQUIRES( (m_Mode & nn::fs::OpenMode_Write) != 0, ResultInvalidOperationForOpenMode() );

            NN_RESULT_THROW_UNLESS( 0 <= offset && offset <= static_cast<int64_t>(m_pEntry->size), ResultOutOfRange() );

            NN_FSP_REQUIRES( static_cast<int64_t>(offset + size) <= static_cast<int64_t>(m_pEntry->size) && static_cast<int64_t>(offset + size) >= offset, ResultInvalidSize() );
            NN_RESULT_DO(m_pParent->m_pBaseFile->Write(m_pParent->m_MetaDataSize + m_pEntry->offset + offset, buffer, size));
            NN_RESULT_SUCCESS;
        }

        virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE
        {
            if( (m_Mode & nn::fs::OpenMode_Write) == 0 )
            {
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_DO(m_pParent->m_pBaseFile->Flush());
            NN_RESULT_SUCCESS;
        }

        virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(size);
            NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "SetSize() is not supported.");
        }

        virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *outValue = m_pEntry->size;
            NN_RESULT_SUCCESS;
        }

        virtual Result DoOperateRange(
            void* outBuffer,
            size_t outBufferSize,
            OperationId operationId,
            int64_t offset,
            int64_t size,
            const void* inBuffer,
            size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            switch( operationId )
            {
            case OperationId::Invalidate:
                NN_FSP_REQUIRES(
                    (m_Mode & nn::fs::OpenMode_Write) == 0,
                    ResultInvalidOperationForOpenMode());
                break;

            case OperationId::QueryRange:
                break;

            default:
                return nn::fs::ResultUnsupportedOperation();
            }

            NN_RESULT_THROW_UNLESS(
                0 <= offset && offset <= static_cast<int64_t>(m_pEntry->size),
                ResultOutOfRange());
            NN_FSP_REQUIRES(
                static_cast<int64_t>(offset + size) <= static_cast<int64_t>(m_pEntry->size)
                    && static_cast<int64_t>(offset + size) >= offset,
                ResultInvalidSize());
            NN_RESULT_DO(m_pParent->m_pBaseFile->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                m_pParent->m_MetaDataSize + m_pEntry->offset + offset,
                size,
                inBuffer,
                inBufferSize));
            NN_RESULT_SUCCESS;
        }
    };

    template <>
    Result PartitionFileSystemCore<Sha256PartitionFileSystemMeta>::PartitionFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
    {
        size_t readSize = 0;
        NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));

        int64_t entryStart = m_pParent->m_MetaDataSize + m_pEntry->offset;
        int64_t readEnd = static_cast<int64_t>(offset + readSize);
        int64_t hashTargetEnd = static_cast<int64_t>(m_pEntry->hashTargetOffset) + m_pEntry->hashTargetSize;

        if( !(readEnd <= static_cast<int64_t>(m_pEntry->hashTargetOffset) || hashTargetEnd <= offset) )
        {
            // ハッシュ対象領域の読み込み

            // TODO: HashTargetOffset != 0 に対応
            NN_FSP_REQUIRES(m_pEntry->hashTargetOffset == 0, ResultInvalidSha256PartitionHashTarget());

            const size_t HashTargetBufferSize = 512;
            NN_FSP_REQUIRES(m_pEntry->hashTargetOffset + m_pEntry->hashTargetSize <= m_pEntry->size, ResultInvalidSha256PartitionHashTarget());

            int64_t readOffset = entryStart + offset;
            NN_FSP_REQUIRES(readOffset >= offset, ResultOutOfRange());

            char hash[crypto::Sha256Generator::HashSize];
            crypto::Sha256Generator sha;

            if( (offset <= static_cast<int64_t>(m_pEntry->hashTargetOffset)) && (hashTargetEnd <= readEnd) )
            {
                // ハッシュ対象全域を含む読み込み
                // ユーザーバッファを使ってハッシュ検証
                sha.Initialize();
                NN_RESULT_DO(m_pParent->m_pBaseFile->Read(readOffset, static_cast<char*>(buffer), readSize));
                sha.Update(static_cast<char*>(buffer) + (static_cast<int64_t>(m_pEntry->hashTargetOffset) - offset), m_pEntry->hashTargetSize);
                sha.GetHash(hash, sizeof(hash));
            }
            else if( (static_cast<int64_t>(m_pEntry->hashTargetOffset) <= offset) && (readEnd <= hashTargetEnd) )
            {
                // ハッシュ対象内に収まる読み込み
                // 内部バッファを使って検証
                int64_t hashRestSize = m_pEntry->hashTargetSize;
                int64_t hashOffset = entryStart + m_pEntry->hashTargetOffset;
                int64_t restSize = readSize;
                int64_t copyOffset = 0;
                sha.Initialize();
                while( hashRestSize > 0 )
                {
                    char bufferForHashTarget[HashTargetBufferSize];
                    size_t readableSize = static_cast<size_t>(std::min<int64_t>(HashTargetBufferSize, hashRestSize));
                    NN_RESULT_DO(m_pParent->m_pBaseFile->Read(hashOffset, bufferForHashTarget, readableSize));
                    sha.Update(bufferForHashTarget, readableSize);

                    // ユーザーバッファに必要なデータをコピー
                    if( (readOffset <= (hashOffset + static_cast<int64_t>(readableSize))) && restSize > 0 )
                    {
                        int64_t bufferForHashTargetOffset = std::max<int64_t>(readOffset - hashOffset, 0);
                        size_t copySize = static_cast<size_t>(std::min<int64_t>((readableSize - bufferForHashTargetOffset), restSize));
                        std::memcpy(static_cast<char*>(buffer) + copyOffset, bufferForHashTarget + bufferForHashTargetOffset, copySize);
                        restSize -= copySize;
                        copyOffset += copySize;
                    }

                    hashRestSize -= readableSize;
                    hashOffset   += readableSize;
                }
                sha.GetHash(hash, sizeof(hash));
            }
            else
            {
                // ハッシュ対象領域を部分的に含む読み込みは未サポート
                return ResultInvalidSha256PartitionHashTarget();
            }

            if( !nn::crypto::IsSameBytes(m_pEntry->hash, hash, sizeof(hash)) )
            {
                // 検証失敗時は読み出したデータを無効化してエラーを返す
                std::memset(reinterpret_cast<char*>(buffer), 0x0, readSize);
                return ResultSha256PartitionHashVerificationFailed();
            }
        }
        else
        {
            NN_RESULT_DO(m_pParent->m_pBaseFile->Read(m_pParent->m_MetaDataSize + m_pEntry->offset + offset, buffer, readSize));
        }
        *outValue = readSize;

        NN_RESULT_SUCCESS;
    }


    template <>
    Result PartitionFileSystemCore<PartitionFileSystemMeta>::PartitionFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
    {
        size_t readSize = 0;
        NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));

        NN_RESULT_DO(m_pParent->m_pBaseFile->Read(m_pParent->m_MetaDataSize + m_pEntry->offset + offset, buffer, readSize));
        *outValue = readSize;

        NN_RESULT_SUCCESS;
    }



    template<typename MetaData>
    class PartitionFileSystemCore<MetaData>::PartitionDirectory : public nn::fs::fsa::IDirectory, public nn::fs::detail::Newable
    {
    private:
        uint32_t m_CurrentIndex;
        const PartitionFileSystemCore<MetaData>* m_pParent;
        const nn::fs::OpenDirectoryMode m_Mode;

    public:
        PartitionDirectory(const PartitionFileSystemCore<MetaData>* pParent, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
            : m_CurrentIndex(0)
            , m_pParent(pParent)
            , m_Mode(mode)
        {
        }

        virtual Result DoRead(int64_t *outValue, nn::fs::DirectoryEntry *entryEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE
        {
            if( (m_Mode & nn::fs::OpenDirectoryMode_File) != 0 )
            {
                int64_t restNumber = m_pParent->m_pMetaData->GetEntryCount() - m_CurrentIndex;
                int64_t readNumber = (restNumber < count) ? restNumber : count;
                for(auto i = 0; i < readNumber; ++i, ++m_CurrentIndex)
                {
                    auto pEntry = m_pParent->m_pMetaData->GetEntry(m_CurrentIndex);
                    entryEntries[i].directoryEntryType = nn::fs::DirectoryEntryType_File;
                    entryEntries[i].fileSize = pEntry->size;
                    strncpy(entryEntries[i].name, m_pParent->m_pMetaData->GetEntryName(m_CurrentIndex), sizeof(entryEntries[i].name) - 1);
                    entryEntries[i].name[sizeof(entryEntries[i].name) - 1] = '\0';
                }

                *outValue = readNumber;
                NN_RESULT_SUCCESS;
            }
            else
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }
        }

        virtual Result DoGetEntryCount(int64_t *outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            if( (m_Mode & nn::fs::OpenDirectoryMode_File) != 0 )
            {
                *outValue = m_pParent->m_pMetaData->GetEntryCount();
                NN_RESULT_SUCCESS;
            }
            else
            {
                *outValue = 0;
                NN_RESULT_SUCCESS;
            }
        }
    };


    template <typename MetaData>
    PartitionFileSystemCore<MetaData>::PartitionFileSystemCore() NN_NOEXCEPT
        : m_IsInitialized(false)
    {
    }

    template <typename MetaData>
    PartitionFileSystemCore<MetaData>::~PartitionFileSystemCore() NN_NOEXCEPT
    {
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::Initialize(std::shared_ptr<fs::IStorage> pBaseFile) NN_NOEXCEPT
    {
        m_pSharedStorage = std::move(pBaseFile);
        return Initialize(m_pSharedStorage.get());
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::Initialize(std::shared_ptr<fs::IStorage> pBaseStorage, MemoryResource* pAllocator) NN_NOEXCEPT
    {
        m_pSharedStorage = std::move(pBaseStorage);
        return Initialize(m_pSharedStorage.get(), pAllocator);
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::Initialize(fs::IStorage* pBaseStorage) NN_NOEXCEPT
    {
        return Initialize(pBaseStorage, &g_DefaultAllocatorForPartitionFileSystem);
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::Initialize(fs::IStorage* pBaseStorage, MemoryResource* pAllocator) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(!m_IsInitialized, ResultPreconditionViolation());

        m_pUniqueMetaData.reset(new MetaData());
        NN_RESULT_THROW_UNLESS( m_pUniqueMetaData != nullptr, ResultAllocationMemoryFailedInPartitionFileSystemA() );
        NN_RESULT_DO(m_pUniqueMetaData->Initialize(pBaseStorage, pAllocator));
        m_pMetaData = m_pUniqueMetaData.get();
        m_pBaseFile = pBaseStorage;

        m_MetaDataSize = m_pMetaData->GetMetaDataSize();
        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::Initialize(std::unique_ptr<MetaData>&& pMetaData, std::shared_ptr<fs::IStorage> pBaseStorage) NN_NOEXCEPT
    {
        m_pUniqueMetaData = std::move(pMetaData);
        return Initialize(m_pUniqueMetaData.get(), std::move(pBaseStorage));
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::Initialize(MetaData* pMetaData, std::shared_ptr<fs::IStorage> pBaseStorage) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(!m_IsInitialized, ResultPreconditionViolation());

        m_pSharedStorage = std::move(pBaseStorage);
        m_pBaseFile = m_pSharedStorage.get();

        m_pMetaData = pMetaData;
        m_MetaDataSize = m_pMetaData->GetMetaDataSize();

        m_IsInitialized = true;
        NN_RESULT_SUCCESS;
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::GetFileBaseOffset(int64_t* outValue, const char* path) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());

        int index = m_pMetaData->GetEntryIndex(path + 1);
        NN_RESULT_THROW_UNLESS( index >= 0, ResultPathNotFound() );
        auto pEntry = m_pMetaData->GetEntry(index);
        *outValue = m_MetaDataSize + pEntry->offset;

        NN_RESULT_SUCCESS;
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        NN_FSP_REQUIRES(path[0] == '/', ResultInvalidPath());

        if( strncmp("/", path, 2) == 0 )
        {
            // ルートディレクトリのみがディレクトリ
            *outValue = nn::fs::DirectoryEntryType_Directory;
            NN_RESULT_SUCCESS;
        }
        else if( m_pMetaData->GetEntryIndex(path + 1) >= 0 ) // 冒頭の '/' を飛ばす
        {
            // 存在すれば必ずファイル
            *outValue = nn::fs::DirectoryEntryType_File;
            NN_RESULT_SUCCESS;
        }
        else
        {
            return nn::fs::ResultPathNotFound();
        }
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        NN_FSP_REQUIRES( (mode & (nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)) != 0, ResultInvalidArgument(), "OpenMode_Read or OpenMode_Write is required.");

        int index = m_pMetaData->GetEntryIndex(path + 1);
        NN_RESULT_THROW_UNLESS( index >= 0, ResultPathNotFound() );
        auto pEntry = m_pMetaData->GetEntry(index);

        std::unique_ptr<PartitionFile> file(new PartitionFile(this, pEntry, mode));
        NN_RESULT_THROW_UNLESS( file != nullptr, ResultAllocationMemoryFailedInPartitionFileSystemB() );
        *outValue = std::move(file);

        NN_RESULT_SUCCESS;
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT
    {
        NN_FSP_REQUIRES(m_IsInitialized, ResultPreconditionViolation());
        NN_RESULT_THROW_UNLESS( strncmp("/", path, 2) == 0, ResultPathNotFound() ); // ディレクトリはルートディレクトリのみ

        std::unique_ptr<PartitionDirectory> directory(new PartitionDirectory(this, mode));
        NN_RESULT_THROW_UNLESS( directory != nullptr, nn::fs::ResultAllocationMemoryFailedInPartitionFileSystemC() );
        *outValue = std::move(directory);

        NN_RESULT_SUCCESS;
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoCreateFile(const char *path, int64_t size, int option) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_UNUSED(size);
        NN_UNUSED(option);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "CreateFile() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoDeleteFile(const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "DeleteFile() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoCreateDirectory(const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "CreateDirectory() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoDeleteDirectory(const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "DeleteDirectory() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoDeleteDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "DeleteDirectoryRecursively() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoCleanDirectoryRecursively(const char *path) NN_NOEXCEPT
    {
        NN_UNUSED(path);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "CleanDirectory() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoRenameFile(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "RenameFile() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoRenameDirectory(const char *currentPath, const char *newPath) NN_NOEXCEPT
    {
        NN_UNUSED(currentPath);
        NN_UNUSED(newPath);
        NN_FSP_REQUIRES(NN_STATIC_CONDITION(false), ResultUnsupportedOperation(), "RenameDirectory() is not supported.");
    }

    template <typename MetaData>
    Result PartitionFileSystemCore<MetaData>::DoCommit() NN_NOEXCEPT
    {
        NN_RESULT_SUCCESS;
    }


    template class PartitionFileSystemCore<PartitionFileSystemMeta>;
    template class PartitionFileSystemCore<Sha256PartitionFileSystemMeta>;

}}

