﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/util/util_IntrusiveList.h>


namespace nn { namespace fssystem {

// 侵入型 lru リストでキーバリューを格納する
template <typename Key, typename Value>
class LruListCache
{

public:

    class Node : public nn::fs::detail::Newable
    {
    NN_DISALLOW_COPY(Node);
    public:
        explicit Node(const Value& value) NN_NOEXCEPT
            : value(value)
        {
        }

        Key key;
        Value value;
        nn::util::IntrusiveListNode m_MruListNode;

    };

private:

    typedef nn::util::IntrusiveList<Node, nn::util::IntrusiveListMemberNodeTraits<Node, &Node::m_MruListNode>> MruListType;
    MruListType m_MruList;

public:

    bool FindValueAndUpdateMru(Value* outValue, Key key) NN_NOEXCEPT
    {
        for (auto it = m_MruList.begin(); it != m_MruList.end(); ++it)
        {
            if(it->key == key)
            {
                *outValue = it->value;

                // update mru
                m_MruList.erase(it);
                m_MruList.push_front(*it);

                return true;
            }
        }

        return false;
    }

    std::unique_ptr<Node> PopLruNode() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS(!m_MruList.empty());
        Node* pLru = &(*m_MruList.rbegin());
        m_MruList.pop_back();

        std::unique_ptr<Node> p(pLru);
        return p;
    }

    void PushMruNode(std::unique_ptr<Node>&& pOutValue, Key key) NN_NOEXCEPT
    {
        pOutValue->key = key;

        m_MruList.push_front(*pOutValue);
        pOutValue.release();
    }

    void DeleteAllNodes() NN_NOEXCEPT
    {
        while (!m_MruList.empty())
        {
            Node* pLru = &(*m_MruList.rbegin());
            m_MruList.erase(m_MruList.iterator_to(*pLru));
            delete pLru;
        }
    }

    typename MruListType::size_type GetSize() const NN_NOEXCEPT
    {
        return m_MruList.size();
    }

    bool IsEmpty() const NN_NOEXCEPT
    {
        return m_MruList.empty();
    }
};


}}
