﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <mutex>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/fs/fs_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkLog.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/fs_ThreadPriorityChanger.h>

#include <nn/fssystem/fs_Assert.h>

#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/util/util_BitUtil.h>

#include "fs_HierarchicalSha256Storage.h"

namespace nn { namespace fssystem {

HierarchicalSha256Storage::HierarchicalSha256Storage() NN_NOEXCEPT
    : m_Mutex(false)
{
}

namespace {
    int Log2(int value)
    {
        NN_SDK_REQUIRES(value > 0);
        NN_SDK_REQUIRES(nn::util::ispow2(value));

        int log2Value = 0;
        while((value >>= 1) > 0)
        {
            ++log2Value;
        }
        return log2Value;
    }
}

Result HierarchicalSha256Storage::Initialize(fs::IStorage* pBaseStorages[], int layerCount, size_t hashTargetBlockSize, void* hashBuffer, size_t hashBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(layerCount == LayerCount);
    NN_SDK_REQUIRES(nn::util::ispow2(hashTargetBlockSize));
    NN_SDK_REQUIRES_NOT_NULL(hashBuffer);
    NN_UNUSED(layerCount);
    NN_UNUSED(hashTargetBlockSize);

    m_HashTargetBlockSize = static_cast<int>(hashTargetBlockSize);
    m_Log2SizeRatio = Log2(static_cast<int>(hashTargetBlockSize / HashSize));

    // データ領域 32 * (hashTargetBlockSize / 32) * (hashTargetBlockSize / 32) まで
    NN_RESULT_DO(pBaseStorages[2]->GetSize(&m_BaseStorageSize));
    if(m_BaseStorageSize > static_cast<int64_t>(HashSize) << m_Log2SizeRatio << m_Log2SizeRatio)
    {
        m_BaseStorageSize = 0;
        return nn::fs::ResultHierarchicalSha256BaseStorageTooLarge();
    }

    m_HashBuffer     = static_cast<char*>(hashBuffer);
    m_HashBufferSize = hashBufferSize;

    m_pBaseStorage = pBaseStorages[2];

    // Read master hash
    char masterHash[HashSize];
    NN_RESULT_DO(pBaseStorages[0]->Read(0, masterHash, HashSize));

    // Read and check hash data
    int64_t hashStorageSize;
    NN_RESULT_DO(pBaseStorages[1]->GetSize(&hashStorageSize));
    NN_SDK_REQUIRES(nn::util::is_aligned(hashStorageSize, HashSize));
    NN_SDK_REQUIRES(hashStorageSize <= m_HashTargetBlockSize);
    NN_SDK_REQUIRES(hashStorageSize <= static_cast<int64_t>(hashBufferSize));

    NN_RESULT_DO(pBaseStorages[1]->Read(0, m_HashBuffer, static_cast<size_t>(hashStorageSize)));

    char hash[HashSize];
    crypto::GenerateSha256Hash(hash, sizeof(hash), m_HashBuffer, static_cast<size_t>(hashStorageSize));

    NN_RESULT_THROW_UNLESS(nn::crypto::IsSameBytes(hash, masterHash, HashSize), fs::ResultHierarchicalSha256HashVerificationFailed());

    NN_RESULT_SUCCESS;
}

Result HierarchicalSha256Storage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    // m_HashTargetBlockSize 単位アクセスしかこない前提
    NN_FSP_REQUIRES(nn::util::is_aligned(offset, m_HashTargetBlockSize), nn::fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(nn::util::is_aligned(size,   m_HashTargetBlockSize), nn::fs::ResultInvalidArgument());

    // ストレージ末端まで
    size_t truncatedSize = static_cast<size_t>(std::min<int64_t>(m_BaseStorageSize, nn::util::align_up(offset + size, m_HashTargetBlockSize)) - offset);

    NN_RESULT_DO(m_pBaseStorage->Read(offset, buffer, truncatedSize));

    ScopedThreadPriorityChanger changeThreadPriority(+1, ScopedThreadPriorityChanger::Mode::Relative);

    auto currentOffset = offset;
    auto restSize = truncatedSize;

    while(restSize > 0)
    {
        auto processSize = static_cast<size_t>(std::min<int64_t>(m_HashTargetBlockSize, restSize));
        char hash[HashSize];
        crypto::GenerateSha256Hash(hash, sizeof(hash), static_cast<char*>(buffer) + (currentOffset - offset), processSize);

        NN_SDK_ASSERT(static_cast<size_t>(currentOffset >> m_Log2SizeRatio) < m_HashBufferSize);

        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);

            if( !nn::crypto::IsSameBytes(hash, &(m_HashBuffer[currentOffset >> m_Log2SizeRatio]), HashSize) )
            {
                // 不整合
                memset(buffer, 0x00, size);
                return fs::ResultHierarchicalSha256HashVerificationFailed();
            }
        }

        currentOffset  += processSize;
        restSize       -= processSize;
    }

    NN_RESULT_SUCCESS;
}


Result HierarchicalSha256Storage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_THROW_UNLESS(buffer != nullptr, fs::ResultNullptrArgument());

    // m_HashTargetBlockSize 単位アクセスしかこないと仮定
    NN_FSP_REQUIRES(nn::util::is_aligned(offset, m_HashTargetBlockSize), nn::fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(nn::util::is_aligned(size,   m_HashTargetBlockSize), nn::fs::ResultInvalidArgument());

    // ストレージ末端まで
    size_t truncatedSize = static_cast<size_t>(std::min<int64_t>(m_BaseStorageSize, nn::util::align_up(offset + size, m_HashTargetBlockSize)) - offset);

    auto currentOffset = offset;
    auto restSize = truncatedSize;

    while(restSize > 0U)
    {
        size_t writeSize = static_cast<size_t>(std::min<int64_t>(m_HashTargetBlockSize, restSize));

        char newHash[HashSize];

        {
            ScopedThreadPriorityChanger changeThreadPriority(+1, ScopedThreadPriorityChanger::Mode::Relative);
            crypto::GenerateSha256Hash(newHash, sizeof(newHash), static_cast<const char*>(buffer) + (currentOffset - offset), writeSize);
        }

        NN_RESULT_DO(m_pBaseStorage->Write(currentOffset, static_cast<const char*>(buffer) + (currentOffset - offset), writeSize));

        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            memcpy(&(m_HashBuffer[currentOffset >> m_Log2SizeRatio]), newHash, HashSize);
        }

        currentOffset += writeSize;
        restSize      -= writeSize;
    }

    NN_RESULT_SUCCESS;
}

Result HierarchicalSha256Storage::GetSize(int64_t* outValue) NN_NOEXCEPT
{
    return m_pBaseStorage->GetSize(outValue);
}

Result HierarchicalSha256Storage::OperateRange(
           void* outBuffer,
           size_t outBufferSize,
           fs::OperationId operationId,
           int64_t offset,
           int64_t size,
           const void* inBuffer,
           size_t inBufferSize
       ) NN_NOEXCEPT
{
    if( size == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    // m_HashTargetBlockSize 単位アクセスしかこない前提
    NN_FSP_REQUIRES(
        nn::util::is_aligned(offset, m_HashTargetBlockSize),
        nn::fs::ResultInvalidArgument());
    NN_FSP_REQUIRES(
        nn::util::is_aligned(size,   m_HashTargetBlockSize),
        nn::fs::ResultInvalidArgument());

    // ストレージ末端まで
    size_t truncatedSize = static_cast<size_t>(std::min<int64_t>(
        m_BaseStorageSize,
        nn::util::align_up(offset + size, m_HashTargetBlockSize)) - offset);

    return m_pBaseStorage->OperateRange(
        outBuffer,
        outBufferSize,
        operationId,
        offset,
        truncatedSize,
        inBuffer,
        inBufferSize);
}


}}
