﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/fs_AesXtsFile.h>
#include <nn/fssystem/fs_AesXtsFileSystem.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fssystem/fs_StorageFile.h>

namespace nn { namespace fssystem {

NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::Size);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::HeaderSignSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::KeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::Reserved2Size);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::EncryptionKeyGenerationKeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::MacKeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFileHeader::CryptoKeySize);
NN_DEFINE_STATIC_CONSTANT(const int AesXtsFileHeader::CryptoKeyCount);

void AesXtsFileHeader::Create(
        const char* path,
        int64_t fileSize,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize,
        GenerateRandomFunction pGenerateRandom
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_LESS_EQUAL(0, fileSize);
    NN_SDK_REQUIRES_NOT_NULL(pEncryptionKeyGenerationKey);
    NN_SDK_REQUIRES_EQUAL(encryptionKeyGenerationKeySize, EncryptionKeyGenerationKeySize);
    NN_SDK_REQUIRES_NOT_NULL(pMacKey);
    NN_SDK_REQUIRES_EQUAL(macKeySize, MacKeySize);
    NN_SDK_REQUIRES_NOT_NULL(pGenerateRandom);

    // パラメータを設定
    this->signature = static_cast<uint32_t>(Signature::V0);
    this->actualFileSize = fileSize;

    // データ暗号化鍵を生成
    pGenerateRandom(this->key1, KeySize);
    pGenerateRandom(this->key2, KeySize);

    // 予約領域を初期化
    this->reserved1 = 0;
    std::memset(this->reserved2, 0, sizeof(this->reserved2));

    // 署名と暗号化
    SignAndEncryptKeys(path, pEncryptionKeyGenerationKey, encryptionKeyGenerationKeySize, pMacKey, macKeySize);
}

void AesXtsFileHeader::ComputeMac(void* pOutMac, size_t macSize, const void* pMacKey, size_t macKeySize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMac);
    NN_SDK_REQUIRES_EQUAL(macSize, HeaderSignSize);
    NN_SDK_REQUIRES_NOT_NULL(pMacKey);
    NN_SDK_REQUIRES_EQUAL(macKeySize, MacKeySize);

    nn::crypto::GenerateHmacSha256Mac(
        pOutMac, macSize,
        pMacKey, macKeySize,
        reinterpret_cast<const char*>(this) + sizeof(this->headerSign),
        sizeof(*this) - sizeof(this->headerSign)
    );
}

void AesXtsFileHeader::ComputeKeys(void* pOutKeys, size_t keysSize, const char* path, const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutKeys);
    NN_SDK_REQUIRES_EQUAL(keysSize, CryptoKeySize * CryptoKeyCount);
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(pEncryptionKeyGenerationKey);
    NN_SDK_REQUIRES_EQUAL(encryptionKeyGenerationKeySize, EncryptionKeyGenerationKeySize);

    nn::crypto::GenerateHmacSha256Mac(
        pOutKeys, keysSize,
        path, strnlen(path, fs::EntryNameLengthMax),
        pEncryptionKeyGenerationKey, encryptionKeyGenerationKeySize
    );
}

void AesXtsFileHeader::Encrypt(const void* pKeys, size_t keysSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKeys);
    NN_SDK_REQUIRES_EQUAL(keysSize, CryptoKeySize * CryptoKeyCount);

    NN_UNUSED(keysSize);

    crypto::AesEncryptor128 encryptor;
    {
        encryptor.Initialize(pKeys, CryptoKeySize);
        encryptor.EncryptBlock(this->key1, sizeof(this->key1), this->key1, sizeof(this->key1));
    }
    {
        encryptor.Initialize(reinterpret_cast<const char*>(pKeys) + CryptoKeySize, CryptoKeySize);
        encryptor.EncryptBlock(this->key2, sizeof(this->key2), this->key2, sizeof(this->key2));
    }
}

void AesXtsFileHeader::Decrypt(const void* pKeys, size_t keysSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pKeys);
    NN_SDK_REQUIRES_EQUAL(keysSize, CryptoKeySize * CryptoKeyCount);

    NN_UNUSED(keysSize);

    crypto::AesDecryptor128 decryptor;
    {
        decryptor.Initialize(pKeys, CryptoKeySize);
        decryptor.DecryptBlock(this->key1, sizeof(this->key1), this->key1, sizeof(this->key1));
    }
    {
        decryptor.Initialize(reinterpret_cast<const char*>(pKeys) + CryptoKeySize, CryptoKeySize);
        decryptor.DecryptBlock(this->key2, sizeof(this->key2), this->key2, sizeof(this->key2));
    }
}

void AesXtsFileHeader::SignAndEncryptKeys(
        const char* path,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(pEncryptionKeyGenerationKey);
    NN_SDK_REQUIRES_EQUAL(encryptionKeyGenerationKeySize, EncryptionKeyGenerationKeySize);
    NN_SDK_REQUIRES_NOT_NULL(pMacKey);
    NN_SDK_REQUIRES_EQUAL(macKeySize, MacKeySize);

    // 署名更新
    ComputeMac(this->headerSign, sizeof(this->headerSign), pMacKey, macKeySize);

    // データ暗号化鍵の暗号化鍵を生成
    uint8_t encryptKeys[CryptoKeySize * CryptoKeyCount];
    ComputeKeys(encryptKeys, sizeof(encryptKeys), path, pEncryptionKeyGenerationKey, encryptionKeyGenerationKeySize);

    // 暗号化
    Encrypt(encryptKeys, sizeof(encryptKeys));
}

bool AesXtsFileHeader::DecryptKeysAndVerify(
        void* pOutKeys, size_t keysSize,
        const char* path,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutKeys);
    NN_SDK_REQUIRES_EQUAL(keysSize, CryptoKeySize * CryptoKeyCount);
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(pEncryptionKeyGenerationKey);
    NN_SDK_REQUIRES_EQUAL(encryptionKeyGenerationKeySize, EncryptionKeyGenerationKeySize);
    NN_SDK_REQUIRES_NOT_NULL(pMacKey);
    NN_SDK_REQUIRES_EQUAL(macKeySize, MacKeySize);

    // 復号の鍵を生成
    ComputeKeys(pOutKeys, keysSize, path, pEncryptionKeyGenerationKey, encryptionKeyGenerationKeySize);

    // 復号
    Decrypt(pOutKeys, keysSize);

    // 署名検証
    char signHeader[HeaderSignSize];
    ComputeMac(signHeader, sizeof(signHeader), pMacKey, macKeySize);

    // 検証結果を返す
    return nn::crypto::IsSameBytes(signHeader, this->headerSign, HeaderSignSize);
}

bool AesXtsFileHeader::Update(int64_t fileSize, const void* pKeys, size_t keysSize, const void* pMacKey, size_t macKeySize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_LESS_EQUAL(0, fileSize);
    NN_SDK_REQUIRES_NOT_NULL(pKeys);
    NN_SDK_REQUIRES_EQUAL(keysSize, CryptoKeySize * CryptoKeyCount);
    NN_SDK_REQUIRES_NOT_NULL(pMacKey);
    NN_SDK_REQUIRES_EQUAL(macKeySize, MacKeySize);

    // 復号
    Decrypt(pKeys, keysSize);

    // 署名検証
    char signHeader[HeaderSignSize];
    ComputeMac(signHeader, sizeof(signHeader), pMacKey, macKeySize);
    if( !nn::crypto::IsSameBytes(signHeader, this->headerSign, HeaderSignSize) )
    {
        return false;
    }

    // パラメータ更新
    this->actualFileSize = fileSize;

    // 署名更新
    ComputeMac(this->headerSign, sizeof(this->headerSign), pMacKey, macKeySize);

    // 暗号化
    Encrypt(pKeys, keysSize);

    return true;
}

NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFile::AesBlockSize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFile::KeySize);
NN_DEFINE_STATIC_CONSTANT(const size_t AesXtsFile::IvSize);

AesXtsFile::AesXtsFile() NN_NOEXCEPT
    : m_pBaseFile(),
      m_pFileStorage(),
      m_pFileSubStorage(),
      m_pAesXtsStorage(),
      m_pAlignmentMatchingStorage(),
      m_pStorageFile(),
      m_FileSize(0),
      m_Mode(static_cast<fs::OpenMode>(0))
{
    std::memset(m_MacKey, 0, sizeof(m_MacKey));
    std::memset(m_HeaderKeyEncryptionKeys, 0, sizeof(m_HeaderKeyEncryptionKeys));
}

AesXtsFile::~AesXtsFile() NN_NOEXCEPT
{
    m_pStorageFile.reset();
    m_pAlignmentMatchingStorage.reset();
    m_pAesXtsStorage.reset();
    m_pFileSubStorage.reset();
    m_pFileStorage.reset();
}

Result AesXtsFile::Initialize(
        fs::OpenMode mode,
        std::unique_ptr<fs::fsa::IFile>&& pBaseFile,
        const char* path,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize,
        const void* pIv, size_t ivSize,
        size_t blockSize
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_pStorageFile == nullptr); // 未初期化である
    NN_SDK_REQUIRES_NOT_NULL(pBaseFile);
    NN_SDK_REQUIRES_EQUAL(encryptionKeyGenerationKeySize, AesXtsFileHeader::EncryptionKeyGenerationKeySize);
    NN_SDK_REQUIRES_EQUAL(macKeySize, AesXtsFileHeader::MacKeySize);

    // ヘッダーを読み出して検証
    size_t readSize;
    AesXtsFileHeader header;
    NN_RESULT_DO(pBaseFile->Read(&readSize, 0, &header, sizeof(header), nn::fs::ReadOption::MakeValue(0)));
    if (readSize != sizeof(header))
    {
        return nn::fs::ResultAesXtsFileSystemFileHeaderSizeCorruptedOnFileOpen();
    }
    if (!header.HasValidSignature())
    {
        return nn::fs::ResultAesXtsFileSystemFileNoHeaderOnFileOpen();
    }
    if (!header.DecryptKeysAndVerify(m_HeaderKeyEncryptionKeys, sizeof(m_HeaderKeyEncryptionKeys), path, pEncryptionKeyGenerationKey, encryptionKeyGenerationKeySize, pMacKey, macKeySize))
    {
        return nn::fs::ResultAesXtsFileSystemFileHeaderCorruptedOnFileOpen();
    }

    m_FileSize = header.actualFileSize;
    std::memcpy(m_MacKey, pMacKey, sizeof(m_MacKey));
    m_Mode = mode;

    // ファイルインスタンスの構築
    std::unique_ptr<fs::IStorage> pFileStorage(new fs::FileStorage(pBaseFile.get()));
    NN_RESULT_THROW_UNLESS(pFileStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInAesXtsFileA());

    // ファイルサイズの妥当性検証
    int64_t baseFileSize = 0;
    NN_RESULT_DO(pFileStorage->GetSize(&baseFileSize));
    if (baseFileSize < static_cast<int>(AesXtsFileHeader::Size) + nn::util::align_up(header.actualFileSize, AesXtsStorage::KeySize))
    {
        return nn::fs::ResultAesXtsFileSystemFileSizeCorruptedOnFileOpen();
    }

    std::unique_ptr<fs::SubStorage> pFileSubStorage(new fs::SubStorage(pFileStorage.get(), AesXtsFileHeader::Size, baseFileSize - AesXtsFileHeader::Size));
    NN_RESULT_THROW_UNLESS(pFileSubStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInAesXtsFileE());
    pFileSubStorage->SetResizable(true);

    std::unique_ptr<fs::IStorage> pAesXtsStorage(new fssystem::AesXtsStorage(pFileSubStorage.get(), header.key1, header.key2, AesXtsFileHeader::KeySize, pIv, ivSize, blockSize));
    NN_RESULT_THROW_UNLESS(pAesXtsStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInAesXtsFileB());

    std::unique_ptr<fs::IStorage> pAlignmentMatchingStorage(new fssystem::AlignmentMatchingStoragePooledBuffer<1>(pAesXtsStorage.get(), AesXtsStorage::KeySize));
    NN_RESULT_THROW_UNLESS(pAlignmentMatchingStorage != nullptr, nn::fs::ResultAllocationMemoryFailedInAesXtsFileC());

    std::unique_ptr<fs::fsa::IFile> pStorageFile(new fssystem::StorageFile(pAlignmentMatchingStorage.get(), mode));
    NN_RESULT_THROW_UNLESS(pStorageFile != nullptr, nn::fs::ResultAllocationMemoryFailedInAesXtsFileD());

    // 設定
    m_pBaseFile = std::move(pBaseFile);
    m_pFileStorage = std::move(pFileStorage);
    m_pFileSubStorage = std::move(pFileSubStorage);
    m_pAesXtsStorage = std::move(pAesXtsStorage);
    m_pAlignmentMatchingStorage = std::move(pAlignmentMatchingStorage);
    m_pStorageFile = std::move(pStorageFile);

    NN_RESULT_SUCCESS;
}

Result AesXtsFile::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorageFile);

    size_t readSize = 0;
    NN_RESULT_DO(DryRead(&readSize, offset, size, option, m_Mode));

    return m_pStorageFile->Read(outValue, offset, buffer, readSize, option);
}

Result AesXtsFile::DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorageFile);

    NN_FSP_REQUIRES((m_Mode & fs::OpenMode::OpenMode_Write) != 0, fs::ResultWriteUnpermitted());

    if (m_FileSize < static_cast<int64_t>(offset + size))
    {
        NN_FSP_REQUIRES((m_Mode & fs::OpenMode::OpenMode_AllowAppend) != 0, fs::ResultFileExtensionWithoutOpenModeAllowAppend());
        NN_RESULT_DO(DoSetSize(offset + size));
    }

    return m_pStorageFile->Write(offset, buffer, size, option);
}

Result AesXtsFile::DoFlush() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorageFile);
    if ((m_Mode & fs::OpenMode_Write) == 0)
    {
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_DO(m_pStorageFile->Flush());
    NN_RESULT_SUCCESS;
}

Result AesXtsFile::DoSetSize(int64_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorageFile);
    NN_RESULT_DO(DrySetSize(size, m_Mode));

    const auto oldSize = m_FileSize;
    AesXtsFileHeader oldHeader;
    bool isRewindNeeded = false;

    NN_UTIL_SCOPE_EXIT
    {
        if( isRewindNeeded )
        {
            m_FileSize = oldSize;
            (void)m_pStorageFile->SetSize(oldSize);
            (void)m_pBaseFile->Write(0, &oldHeader, sizeof(oldHeader), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        }
    };

    // ヘッダー更新
    {
        size_t readSize;
        AesXtsFileHeader header;
        NN_RESULT_DO(m_pBaseFile->Read(&readSize, 0, &header, sizeof(header), nn::fs::ReadOption::MakeValue(0)));
        if (readSize != sizeof(header))
        {
            return nn::fs::ResultAesXtsFileSystemFileSizeCorruptedOnFileSetSize();
        }
        oldHeader = header;

        if (!header.Update(size, m_HeaderKeyEncryptionKeys, sizeof(m_HeaderKeyEncryptionKeys), m_MacKey, sizeof(m_MacKey)))
        {
            return nn::fs::ResultAesXtsFileSystemFileHeaderCorruptedOnFileSetSize();
        }

        isRewindNeeded = true;
        NN_RESULT_DO(m_pBaseFile->Write(0, &header, sizeof(header), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
    }

    // ファイルサイズ更新
    m_FileSize = size;
    NN_RESULT_DO(m_pStorageFile->SetSize(m_FileSize));

    isRewindNeeded = false;
    NN_RESULT_SUCCESS;
}

Result AesXtsFile::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_pStorageFile);
    *outValue = m_FileSize;
    NN_RESULT_SUCCESS;
}

Result AesXtsFile::DoOperateRange(
    void* outBuffer,
    size_t outBufferSize,
    fs::OperationId operationId,
    int64_t offset,
    int64_t size,
    const void* inBuffer,
    size_t inBufferSize) NN_NOEXCEPT
{
    switch( operationId )
    {
    case fs::OperationId::Invalidate:
    case fs::OperationId::QueryRange:
        {
            NN_SDK_REQUIRES_NOT_NULL(m_pStorageFile);
            return m_pStorageFile->OperateRange(
                outBuffer,
                outBufferSize,
                operationId,
                offset,
                size,
                inBuffer,
                inBufferSize);
        }
    default:
        return nn::fs::ResultUnsupportedOperation();
    }
}

}}
