﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fssrv/fssrv_SaveDataSharedFileStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_FormatString.h>

namespace nn { namespace fssrv {

    namespace {

        nn::os::Mutex g_Mutex(false);
        SaveDataFileStorageHolder g_SaveDataFileStorageHolder;

    }

    Result SaveDataOpenTypeSetFileStorage::Initialize(std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem, const char* path, nn::fs::OpenMode mode, nn::fs::SaveDataId saveDataId, OpenType type) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::FileStorageBasedFileSystem::Initialize(std::move(pBaseFileSystem), path, mode));
        NN_RESULT_DO(SetOpenType(type));
        m_SaveDataId = saveDataId;

        NN_RESULT_SUCCESS;
    }

    Result SaveDataOpenTypeSetFileStorage::SetOpenType(OpenType type) NN_NOEXCEPT
    {
        if( type == OpenType::Normal )
        {
            if( m_IsOpenedAsNormal )
            {
                NN_RESULT_THROW(nn::fs::ResultTargetLocked());
            }
            m_IsOpenedAsNormal = true;
            m_ActiveType = type;

            NN_RESULT_SUCCESS;
        }
        else if( type == OpenType::Internal )
        {
            if( m_IsOpenedAsInternal )
            {
                NN_RESULT_THROW(nn::fs::ResultTargetLocked());
            }
            m_IsOpenedAsInternal = true;
            m_ActiveType = type;

            NN_RESULT_SUCCESS;
        }

        NN_RESULT_THROW(nn::fs::ResultUnsupportedOperation());
    }

    void SaveDataOpenTypeSetFileStorage::UnsetOpenType(OpenType type) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_Mutex);

        if( type == OpenType::Normal )
        {
            m_IsOpenedAsNormal = false;
            m_ActiveType = m_IsOpenedAsInternal ? OpenType::Internal : OpenType::None;
        }
        else if( type == OpenType::Internal )
        {
            m_IsOpenedAsInternal = false;
            m_ActiveType = m_IsOpenedAsNormal ? OpenType::Normal : OpenType::None;
        }

        if( !IsActive() )
        {
            g_SaveDataFileStorageHolder.Unregister(m_SaveDataId);
        }
    }

    SaveDataSharedFileStorage::~SaveDataSharedFileStorage() NN_NOEXCEPT
    {
        if( m_pBaseFileStorage != nullptr )
        {
            m_pBaseFileStorage->UnsetOpenType(m_OpenType);
        }
    }

    void SaveDataSharedFileStorage::Initialize(std::shared_ptr<SaveDataOpenTypeSetFileStorage> pBaseFileSystem, nn::fs::SaveDataId saveDataId, SaveDataOpenTypeSetFileStorage::OpenType type) NN_NOEXCEPT
    {
        m_pBaseFileStorage = std::move(pBaseFileSystem);
        m_SaveDataId = saveDataId;
        m_OpenType = type;
    }

    Result SaveDataSharedFileStorage::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(AccessCheck());

        return m_pBaseFileStorage->Read(offset, buffer, size);
    }

    Result SaveDataSharedFileStorage::Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(AccessCheck());

        return m_pBaseFileStorage->Write(offset, buffer, size);
    }

    Result SaveDataSharedFileStorage::SetSize(int64_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(AccessCheck());

        return m_pBaseFileStorage->SetSize(size);
    }

    Result SaveDataSharedFileStorage::GetSize(int64_t* outValue) NN_NOEXCEPT
    {
        NN_RESULT_DO(AccessCheck());

        return m_pBaseFileStorage->GetSize(outValue);
    }

    Result SaveDataSharedFileStorage::Flush() NN_NOEXCEPT
    {
        NN_RESULT_DO(AccessCheck());

        return m_pBaseFileStorage->Flush();
    }

    Result SaveDataSharedFileStorage::OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize
        ) NN_NOEXCEPT
    {
        NN_RESULT_DO(AccessCheck());

        return m_pBaseFileStorage->OperateRange(outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize);
    }

    SaveDataFileStorageHolder::~SaveDataFileStorageHolder() NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_Mutex);
        while(!m_FileStorageList.empty())
        {
            auto entry = &(*m_FileStorageList.rbegin());
            m_FileStorageList.erase(m_FileStorageList.iterator_to(*entry));
            delete entry;
        }
    }

    nn::Result SaveDataFileStorageHolder::OpenSaveDataStorage(std::shared_ptr<fs::IStorage>* pOutStorage, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem,
        nn::fs::SaveDataId saveDataId, nn::fs::SaveDataType saveDataType, nn::fs::OpenMode mode, SaveDataOpenTypeSetFileStorage::OpenType type) NN_NOEXCEPT
    {
        char saveImageName[1 + 16 + 1];
        util::SNPrintf(saveImageName, sizeof(saveImageName), "/%016llx", saveDataId);

        if( saveDataType != nn::fs::SaveDataType::Account && saveDataType != nn::fs::SaveDataType::Device )
        {
            std::shared_ptr<fs::FileStorageBasedFileSystem> fileStorage = fssystem::AllocateShared<fs::FileStorageBasedFileSystem>();
            NN_RESULT_THROW_UNLESS(fileStorage, nn::fs::ResultAllocationMemoryFailedAllocateShared());
            NN_RESULT_DO(fileStorage->Initialize(std::move(pBaseFileSystem), saveImageName, mode));
            *pOutStorage = std::move(fileStorage);

            NN_RESULT_SUCCESS;
        }

        std::lock_guard<nn::os::Mutex> scopedLock(g_Mutex);
        std::shared_ptr<SaveDataSharedFileStorage> fileStorage = nn::fssystem::AllocateShared<SaveDataSharedFileStorage>();
        NN_RESULT_THROW_UNLESS(fileStorage, nn::fs::ResultAllocationMemoryFailedAllocateShared());
        std::shared_ptr<SaveDataOpenTypeSetFileStorage> baseFileStorage = GetStorage(saveDataId);
        if( baseFileStorage != nullptr )
        {
            NN_RESULT_DO(baseFileStorage->SetOpenType(type));
        }
        else
        {
            baseFileStorage = nn::fssystem::AllocateShared<SaveDataOpenTypeSetFileStorage>();
            NN_RESULT_THROW_UNLESS(baseFileStorage, nn::fs::ResultAllocationMemoryFailedAllocateShared());
            NN_RESULT_DO(baseFileStorage->Initialize(std::move(pBaseFileSystem), saveImageName, mode, saveDataId, type));
            NN_RESULT_DO(Register(baseFileStorage, saveDataId));
        }

        fileStorage->Initialize(std::move(baseFileStorage), saveDataId, type);

        *pOutStorage = std::move(fileStorage);

        NN_RESULT_SUCCESS;
    }

    Result SaveDataFileStorageHolder::Register(std::shared_ptr<SaveDataOpenTypeSetFileStorage> fileStorage, nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Mutex.IsLockedByCurrentThread());

        Entry* entry = new Entry(std::move(fileStorage), saveDataId);
        NN_RESULT_THROW_UNLESS(entry, nn::fs::ResultAllocationMemoryFailed());
        m_FileStorageList.push_back(*entry);
        NN_RESULT_SUCCESS;
    }
    std::shared_ptr<SaveDataOpenTypeSetFileStorage> SaveDataFileStorageHolder::GetStorage(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Mutex.IsLockedByCurrentThread());

        for(auto& entry : m_FileStorageList)
        {
            if(entry.GetSaveDataId() == saveDataId)
            {
                return entry.GetStorage();
            }
        }
        return nullptr;
    }
    void SaveDataFileStorageHolder::Unregister(nn::fs::SaveDataId saveDataId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(g_Mutex.IsLockedByCurrentThread());

        for(auto& entry : m_FileStorageList)
        {
            if(entry.GetSaveDataId() == saveDataId)
            {
                m_FileStorageList.erase(m_FileStorageList.iterator_to(entry));
                delete &entry;
                return;
            }
        }
    }

    nn::Result OpenSaveDataStorage(std::shared_ptr<fs::IStorage>* pOutStorage, std::shared_ptr<nn::fs::fsa::IFileSystem> pBaseFileSystem,
        nn::fs::SaveDataId saveDataId, nn::fs::SaveDataType saveDataType, nn::fs::OpenMode mode, SaveDataOpenTypeSetFileStorage::OpenType type) NN_NOEXCEPT
    {
        return g_SaveDataFileStorageHolder.OpenSaveDataStorage(pOutStorage, pBaseFileSystem, saveDataId, saveDataType, mode, type);
    }

}}
