﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Allocator.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/fs_RomFsFileSystem.h>

#include <nn/fssrv/fscreator/fssrv_RomFileSystemCreator.h>

using nn::fssystem::RomFsFileSystem;

namespace nn { namespace fssrv { namespace fscreator {

    namespace {
        class RomFileSystemWithBuffer : public RomFsFileSystem
        {
        public:
            explicit RomFileSystemWithBuffer(nn::MemoryResource* pAllocator) NN_NOEXCEPT
                : m_MetaCacheBuffer(nullptr)
            {
                m_Allocator = pAllocator;
            }
            ~RomFileSystemWithBuffer() NN_NOEXCEPT
            {
                if (m_MetaCacheBuffer != nullptr)
                {
                    m_Allocator->deallocate(m_MetaCacheBuffer, m_MetaCacheBufferSize);
                }
            }
            Result Initialize(std::shared_ptr<fs::IStorage> storage) NN_NOEXCEPT
            {
                size_t bufferSize = 0;
                auto result = RomFsFileSystem::GetRequiredWorkingMemorySize(&bufferSize, storage.get());
                if (result.IsFailure() || bufferSize == 0 || bufferSize >= 128 * 1024)
                {
                    return RomFsFileSystem::Initialize(std::move(storage), nullptr, 0, false);
                }

                m_MetaCacheBuffer = m_Allocator->allocate(bufferSize);

                if (m_MetaCacheBuffer == nullptr)
                {
                    return RomFsFileSystem::Initialize(std::move(storage), nullptr, 0, false);
                }

                m_MetaCacheBufferSize = bufferSize;

                return RomFsFileSystem::Initialize(std::move(storage), m_MetaCacheBuffer, bufferSize, true);
            }
        private:
            void* m_MetaCacheBuffer;
            size_t m_MetaCacheBufferSize;
            nn::MemoryResource* m_Allocator;
        };
    }

    RomFileSystemCreator::RomFileSystemCreator(nn::MemoryResource* pAllocator) NN_NOEXCEPT
    {
        m_Allocator = pAllocator;
    }

    Result RomFileSystemCreator::Create(std::shared_ptr<nn::fs::fsa::IFileSystem>* outValue, std::shared_ptr<fs::IStorage> storage) NN_NOEXCEPT
    {
        std::shared_ptr<RomFileSystemWithBuffer> pFs = fssystem::AllocateShared<RomFileSystemWithBuffer>(m_Allocator);
        NN_RESULT_THROW_UNLESS(pFs, nn::fs::ResultAllocationMemoryFailedInRomFileSystemCreatorA());
        NN_RESULT_DO(pFs->Initialize(std::move(storage)));
        *outValue = std::move(pFs);
        NN_RESULT_SUCCESS;
    }

}}}
