﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#if !defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Result.h>
#include <nn/svc/svc_Tcb.h>
#endif

#include "fssrv_ProgramInfo.h"

namespace nn { namespace fssrv { namespace detail {

namespace {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
    class InitialProgramIdRange
    {
    public:
        InitialProgramIdRange()
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(svc::GetSystemInfo(&m_InitialProcessIdMin, svc::SystemInfoType_InitialProcessIdRange, svc::Handle(0), 0));
            NN_ABORT_UNLESS_RESULT_SUCCESS(svc::GetSystemInfo(&m_InitialProcessIdMax, svc::SystemInfoType_InitialProcessIdRange, svc::Handle(0), 1));
            NN_ABORT_UNLESS(m_InitialProcessIdMin <= m_InitialProcessIdMax);
            NN_ABORT_UNLESS(0 < m_InitialProcessIdMin);
        }
        bool IsInitialProgram(Bit64 processId) const
        {
            NN_ABORT_UNLESS(0 < m_InitialProcessIdMin);
            return (m_InitialProcessIdMin <= processId && processId <= m_InitialProcessIdMax);
        }

    private:
        Bit64 m_InitialProcessIdMin;
        Bit64 m_InitialProcessIdMax;
    };

    Bit64 GetCurrentProcessImpl() NN_NOEXCEPT
    {
        nn::Bit64 currentProcessId = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::svc::GetProcessId(&currentProcessId, nn::svc::PSEUDO_HANDLE_CURRENT_PROCESS));
        return currentProcessId;
    }
#endif

std::aligned_storage<128>::type g_StaticBuffer;
template <typename T>
class StaticAllocatorForProgramInfoForInitialProcess : public std::allocator<T>
{
public:
    StaticAllocatorForProgramInfoForInitialProcess() {}
    StaticAllocatorForProgramInfoForInitialProcess(const StaticAllocatorForProgramInfoForInitialProcess&) {}
    template<class U>
    StaticAllocatorForProgramInfoForInitialProcess(const StaticAllocatorForProgramInfoForInitialProcess<U>&) {}

    template<typename U>
    struct rebind
    {
        typedef StaticAllocatorForProgramInfoForInitialProcess<U> other;
    };

    T* allocate(size_t size, const T* hint = 0)
    {
        NN_UNUSED(hint);
        NN_ABORT_UNLESS(sizeof(T) * size <= sizeof(g_StaticBuffer));
        return reinterpret_cast<T*>(&g_StaticBuffer);
    }
    void deallocate(T* p, size_t size)
    {
        NN_UNUSED(p);
        NN_UNUSED(size);
    }
};

} // namespace


    bool IsInitialProgram(Bit64 processId) NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_FUNCTION_LOCAL_STATIC(InitialProgramIdRange, s_InitialProgramIdRange);
        return s_InitialProgramIdRange.IsInitialProgram(processId);
#else
        NN_UNUSED(processId);
        return true;
#endif
    }

    bool IsCurrentProcess(Bit64 processId) NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        NN_FUNCTION_LOCAL_STATIC(Bit64, s_CurrentProcessId, = GetCurrentProcessImpl());
        return s_CurrentProcessId == processId;
#else
        NN_UNUSED(processId);
        return true;
#endif
    }


    std::shared_ptr<ProgramInfo> ProgramInfo::GetProgramInfoForInitialProcess() NN_NOEXCEPT
    {
        static const unsigned char fac[] =
        {
            0x01, 0x00, 0x00, 0x00, // version
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, // full permission

            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
        };

        static const unsigned char facd[] =
        {
            0x01, 0x00, 0x00, 0x00, // version
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, // full permission
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        };

        struct ProgramInfoForInitialProcessHolder
        {
        public:
            ProgramInfoForInitialProcessHolder()
            {
                class ProgramInfoAllocateHelper : public ProgramInfo
                {
                public:
                    ProgramInfoAllocateHelper(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize)
                        : ProgramInfo(fsAccessControlData, dataSize, fsAccessControlDesc, descSize)
                    {
                    }
                };
                StaticAllocatorForProgramInfoForInitialProcess<char> allocator;
                info = std::allocate_shared<ProgramInfoAllocateHelper>(allocator, fac, sizeof(fac), facd, sizeof(facd));
            }

            std::shared_ptr<ProgramInfo> info;
        };

        static ProgramInfoForInitialProcessHolder s_ProgramInfoForInitialProcess;
        return s_ProgramInfoForInitialProcess.info;
    }

}}}
