﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <cstring>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/crypto/crypto_Compare.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_RightsId.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/spl/spl_Types.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace fssrv { namespace detail {

    class ExternalKeyEntry : public util::IntrusiveListBaseNode<ExternalKeyEntry>, public nn::fs::detail::Newable
    {
    public:
        ExternalKeyEntry(const fs::RightsId& rightsId, const spl::AccessKey& accessKey)
            : m_RightsId(rightsId)
            , m_AccessKey(accessKey)
        {
        }

        bool Contains(const fs::RightsId& rightsId) const NN_NOEXCEPT
        {
            return nn::crypto::IsSameBytes(&m_RightsId, &rightsId, sizeof(fs::RightsId));
        }

        void CopyAccessKey(spl::AccessKey* pOutValue) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(pOutValue != nullptr);
            memcpy(pOutValue, &m_AccessKey, sizeof(spl::AccessKey));
        }

    private:
        fs::RightsId m_RightsId;
        spl::AccessKey m_AccessKey;
    };

    class ExternalKeyManager
    {
        NN_DISALLOW_COPY(ExternalKeyManager);
        NN_DISALLOW_MOVE(ExternalKeyManager);

    public:
        ExternalKeyManager() NN_NOEXCEPT
            : m_Mutex(false)
        {
        }

        Result Register(const fs::RightsId& rightsId, const spl::AccessKey& accessKey)
        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);

            spl::AccessKey existingKey;
            if (FindCore(&existingKey, rightsId).IsSuccess())
            {
                if (nn::crypto::IsSameBytes(&existingKey, &accessKey, sizeof(spl::AccessKey)))
                {
                    // 既に同じ鍵が登録されている場合は何もしない
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    // 異なる鍵が指定された
                    return fs::ResultNcaExternalKeyInconsistent();
                }
            }
            else
            {
                auto pKey = new ExternalKeyEntry(rightsId, accessKey);
                NN_RESULT_THROW_UNLESS(pKey != nullptr, fs::ResultAllocationMemoryFailed());

                m_KeyList.push_back(*pKey);
                NN_RESULT_SUCCESS;
            }
        }

        Result UnregisterAll()
        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            while (!m_KeyList.empty())
            {
                auto* p = &(*m_KeyList.begin());
                m_KeyList.erase(m_KeyList.iterator_to(*p));
                delete p;
            }

            NN_RESULT_SUCCESS;
        }

        Result Find(spl::AccessKey* pOutValue, fs::RightsId rightsId) NN_NOEXCEPT
        {
            std::lock_guard<os::Mutex> scopedLock(m_Mutex);
            return FindCore(pOutValue, rightsId);
        }

    private:
        Result FindCore(spl::AccessKey* pOutValue, fs::RightsId rightsId) NN_NOEXCEPT
        {
            for (const auto& entry : m_KeyList)
            {
                if (entry.Contains(rightsId))
                {
                    entry.CopyAccessKey(pOutValue);
                    NN_RESULT_SUCCESS;
                }
            }

            return fs::ResultNcaExternalKeyUnregistered();
        }


    private:
        typedef util::IntrusiveList<ExternalKeyEntry, util::IntrusiveListBaseNodeTraits<ExternalKeyEntry>> ExternalKeyList;

    private:
        ExternalKeyList m_KeyList;
        os::Mutex m_Mutex;
    };

}}}
