﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/os/os_Event.h>
#include <nn/result/result_HandlingUtility.h>

#include "fssrv_AccessFailureDetectionEventManager.h"
#include "fssrv_AccessFailureDetectionEventNotifier.h"

namespace nn { namespace fssrv { namespace detail {

    AccessFailureDetectionEventManager::AccessFailureDetectionEventManager() NN_NOEXCEPT : m_Mutex(false), m_SystemEvent(os::EventClearMode_ManualClear, true)
    {
    }

    AccessFailureDetectionEventManager::~AccessFailureDetectionEventManager() NN_NOEXCEPT
    {
    }

    Result AccessFailureDetectionEventManager::CreateNotifier(std::unique_ptr<AccessFailureDetectionEventNotifier>* outValue, Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        for (auto& eventNotifier : m_List)
        {
            if (eventNotifier.GetProcessId() == processId)
            {
                return fs::ResultAlreadyExists();
            }
        }

        std::unique_ptr<AccessFailureDetectionEventNotifier> pEventNotifier(new AccessFailureDetectionEventNotifier(this, processId));
        NN_RESULT_THROW_UNLESS(pEventNotifier, fs::ResultAllocationMemoryFailedNew());

        m_List.push_back(*(pEventNotifier.get()));

        *outValue = std::move(pEventNotifier);

        NN_RESULT_SUCCESS;
    }

    // この関数は直接実行しないでください。 AccessFailureDetectionEventNotifier のデストラクタでのみ呼ばれる想定です。
    void AccessFailureDetectionEventManager::DeleteNotifier(AccessFailureDetectionEventNotifier* pEventNotifier) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        m_List.erase(m_List.iterator_to(*pEventNotifier));
    }

    bool AccessFailureDetectionEventManager::IsAccessFailureDetectionObserved(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        for (auto& eventNotifier : m_List)
        {
            if (eventNotifier.GetProcessId() == processId)
            {
                switch (eventNotifier.GetState())
                {
                case AccessFailureDetectionEventNotifier::State::None: return true;
                case AccessFailureDetectionEventNotifier::State::Signaled: return true;
                case AccessFailureDetectionEventNotifier::State::Disabled: return false;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }
        return false;
    }

    void AccessFailureDetectionEventManager::NotifyAccessFailureDetection(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        for (auto& eventNotifier : m_List)
        {
            if (eventNotifier.GetProcessId() == processId)
            {
                switch (eventNotifier.GetState())
                {
                case AccessFailureDetectionEventNotifier::State::None:
                {
                    eventNotifier.SetState(AccessFailureDetectionEventNotifier::State::Signaled);
                    auto pEvent = eventNotifier.GetEvent();
                    if (pEvent != nullptr)
                    {
                        os::SignalSystemEvent(pEvent);
                    }
                    m_SystemEvent.Signal();
                    return;
                }
                case AccessFailureDetectionEventNotifier::State::Signaled: return;
                case AccessFailureDetectionEventNotifier::State::Disabled: return;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }

    void AccessFailureDetectionEventManager::ResetAccessFailureDetection(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        for (auto& eventNotifier : m_List)
        {
            if (eventNotifier.GetProcessId() == processId)
            {
                eventNotifier.SetState(AccessFailureDetectionEventNotifier::State::None);
            }
        }
    }

    void AccessFailureDetectionEventManager::DisableAccessFailureDetection(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        for (auto& eventNotifier : m_List)
        {
            if (eventNotifier.GetProcessId() == processId)
            {
                eventNotifier.SetState(AccessFailureDetectionEventNotifier::State::Disabled);
            }
        }
    }

    bool AccessFailureDetectionEventManager::IsAccessFailureDetectionNotified(Bit64 processId) NN_NOEXCEPT
    {
        std::lock_guard<os::Mutex> scopedLock(m_Mutex);

        for (auto& eventNotifier : m_List)
        {
            if (eventNotifier.GetProcessId() == processId)
            {
                return eventNotifier.GetState() == AccessFailureDetectionEventNotifier::State::Signaled;
            }
        }

        return false;
    }

    os::NativeHandle AccessFailureDetectionEventManager::GetEvent() NN_NOEXCEPT
    {
        return m_SystemEvent.GetReadableHandle();
    }

}}}
