﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include "fssrv_AccessControlBits.h"
#include <nn/fs/detail/fs_Newable.h> // TORIAEZU
#include <nn/util/util_Optional.h>
#include <nn/util/util_IntrusiveList.h>

namespace nn { namespace ncm {
    struct ApplicationId;
}}

namespace nn { namespace fssrv { namespace detail {

    struct Accessibility
    {
        char value;

        bool CanRead() NN_NOEXCEPT
        {
            return ((value >> 0) & 0x1) == 1;
        }

        bool CanWrite() NN_NOEXCEPT
        {
            return ((value >> 1) & 0x1) == 1;
        }

        static Accessibility MakeAccessibility(bool canRead, bool canWrite) NN_NOEXCEPT
        {
            Accessibility accessibility = { static_cast<char>(canRead * (1 << 0) + canWrite * (1 << 1)) };
            return accessibility;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<Accessibility>::value);

    class ContentOwnerInfo : public util::IntrusiveListBaseNode<ContentOwnerInfo>, public nn::fs::detail::Newable
    {
    public:
        NN_IMPLICIT ContentOwnerInfo(Bit64 id) NN_NOEXCEPT : m_Id(id) {}
        Bit64 GetId() const NN_NOEXCEPT { return m_Id; }
    private:
        Bit64 m_Id;
    };

    class SaveDataOwnerInfo : public util::IntrusiveListBaseNode<SaveDataOwnerInfo>, public nn::fs::detail::Newable
    {
    public:
        SaveDataOwnerInfo(Bit64 id, Accessibility accessibility) NN_NOEXCEPT : m_Id(id), m_Accessibility(accessibility) {}
        Bit64 GetId() const NN_NOEXCEPT { return m_Id; }
        Accessibility GetAccessibility() const NN_NOEXCEPT { return m_Accessibility; }
    private:
        Bit64 m_Id;
        Accessibility m_Accessibility;
    };

    class AccessControl
    {
    public:
        AccessControl(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize) NN_NOEXCEPT;
        AccessControl(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize, Bit64 flagDisableMask) NN_NOEXCEPT;
        ~AccessControl() NN_NOEXCEPT;

        bool HasContentOwnerId(Bit64 ownerId) const NN_NOEXCEPT;
        Accessibility GetAccessibilitySaveDataOwnedBy(Bit64 ownerId) const NN_NOEXCEPT;
        void ListSaveDataOwnedId(int32_t* outValue, ncm::ApplicationId* outOwnerId, int32_t offset, int32_t count) const NN_NOEXCEPT;

        enum class AccessibilityType : uint32_t
        {
            MountLogo,
            MountContentMeta,
            MountContentControl,
            MountContentManual,
            MountContentData,
            MountApplicationPackage,
            MountSaveDataStorage,
            MountContentStorage,
            MountImageAndVideoStorage,
            MountCloudBackupWorkStorage,
            MountBisCalibrationFile,
            MountBisSafeMode,
            MountBisUser,
            MountBisSystem,
            MountBisSystemProperEncryption,
            MountBisSystemProperPartition,
            MountSdCard,
            MountGameCard,
            MountDeviceSaveData,
            MountSystemSaveData,
            MountOthersSaveData,
            MountOthersSystemSaveData,
            OpenBisPartitionBootPartition1Root,
            OpenBisPartitionBootPartition2Root,
            OpenBisPartitionUserDataRoot,
            OpenBisPartitionBootConfigAndPackage2Part1,
            OpenBisPartitionBootConfigAndPackage2Part2,
            OpenBisPartitionBootConfigAndPackage2Part3,
            OpenBisPartitionBootConfigAndPackage2Part4,
            OpenBisPartitionBootConfigAndPackage2Part5,
            OpenBisPartitionBootConfigAndPackage2Part6,
            OpenBisPartitionCalibrationBinary,
            OpenBisPartitionCalibrationFile,
            OpenBisPartitionSafeMode,
            OpenBisPartitionUser,
            OpenBisPartitionSystem,
            OpenBisPartitionSystemProperEncryption,
            OpenBisPartitionSystemProperPartition,
            OpenSdCardStorage,
            OpenGameCardStorage,
            MountSystemDataPrivate,
            MountHost,
            MountRegisteredUpdatePartition,
            MountSaveDataInternalStorage,
        };

        Accessibility GetAccessibilityFor(AccessibilityType type) const NN_NOEXCEPT;

        enum class OperationType : uint32_t
        {
            InvalidateBisCache,
            EraseMmc,
            GetGameCardDeviceCertificate,
            GetGameCardIdSet,
            FinalizeGameCardDriver,
            GetGameCardAsicInfo,
            CreateSaveData,
            DeleteSaveData,
            CreateSystemSaveData,
            CreateOthersSystemSaveData,
            DeleteSystemSaveData,
            OpenSaveDataInfoReader,
            OpenSaveDataInfoReaderForSystem,
            OpenSaveDataInfoReaderForInternal,
            OpenSaveDataMetaFile,
            SetCurrentPosixTime,
            ReadSaveDataFileSystemExtraData,
            SetGlobalAccessLogMode,
            SetSpeedEmulationMode,
            Debug_Deprecated,
            FillBis,
            CorruptSaveData,
            CorruptSystemSaveData,
            VerifySaveData,
            DebugSaveData,
            FormatSdCard,
            GetRightsId,
            RegisterExternalKey,
            SetEncryptionSeed,
            WriteSaveDataFileSystemExtraDataTimeStamp,
            WriteSaveDataFileSystemExtraDataFlags,
            WriteSaveDataFileSystemExtraDataCommitId,
            WriteSaveDataFileSystemExtraDataAll,
            ExtendSaveData,
            ExtendSystemSaveData,
            RegisterUpdatePartition,
            OpenSaveDataTransferManager,
            OpenSaveDataTransferManagerVersion2,
            OpenSaveDataTransferProhibiter,
            ListAccessibleSaveDataOwnerId,
            ControlMmcPatrol,
            OverrideSaveDataTransferTokenSignVerificationKey,
            OpenSdCardDetectionEventNotifier,
            OpenGameCardDetectionEventNotifier,
            OpenSystemDataUpdateEventNotifier,
            NotifySystemDataUpdateEvent,
            OpenAccessFailureDetectionEventNotifier,
            GetAccessFailureDetectionEvent,
            IsAccessFailureDetected,
            ResolveAccessFailure,
            AbandonAccessFailure,
            QuerySaveDataInternalStorageTotalSize,
            GetSaveDataCommitId,
            SetSdCardAccessibility,
            SimulateDevice,
            SetDataStorageRedirectTarget,
            CreateSaveDataWithHashSalt,
        };

        bool CanCall(OperationType type) const NN_NOEXCEPT;

    protected:
        const Bit64 GetRawFlagBits() NN_NOEXCEPT
        {
            return m_FlagBits.value().GetValue();
        }

    private:
        nn::util::optional<AccessControlBits> m_FlagBits;

        typedef util::IntrusiveList<ContentOwnerInfo, util::IntrusiveListBaseNodeTraits<ContentOwnerInfo>> ContentOwnerInfoList;
        typedef util::IntrusiveList<SaveDataOwnerInfo, util::IntrusiveListBaseNodeTraits<SaveDataOwnerInfo>> SaveDataOwnerInfoList;

        ContentOwnerInfoList m_ContentOwnerInfos;
        SaveDataOwnerInfoList m_SaveDataOwnerInfos;
    };

    void SetDebugFlagEnabled(bool enabled) NN_NOEXCEPT;

}}}

