﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/crypto.h>
#include <nn/crypto/crypto_Compare.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs {

    namespace {

        const size_t HashSize = crypto::Sha256Generator::HashSize;

        struct SaveDataThumbnailMeta
        {
            char headerDirtyFlag;
            char bodyDirtyFlag;
            char reserved[30];
            char headerHash[HashSize];
            char bodyHash[HashSize];
        };
        NN_STATIC_ASSERT(std::is_pod<SaveDataThumbnailMeta>::value);

        const size_t EntityOffset = sizeof(SaveDataThumbnailMeta);

        Result ReadMeta(void* outValue, nn::fs::fsa::IFile* file)
        {
            size_t readSize = 0;
            NN_RESULT_THROW_UNLESS(outValue != nullptr, nn::fs::ResultNullptrArgument());
            NN_FS_RESULT_DO(file->Read(&readSize, 0, outValue, sizeof(SaveDataThumbnailMeta), nn::fs::ReadOption::MakeValue(0)));
            NN_SDK_ASSERT_EQUAL(sizeof(SaveDataThumbnailMeta), readSize);
            NN_RESULT_SUCCESS;
        }

        Result ReadAndCheckHash(void *outValue, nn::fs::fsa::IFile* file, const void* hashExpected, size_t dataOffset, size_t size) NN_NOEXCEPT
        {
            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }

            size_t readSize = 0;

            NN_FS_RESULT_DO(file->Read(&readSize, EntityOffset + dataOffset, outValue, size, nn::fs::ReadOption::MakeValue(0)));
            NN_SDK_ASSERT_EQUAL(size, readSize);

            char hashActual[HashSize];
            crypto::GenerateSha256Hash(hashActual, sizeof(hashActual), outValue, size);

            if( nn::crypto::IsSameBytes(hashActual, hashExpected, sizeof(hashActual)) )
            {
                NN_RESULT_SUCCESS;
            }
            else
            {
                // ファイルが壊れている場合ゼロ埋めする
                memset(outValue, 0x00, size);
                return ResultThumbnailHashVerificationFailed();
            }
        }

        Result WriteAndCalcHash(void* outValue, nn::fs::fsa::IFile* file, const void* buffer, size_t dataOffset, size_t size) NN_NOEXCEPT
        {
            if( size == 0 )
            {
                NN_RESULT_SUCCESS;
            }

            NN_RESULT_THROW_UNLESS(buffer != nullptr, nn::fs::ResultNullptrArgument());
            crypto::GenerateSha256Hash(outValue, HashSize, buffer, size);
            NN_FS_RESULT_DO(file->Write(EntityOffset + dataOffset, buffer, size, nn::fs::WriteOption::MakeValue(0)));

            NN_RESULT_SUCCESS;
        }
    }

Result OpenSaveDataThumbnailFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, Bit64 applicationId, UserId& userId) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = detail::GetFileSystemProxyServiceObject();
    nn::ncm::ProgramId programId = { applicationId };
    auto attribute = SaveDataAttribute::Make(programId, SaveDataType::Account, userId, InvalidSystemSaveDataId);

    nn::sf::SharedPointer<nn::fssrv::sf::IFile> file;
    NN_FS_RESULT_DO(fileSystemProxy->OpenSaveDataMetaFile(&file, static_cast<uint32_t>(SaveDataSpaceId::User), attribute, static_cast<uint32_t>(nn::fs::SaveDataMetaType::Thumbnail)));

    std::unique_ptr<nn::fs::fsa::IFile> pFile(new detail::FileServiceObjectAdapter(std::move(file)));
    NN_RESULT_THROW_UNLESS(pFile, ResultAllocationMemoryFailedInSaveDataThumbnailA());

    *outValue = std::move(pFile);
    NN_RESULT_SUCCESS;
}


Result ReadSaveDataThumbnailFile(Bit64 applicationId, UserId userId , void* headerData, size_t headerSize, void* bodyData, size_t bodySize) NN_NOEXCEPT
{
    if( headerSize == 0 && bodySize == 0)
    {
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<nn::fs::fsa::IFile> file;
    NN_FS_RESULT_DO(OpenSaveDataThumbnailFile(&file, applicationId, userId));

    SaveDataThumbnailMeta meta;
    NN_RESULT_DO(ReadMeta(&meta, file.get()));

    // header
    if( meta.headerDirtyFlag == 0 )
    {
        // 未書き込み状態はゼロ埋めとみなす
        memset(headerData, 0x00, headerSize);
    }
    else
    {
        NN_RESULT_TRY(ReadAndCheckHash(headerData, file.get(), meta.headerHash, 0, headerSize))
            NN_RESULT_CATCH(ResultThumbnailHashVerificationFailed)
            {
                memset(bodyData, 0x00, bodySize);
                NN_RESULT_RETHROW;
            }
        NN_RESULT_END_TRY;
    }

    // body
    if( meta.bodyDirtyFlag == 0 )
    {
        memset(bodyData, 0x00, bodySize);
    }
    else
    {
        NN_RESULT_DO(ReadAndCheckHash(bodyData, file.get(), meta.bodyHash, headerSize, bodySize)); // NN_FS_RESULT_DO 使用不可
    }

    NN_RESULT_SUCCESS;
}

Result ReadSaveDataThumbnailFileHeader(Bit64 applicationId, UserId userId , void* headerData, size_t headerSize) NN_NOEXCEPT
{
    if( headerSize == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<nn::fs::fsa::IFile> file;
    NN_FS_RESULT_DO(OpenSaveDataThumbnailFile(&file, applicationId, userId));

    SaveDataThumbnailMeta meta;
    NN_RESULT_DO(ReadMeta(&meta, file.get()));

    if( meta.headerDirtyFlag == 0 )
    {
        memset(headerData, 0x00, headerSize);
        NN_RESULT_SUCCESS;
    }

    return ReadAndCheckHash(headerData, file.get(), meta.headerHash, 0, headerSize);
}

Result WriteSaveDataThumbnailFile(Bit64 applicationId, UserId userId , const void* headerData, size_t headerSize, const void* bodyData, size_t bodySize) NN_NOEXCEPT
{
    if( headerSize == 0 && bodySize == 0)
    {
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<nn::fs::fsa::IFile> file;
    NN_FS_RESULT_DO(OpenSaveDataThumbnailFile(&file, applicationId, userId));

    SaveDataThumbnailMeta meta;

    // header
    NN_FS_RESULT_DO(WriteAndCalcHash(meta.headerHash, file.get(), headerData, 0, headerSize));
    meta.headerDirtyFlag = 0x1;

    // body
    NN_FS_RESULT_DO(WriteAndCalcHash(meta.bodyHash, file.get(), bodyData, headerSize, bodySize));
    meta.bodyDirtyFlag = 0x1;

    NN_FS_RESULT_DO(file->Write(0, &meta, sizeof(meta), nn::fs::WriteOption::MakeValue(WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

Result WriteSaveDataThumbnailFileHeader(Bit64 applicationId, UserId userId , const void* headerData, size_t headerSize) NN_NOEXCEPT
{
    if( headerSize == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    std::unique_ptr<nn::fs::fsa::IFile> file;
    NN_FS_RESULT_DO(OpenSaveDataThumbnailFile(&file, applicationId, userId));

    SaveDataThumbnailMeta meta;
    NN_RESULT_DO(ReadMeta(&meta, file.get()));

    NN_FS_RESULT_DO(WriteAndCalcHash(meta.headerHash, file.get(), headerData, 0, headerSize));
    meta.headerDirtyFlag = 0x1;

    NN_FS_RESULT_DO(file->Write(0, &meta, sizeof(meta), nn::fs::WriteOption::MakeValue(WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}

Result CorruptSaveDataThumbnailFileForDebug(Bit64 applicationId, UserId userId) NN_NOEXCEPT
{
    std::unique_ptr<nn::fs::fsa::IFile> file;
    NN_FS_RESULT_DO(OpenSaveDataThumbnailFile(&file, applicationId, userId));

    char hash[HashSize] = {0x1};
    NN_FS_RESULT_DO(file->Write(HashSize, hash, sizeof(hash), WriteOption::MakeValue(WriteOptionFlag_Flush)));

    NN_RESULT_SUCCESS;
}


}}

