﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_RomOnFile.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_RomFsFileSystem.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include "fsa/fs_MountUtility.h"

namespace nn { namespace fs {

namespace {

    Result MountRomOnFileImpl(
        const char* name, FileHandle fileHandle, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize
    ) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

        std::unique_ptr<IStorage> storage(new FileHandleStorage(fileHandle));
        NN_RESULT_THROW_UNLESS(storage, ResultAllocationMemoryFailedInRomOnFileA());

        std::unique_ptr<RomFsFileSystem> fileSystem(new RomFsFileSystem());
        NN_RESULT_THROW_UNLESS(fileSystem, ResultAllocationMemoryFailedInRomOnFileB());

        NN_RESULT_DO(
            fileSystem->Initialize(
                std::move(storage),
                pFileSystemCacheBuffer,
                fileSystemCacheBufferSize,
                true
            )
        );
        return fsa::Register(name, std::move(fileSystem));
    }
}

Result QueryMountRomOnFileCacheSize(size_t* pOutValue, FileHandle fileHandle) NN_NOEXCEPT
{
    auto query = [=]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

        std::unique_ptr<IStorage> storage(new FileHandleStorage(fileHandle));
        NN_RESULT_THROW_UNLESS(storage, ResultAllocationMemoryFailedInRomOnFileC());

        NN_RESULT_DO(RomFsFileSystem::GetRequiredWorkingMemorySize(pOutValue, storage.get()));

        // サイズ 0 でメモリ確保すると問題のある環境があるため 0 ではない値を返します。
        static const size_t MinCacheSize = 32;
        if( *pOutValue < MinCacheSize )
        {
            *pOutValue = MinCacheSize;
        }
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM(query(),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_FILEHANDLE NN_DETAIL_FS_ACCESS_LOG_FORMAT_QUERY_SIZE, fileHandle, detail::ReferenceQuerySizeOutValue(pOutValue)));
    NN_RESULT_SUCCESS;
}

Result MountRomOnFile(
    const char* name, FileHandle fileHandle, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize
) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_DO(detail::CheckMountName(name)); // マウント名チェック
        NN_RESULT_THROW_UNLESS(pFileSystemCacheBuffer != nullptr, ResultNullptrArgument());

        NN_RESULT_DO(MountRomOnFileImpl(name, fileHandle, pFileSystemCacheBuffer, fileSystemCacheBufferSize));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTROMONFILE(name, fileHandle)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

}}

