﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_Rom.h>
#include <nn/fs/fs_RomPrivate.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_RomFsFileSystem.h>
#include <nn/fs/fs_ReadOnlyFileSystem.h>
#include <nn/fs/fs_PathUtility.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/detail/fs_Log.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/fs/detail/fs_OpenHostFileSystem.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssystem/fs_PathOnExecutionDirectory.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/util/util_FormatString.h>

#include "fs_Library.h"
#include "fs_StorageServiceObjectAdapter.h"
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fs_FileSystemProxyServiceObject.h"
#include "fsa/fs_FileSystemAccessor.h"
#include "fsa/fs_MountUtility.h"
#include "fsa/fs_UserMountTable.h"

namespace {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    const char* GetDataPathOnExecutionDirectory()
    {
        static const nn::fssystem::PathOnExecutionDirectory g_DataPathOnExecutionDirectory("data");
        return g_DataPathOnExecutionDirectory.Get();
    }
#endif
}

namespace nn { namespace fs {

namespace {

    Result OpenDataStorageImpl(std::unique_ptr<nn::fs::IStorage>* outValue, nn::ApplicationId applicationId) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy
            = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IStorage> storage;

        if (applicationId == nn::ApplicationId::GetInvalidId())
        {
            NN_RESULT_DO(fileSystemProxy->OpenDataStorageByCurrentProcess(&storage));
        }
        else
        {
            nn::ncm::ProgramId programId = { applicationId.value };
            NN_RESULT_DO(fileSystemProxy->OpenDataStorageByProgramId(&storage, programId));
        }

        std::unique_ptr<nn::fs::IStorage> storageAbstract;
        storageAbstract.reset(new detail::StorageServiceObjectAdapter(std::move(storage)));
        NN_RESULT_THROW_UNLESS(storageAbstract, ResultAllocationMemoryFailedInRomA());

        *outValue = std::move(storageAbstract);
        NN_RESULT_SUCCESS;
    }

    Result OpenDataFileSystemImpl(std::unique_ptr<nn::fs::fsa::IFileSystem>* outValue, nn::ApplicationId applicationId) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy
            = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;

        if (applicationId == nn::ApplicationId::GetInvalidId())
        {
            NN_RESULT_DO(fileSystemProxy->OpenDataFileSystemByCurrentProcess(&fileSystem));
        }
        else
        {
            nn::ncm::ProgramId programId = { applicationId.value };
            NN_RESULT_DO(fileSystemProxy->OpenDataFileSystemByProgramId(&fileSystem, programId));
        }

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInRomB());

        std::unique_ptr<fsa::IFileSystem> readOnlyFileSystem;
        readOnlyFileSystem.reset(new ReadOnlyFileSystem(std::move(fileSystemAbstract)));
        NN_RESULT_THROW_UNLESS(readOnlyFileSystem, ResultAllocationMemoryFailedInRomG());

        *outValue = std::move(readOnlyFileSystem);
        NN_RESULT_SUCCESS;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    Result OpenHostFileSystemForRom(std::unique_ptr<nn::fs::fsa::IFileSystem>* outValue) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

        nn::fssrv::sf::FspPath sfPath;
        NN_RESULT_DO(FspPathPrintf(&sfPath, "/%s", GetDataPathOnExecutionDirectory()));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy
            = detail::GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
        NN_RESULT_DO(fileSystemProxy->OpenHostFileSystem(&fileSystem, sfPath));

        std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
        fileSystemAbstract.reset(new detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailedInRomC());

        std::unique_ptr<fsa::IFileSystem> readOnlyFileSystem;
        readOnlyFileSystem.reset(new ReadOnlyFileSystem(std::move(fileSystemAbstract)));
        NN_RESULT_THROW_UNLESS(readOnlyFileSystem, ResultAllocationMemoryFailedInRomD());

        *outValue = std::move(readOnlyFileSystem);
        NN_RESULT_SUCCESS;
    }

#endif

    Result QueryMountRomCacheSizeImpl(size_t* pOutValue, nn::ApplicationId applicationId) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）
        NN_RESULT_THROW_UNLESS(pOutValue != nullptr, ResultNullptrArgument());

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_UNUSED(applicationId);
        *pOutValue = 0;
#else
        std::unique_ptr<IStorage> storage;
        auto result = OpenDataStorageImpl(&storage, applicationId);
        if( result.IsSuccess() )
        {
            size_t size = 0;
            NN_RESULT_DO(RomFsFileSystem::GetRequiredWorkingMemorySize(&size, storage.get()));
            *pOutValue = size;
        }
        else if( ResultTargetNotFound::Includes(result) || ResultPartitionNotFound::Includes(result) || ResultProgramInfoNotFound::Includes(result) )
        {
            *pOutValue = 0;
        }
        else
        {
            NN_RESULT_DO(result);
        }
#endif
        // サイズ 0 でメモリ確保すると問題のある環境があるため 0 ではない値を返します。
        static const size_t MinCacheSize = 32;
        if( *pOutValue < MinCacheSize )
        {
            *pOutValue = MinCacheSize;
        }
        NN_RESULT_SUCCESS;
    }

    Result MountRomImpl(const char* name, nn::ApplicationId applicationId, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize) NN_NOEXCEPT
    {
        // ユーザーから直接呼び出されないため、この関数内では NN_FS_RESULT_DO を使いません。
        // （アクセスログを追加した時にアクセスログ出力前にアボートしてしまうため）

#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NN_UNUSED(pFileSystemCacheBuffer);
        NN_UNUSED(fileSystemCacheBufferSize);

        if (applicationId != nn::ApplicationId::GetInvalidId())
        {
            return nn::fs::ResultNotImplemented();
        }

        std::unique_ptr<nn::fs::fsa::IFileSystem> hostFileSystem;
        NN_RESULT_DO(
            nn::fs::detail::OpenHostFileSystem(
                &hostFileSystem, name, GetDataPathOnExecutionDirectory()));

        std::unique_ptr<nn::fs::fsa::IFileSystem> readOnlyFileSystem(
            new ReadOnlyFileSystem(std::move(hostFileSystem)));
        NN_RESULT_THROW_UNLESS(readOnlyFileSystem, ResultAllocationMemoryFailedInRomE());

        return nn::fs::fsa::Register(name, std::move(readOnlyFileSystem));
#else
        NN_RESULT_THROW_UNLESS(pFileSystemCacheBuffer != nullptr, ResultNullptrArgument());

        std::unique_ptr<IStorage> storage;
        Result result = OpenDataStorageImpl(&storage, applicationId);
        if( result.IsSuccess() )
        {
            // nca 形式の実行
            std::unique_ptr<RomFsFileSystem> fileSystem(new RomFsFileSystem());
            NN_RESULT_THROW_UNLESS(fileSystem != nullptr, ResultAllocationMemoryFailedInRomF());
            NN_RESULT_DO(
                fileSystem->Initialize(
                    std::move(storage),
                    pFileSystemCacheBuffer,
                    fileSystemCacheBufferSize,
                    true
                )
            );
            return fsa::Register(name, std::move(fileSystem), nullptr, true, true, false);
        }
        else if( ResultTargetNotFound::Includes(result) )
        {
            // Raw 形式の実行
            std::unique_ptr<fsa::IFileSystem> fileSystem;
            result = OpenDataFileSystemImpl(&fileSystem, applicationId);
            if( result.IsSuccess() )
            {
                return fsa::Register(name, std::move(fileSystem));
            }
        }

        return result;
#endif
    }

    bool CanMountRomImpl(nn::ApplicationId applicationId) NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        if (applicationId != nn::ApplicationId::GetInvalidId())
        {
            return false; // TODO: Windows 版のケア
        }
        std::unique_ptr<fsa::IFileSystem> fileSystem;
        auto result = OpenHostFileSystemForRom(&fileSystem);
#else
        std::unique_ptr<IStorage> storage;
        auto result = OpenDataStorageImpl(&storage, applicationId);
        if( ResultTargetNotFound::Includes(result) )
        {
            // Raw 形式
            std::unique_ptr<fsa::IFileSystem> fileSystem;
            result = OpenDataFileSystemImpl(&fileSystem, applicationId);
        }
#endif
        if( result.IsSuccess() )
        {
            return true;
        }
        else
        {
            return false;
        }
    }
}

Result QueryMountRomCacheSize(size_t* pOutValue) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(QueryMountRomCacheSizeImpl(pOutValue, nn::ApplicationId::GetInvalidId()),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_QUERY_SIZE, detail::ReferenceQuerySizeOutValue(pOutValue)));
    NN_RESULT_SUCCESS;
}

Result QueryMountRomCacheSize(size_t* pOutValue, nn::ApplicationId applicationId) NN_NOEXCEPT
{
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(QueryMountRomCacheSizeImpl(pOutValue, applicationId),
        nullptr, NN_DETAIL_FS_ACCESS_LOG_FORMAT_APPLICATIONID NN_DETAIL_FS_ACCESS_LOG_FORMAT_QUERY_SIZE, applicationId, detail::ReferenceQuerySizeOutValue(pOutValue)));
    NN_RESULT_SUCCESS;
}

Result MountRom(const char* name, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_DO(detail::CheckMountName(name)); // マウント名チェック
        NN_RESULT_THROW_UNLESS(pFileSystemCacheBuffer != nullptr, ResultNullptrArgument());
        NN_RESULT_DO(MountRomImpl(name, nn::ApplicationId::GetInvalidId(), pFileSystemCacheBuffer, fileSystemCacheBufferSize));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountRom(const char* name, nn::ApplicationId applicationId, void* pFileSystemCacheBuffer, size_t fileSystemCacheBufferSize) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        NN_RESULT_DO(detail::CheckMountName(name)); // マウント名チェック
        NN_RESULT_THROW_UNLESS(pFileSystemCacheBuffer != nullptr, ResultNullptrArgument());
        NN_RESULT_DO(MountRomImpl(name, applicationId, pFileSystemCacheBuffer, fileSystemCacheBufferSize));
        NN_RESULT_SUCCESS;
    };
    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    NN_DETAIL_FS_ACCESS_LOG_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

bool CanMountRomForDebug() NN_NOEXCEPT
{
    return CanMountRomImpl(nn::ApplicationId::GetInvalidId());
}

bool CanMountRom(nn::ApplicationId applicationId) NN_NOEXCEPT
{
    return CanMountRomImpl(applicationId);
}

}}

