﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Priority.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_PriorityUtility.h>
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include <nn/os/os_SdkThreadLocalStorage.h>
#include <nn/sf/sf_FsInlineContext.h>

namespace nn { namespace fs {

namespace
{
    PriorityRaw ConvertPriorityToPriorityRaw(fs::Priority priority) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(
            priority == Priority_Low
            || priority == Priority_Normal
            || priority == Priority_Realtime
        );
        switch( priority )
        {
        case Priority_Low:
            return PriorityRaw_Low;
        case Priority_Normal:
            return PriorityRaw_Normal;
        case Priority_Realtime:
            return PriorityRaw_Realtime;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    Priority ConvertPriorityRawToPriority(fs::PriorityRaw priorityRaw) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(
            priorityRaw == PriorityRaw_Background
            || priorityRaw == PriorityRaw_Low
            || priorityRaw == PriorityRaw_Normal
            || priorityRaw == PriorityRaw_Realtime
        );
        switch( priorityRaw )
        {
        case PriorityRaw_Low:
        case PriorityRaw_Background:
            return Priority_Low;
        case PriorityRaw_Normal:
            return Priority_Normal;
        case PriorityRaw_Realtime:
            return Priority_Realtime;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void UpdateTlsIoPriority(uint8_t tlsIoPriority) NN_NOEXCEPT
    {
        sf::SetFsInlineContext((tlsIoPriority & detail::TlsIoPriorityMask) | (sf::GetFsInlineContext() & (~detail::TlsIoPriorityMask)));
    }
}

void SetPriorityOnCurrentThread(Priority priority) NN_NOEXCEPT
{
    auto setPriority = [&]() NN_NOEXCEPT -> Result
    {
        uint8_t tlsIoPriority;
        if(priority != Priority_Low
            && priority != Priority_Normal
            && priority != Priority_Realtime)
        {
            NN_RESULT_THROW(ResultInvalidArgument());
        }
        PriorityRaw priorityRaw = ConvertPriorityToPriorityRaw(priority);
        NN_RESULT_DO(detail::ConvertFsPriorityToTlsIoPriority(&tlsIoPriority, priorityRaw));
        UpdateTlsIoPriority(tlsIoPriority);
        NN_RESULT_SUCCESS;
    };
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY(
            setPriority(), priority, ""
        )
    );
}

Priority GetPriorityOnCurrentThread() NN_NOEXCEPT
{
    Priority priority;
    NN_DETAIL_FS_ACCESS_LOG(
        [&]() NN_NOEXCEPT -> Result
        {
            PriorityRaw priorityRaw;
            // アプリ開発者の実装ミスで ConvertTlsIoPriorityToFsPriority がエラーリザルトを返すことはないので ABORT する
            NN_FS_ABORT_UNLESS_RESULT_SUCCESS(detail::ConvertTlsIoPriorityToFsPriority(&priorityRaw, detail::GetTlsIoPriority()));
            priority = ConvertPriorityRawToPriority(priorityRaw);
            NN_RESULT_SUCCESS;
        }(),
        nullptr, ""
    );
    return priority;
}

void SetPriorityRawOnCurrentThread(PriorityRaw  priorityRaw) NN_NOEXCEPT
{
    auto setPriorityRaw = [&]() NN_NOEXCEPT -> Result
    {
        uint8_t tlsIoPriority;
        if(priorityRaw != PriorityRaw_Background
            && priorityRaw != PriorityRaw_Low
            && priorityRaw != PriorityRaw_Normal
            && priorityRaw != PriorityRaw_Realtime)
        {
            NN_RESULT_THROW(ResultInvalidArgument());
        }
        NN_RESULT_DO(detail::ConvertFsPriorityToTlsIoPriority(&tlsIoPriority, priorityRaw));
        UpdateTlsIoPriority(tlsIoPriority);
        NN_RESULT_SUCCESS;
    };
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
        NN_DETAIL_FS_ACCESS_LOG_WITH_PRIORITY_SYSTEM(
            setPriorityRaw(), priorityRaw, ""
        )
    );
}

PriorityRaw GetPriorityRawOnCurrentThreadImpl() NN_NOEXCEPT
{
    PriorityRaw priority;
    // アプリ開発者の実装ミスで ConvertTlsIoPriorityToFsPriority がエラーリザルトを返すことはないので ABORT する
    NN_FS_ABORT_UNLESS_RESULT_SUCCESS(detail::ConvertTlsIoPriorityToFsPriority(&priority, detail::GetTlsIoPriority()));
    return priority;
}

PriorityRaw GetPriorityRawOnCurrentThread() NN_NOEXCEPT
{
    PriorityRaw priority;
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM(
        [&]() NN_NOEXCEPT -> Result
        {
            priority = GetPriorityRawOnCurrentThreadImpl();
            NN_RESULT_SUCCESS;
        }(),
        nullptr, ""
    );
    return priority;
}

}}
