﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#if defined(NN_BUILD_CONFIG_OS_WIN32)
#include <nn/nn_Allocator.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fssrv/fssrv_MemoryResourceFromExpHeap.h>
#include <nn/fssrv/fssrv_SaveDataIndexer.h>
#include <nn/fssrv/fssrv_SaveDataIndexerManager.h>
#include <nn/fssrv/fssrv_FileSystemProxyImpl.h>
#endif
#include "fs_Library.h"
#include "fs_FileSystemCreatorInterfaces.h"
#include <nn/util/util_Optional.h>

namespace nn { namespace fs {

namespace detail {
namespace {

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    const int ExpHeapSize = (4 * 1024 + 512) * 1024; // FS プロセスと合わせる。TODO: サイズ削減
    NN_ALIGNAS(4096) char g_ExpHeapStack[ExpHeapSize];
    nn::lmem::HeapHandle g_ExpHeapHandle = nullptr;
    nn::os::MutexType g_ExpHeapMutex = NN_OS_MUTEX_INITIALIZER(false);
    size_t g_ExpHeapSizeFreePeak = ExpHeapSize;

    void InitializeExpHeap()
    {
        NN_SDK_ASSERT(nn::os::IsMutexLockedByCurrentThread(&g_ExpHeapMutex));
        if( g_ExpHeapHandle == nullptr )
        {
            g_ExpHeapHandle = nn::lmem::CreateExpHeap(&g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill);
            NN_ABORT_UNLESS(g_ExpHeapHandle);
        }
    }

    void* Allocate(size_t size)
    {
        nn::os::LockMutex(&g_ExpHeapMutex);
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::UnlockMutex(&g_ExpHeapMutex);
        };

        if( g_ExpHeapHandle == nullptr )
        {
            InitializeExpHeap();
        }
        void* buffer = nn::lmem::AllocateFromExpHeap(g_ExpHeapHandle, size);
        const size_t freeSize = nn::lmem::GetExpHeapTotalFreeSize(g_ExpHeapHandle);
        if( freeSize < g_ExpHeapSizeFreePeak )
        {
            g_ExpHeapSizeFreePeak = freeSize;
        }
        return buffer;
    }

    void Deallocate(void* p, size_t size)
    {
        nn::os::LockMutex(&g_ExpHeapMutex);
        NN_UTIL_SCOPE_EXIT
        {
            nn::os::UnlockMutex(&g_ExpHeapMutex);
        };

        NN_UNUSED(size);
        if( g_ExpHeapHandle == nullptr )
        {
            InitializeExpHeap();
        }
        return nn::lmem::FreeToExpHeap(g_ExpHeapHandle, p);
    }

    class MemoryResource : public nn::MemoryResource
    {
    protected:
        virtual void* do_allocate(std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(alignment);
            return Allocate(bytes);
        }
        virtual void do_deallocate(void* p, std::size_t bytes, std::size_t alignment) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(alignment);
            return Deallocate(p, bytes);
        }
        virtual bool do_is_equal(const nn::MemoryResource& other) const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(other);
            return false;
        }
    };

    fssrv::DeviceHandle g_SdCardHandle = 1;

    class SdHandleManager : public nn::fssrv::IDeviceHandleManager
    {
    public:
        virtual nn::Result GetHandle(nn::fssrv::DeviceHandle* pOutValue) NN_NOEXCEPT NN_OVERRIDE
        {
            *pOutValue = g_SdCardHandle;
            NN_RESULT_SUCCESS;
        }
        virtual bool IsValid(nn::fssrv::DeviceHandle handle) NN_NOEXCEPT NN_OVERRIDE
        {
            return (handle == g_SdCardHandle);
        }
    };
    SdHandleManager g_SdHandleManager;


#endif

    class FileSystemLibraryInitializer
    {
    public:
        FileSystemLibraryInitializer()
        {
            nn::fs::detail::InitializeFileSystemCreatorInterfaces();
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            const int SaveDataFileSystemCacheCount = 0;
            nn::fssrv::InitializeSaveDataFileSystemCacheManager(SaveDataFileSystemCacheCount);

            static nn::util::TypedStorage<MemoryResource, sizeof(MemoryResource), NN_ALIGNOF(MemoryResource)> s_AllocatorForIndexer;
            const nn::fs::StaticSaveDataId SaveDataIndexerId = 0x8000000000000000;
            new(&s_AllocatorForIndexer.storage) MemoryResource();
            nn::fssrv::InitializeSaveDataIndexerManager(SaveDataIndexerId, &Get(s_AllocatorForIndexer), &g_SdHandleManager, false);
#endif
        }
    };
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // TORIAEZU: sd カード挿抜 + 再 OpenSdStorage() を模倣
    void InvalidateSdCardForWin() NN_NOEXCEPT
    {
        g_SdCardHandle++;
    }
#endif

    void InitializeFileSystemLibrary() NN_NOEXCEPT
    {
        static FileSystemLibraryInitializer s_Initializer;
    }

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    size_t GetExpHeapFreeSizePeak() NN_NOEXCEPT
    {
        return g_ExpHeapSizeFreePeak;
    }

    void ClearExpHeapFreeSizePeak() NN_NOEXCEPT
    {
        g_ExpHeapSizeFreePeak = ExpHeapSize;
    }
#endif

}

}}
