﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystemProxy.h>
#include <nn/util/util_FormatString.h>
#include <nn/os/os_ThreadApi.h>

#include "fs_Library.h"
#include "fs_FileSystemProxyServiceObject.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>
#include "fs_FileSystemServiceObjectAdapter.h"
#include "fsa/fs_MountUtility.h"

using namespace nn::fs::detail;

namespace nn { namespace fs {

namespace {

class ContentStorageCommonMountNameGenerator : public fsa::ICommonMountNameGenerator, public Newable
{
public:
    explicit ContentStorageCommonMountNameGenerator(ContentStorageId id)
        : m_Id(id)
    {
    }

    virtual Result GenerateCommonMountName(char* name, size_t nameSize) NN_NOEXCEPT NN_OVERRIDE
    {
        const size_t RequiredNameBufferSize = strnlen(GetContentStorageMountName(m_Id), MountNameLengthMax) + 1 + 1;
        NN_SDK_REQUIRES(nameSize >= static_cast<size_t>(RequiredNameBufferSize));
        NN_UNUSED(RequiredNameBufferSize);
        auto size = nn::util::SNPrintf(name, nameSize, "%s:", GetContentStorageMountName(m_Id));
        NN_SDK_ASSERT(static_cast<size_t>(size) == RequiredNameBufferSize - 1);
        NN_UNUSED(size);

        NN_RESULT_SUCCESS;
    }

private:
    const ContentStorageId m_Id;
};

}

Result MountContentStorage(const char* name, ContentStorageId id) NN_NOEXCEPT
{
    auto mount = [=]() NN_NOEXCEPT -> Result
    {
        // マウント名チェック
        NN_RESULT_DO(detail::CheckMountNameAcceptingReservedMountName(name));

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = GetFileSystemProxyServiceObject();
        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;

        const int RetryCountMax = 10;
        const int RetryIntervalMsec = 1000;

        // SignedSystemPartition on SD 向けのリトライ
        for (int i = 0; i < RetryCountMax; i++)
        {
            auto result = fileSystemProxy->OpenContentStorageFileSystem(nn::sf::Out<nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem>>(&fileSystem), static_cast<uint32_t>(id));
            NN_RESULT_TRY(result)
                NN_RESULT_CATCH(ResultSystemPartitionNotReady)
                {
                    if (i == RetryCountMax)
                    {
                        NN_RESULT_RETHROW;
                    }
                    else
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryIntervalMsec));
                        continue;
                    }
                }
            NN_RESULT_END_TRY
        }

        std::unique_ptr<fsa::IFileSystem> fileSystemAdapter(new FileSystemServiceObjectAdapter(std::move(fileSystem)));
        NN_RESULT_THROW_UNLESS(fileSystemAdapter.get() != nullptr, ResultAllocationMemoryFailedInContentStorageA());

        std::unique_ptr<ContentStorageCommonMountNameGenerator> mountNameGenerator(new ContentStorageCommonMountNameGenerator(id));
        NN_RESULT_THROW_UNLESS(mountNameGenerator.get() != nullptr, ResultAllocationMemoryFailedInContentStorageB());

        NN_RESULT_DO(fsa::Register(name, std::move(fileSystemAdapter), std::move(mountNameGenerator)));
        NN_RESULT_SUCCESS;
    };

    NN_FS_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG_SYSTEM_MOUNT(mount(),
        name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNTCONTENTSTORAGE(name, id)));
    NN_DETAIL_FS_ACCESS_LOG_SYSTEM_FSACCESSOR_ENABLE(name);
    NN_RESULT_SUCCESS;
}

Result MountContentStorage(ContentStorageId id) NN_NOEXCEPT
{
    return MountContentStorage(GetContentStorageMountName(id), id);
}

const char* GetContentStorageMountName(ContentStorageId id) NN_NOEXCEPT
{
    switch (id)
    {
    case ContentStorageId::System:
        return detail::ContentStorageSystemMountName;
    case ContentStorageId::User:
        return detail::ContentStorageUserMountName;
    case ContentStorageId::SdCard:
        return detail::ContentStorageSdCardMountName;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}

