﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Abort.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fs/detail/fs_CommonMountName.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Utf8StringUtil.h>
#include <nn/fs/detail/fs_AccessLog.h>

#include "shim/fs_FileDataCacheAccessor.h"
#include "shim/fs_FileDataCacheDetail.h"
#include "shim/fs_Library.h"
#include "fs_FileSystemAccessor.h"
#include "fs_MountUtility.h"
#include "fs_UserMountTable.h"
#include <nn/fs/detail/fs_ResultHandlingUtility.h>

namespace nn { namespace fs { namespace detail {

    MountName GetMountName(const char* path) NN_NOEXCEPT
    {
        MountName mountName;

        if( IsWindowsDrive(path) )
        {
            NN_SDK_ASSERT(strnlen(HostRootFileSystemMountName, MountName::LENGTH) < MountName::LENGTH, "HostRootFileSystemName is too long.\n");
            strncpy(mountName.string, HostRootFileSystemMountName, MountName::LENGTH - 1);
            mountName.string[MountName::LENGTH - 1] = '\0';
            return mountName;
        }
        else
        {
            const char* delimiter = [](
                const char* pStr,
                int value,
                size_t lenMax) NN_NOEXCEPT -> const char*
            {
                for( auto pChar = pStr; pChar < pStr + lenMax; ++pChar )
                {
                    if( *pChar == value )
                    {
                        return pChar;
                    }
                    else if( *pChar == '\0' )
                    {
                        break;
                    }
                }
                return nullptr;
            }(path, ':', MountNameLengthMax + 1);
            NN_FS_ABORT_UNLESS_WITH_RESULT( delimiter != nullptr, ResultInvalidMountName(), "Path doesn't include delimiter ':'.\n");

            intptr_t length = delimiter - path;
            NN_FS_ABORT_UNLESS_WITH_RESULT( length + 1 <= MountName::LENGTH, ResultInvalidMountName(), "Mount name is too long.\n");

            memcpy(mountName.string, path, length);
            mountName.string[length] = '\0';
            return mountName;
        }
    }

    bool IsWindowsDrive(const char* name) NN_NOEXCEPT
    {
        return
        ((('a' <= name[0] && name[0] <= 'z') ||
        ('A' <= name[0] && name[0] <= 'Z')) &&
        (name[1] == ':'));
    }

    bool IsValidMountName(const char* name) NN_NOEXCEPT
    {
        // 終端の NULL 文字を含めず 1 バイト以上である。
        if( name[0] == '\0' )
        {
            return false;
        }

        // アルファベット一文字でない。
        if( (('a' <= name[0] && name[0] <= 'z') ||
            ('A' <= name[0] && name[0] <= 'Z')) &&
            (name[1] == '\0') )
        {
            return false;
        }

        // ':' および '/' を含まない。
        size_t length = 0;
        for( auto pChar = name; *pChar != '\0'; ++pChar )
        {
            if( *pChar == ':' || *pChar == '/' )
            {
                return false;
            }
            ++length;

            // nn::fs::MountNameLengthMax バイト以下の文字列である。
            if( length > nn::fs::MountNameLengthMax )
            {
                return false;
            }
        }

        // UTF-8 エンコーディングである。
        if( !util::VerifyUtf8String(name, length) )
        {
            return false;
        }

        return true;
    }

    bool IsUsedReservedMountName(const char* name) NN_NOEXCEPT
    {
        return name[0] == ReservedMountNameCharForSystem;
    }

    const char* GetSubPath(const char* path) NN_NOEXCEPT
    {
        if (IsWindowsDrive(path))
        {
            return path;
        }

        const char* subPath = path;
        while (*subPath != ':')
        {
            subPath++;
        }

        // マウント名に続く文字列は ":/" または ":\" である必要がある
        NN_FS_ABORT_UNLESS_WITH_RESULT(
            *(subPath + 1) == '/' || *(subPath + 1) == '\\', ResultInvalidMountName(),
            "\":/\" or \":\\\" is required after a mount name: \"%s\"", path);

        return subPath + 1;
    }

    Result FindFileSystem(detail::FileSystemAccessor** outValue, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(path != nullptr, ResultInvalidPath());

        NN_RESULT_THROW_UNLESS(
            strncmp(
                path,
                HostRootFileSystemMountName,
                strnlen(HostRootFileSystemMountName, MountName::LENGTH)) != 0,
            nn::fs::ResultInvalidPathFormat());

        MountName mountName = GetMountName(path);
        return detail::Find(outValue, mountName.string);
    }

    Result CheckMountName(const char* name) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(name != nullptr, ResultInvalidMountName());
        NN_RESULT_THROW_UNLESS(!detail::IsUsedReservedMountName(name), ResultInvalidMountName());
        NN_RESULT_THROW_UNLESS(detail::IsValidMountName(name), ResultInvalidMountName());
        NN_RESULT_SUCCESS;
    }

    Result CheckMountNameAcceptingReservedMountName(const char* name) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(name != nullptr, ResultInvalidMountName());
        NN_RESULT_THROW_UNLESS(detail::IsValidMountName(name), ResultInvalidMountName());
        NN_RESULT_SUCCESS;
    }

}}}

namespace nn { namespace fs {
    // fs_Mount.h : Unmount
    void Unmount(const char* name) NN_NOEXCEPT
    {
        auto unmountImpl = [name]() NN_NOEXCEPT -> Result
        {
            detail::FileSystemAccessor* pFileSystemAccessor;
            NN_RESULT_DO(detail::Find(&pFileSystemAccessor, name));

            if (pFileSystemAccessor->IsFileDataCacheAttachable())
            {
                detail::GlobalFileDataCacheAccessorReadableScopedPointer fileDataCacheAccessor;
                if (detail::TryGetGlobalFileDataCacheAccessor(&fileDataCacheAccessor))
                {
                    pFileSystemAccessor->PurgeFileDataCache(fileDataCacheAccessor.Get());
                }
            }

            detail::Unregister(name);
            NN_RESULT_SUCCESS;
        };
        NN_FS_ABORT_UNLESS_RESULT_SUCCESS(
            NN_DETAIL_FS_ACCESS_LOG_UNMOUNT(unmountImpl(), name, NN_DETAIL_FS_ACCESS_LOG_FORMAT_MOUNT, name));
    }

    Result ConvertToFsCommonPath(char* outPathBuffer, size_t pathBufferSize, const char* path) NN_NOEXCEPT
    {
        MountName mountName = detail::GetMountName(path);
        detail::FileSystemAccessor* fs;
        NN_FS_RESULT_DO(detail::Find(&fs, mountName.string));

        NN_FS_RESULT_DO(fs->GetCommonMountName(outPathBuffer, pathBufferSize));

        auto mountNameLength = strnlen(outPathBuffer, pathBufferSize);

        const auto commonPathLength = nn::util::SNPrintf(outPathBuffer + mountNameLength, pathBufferSize - mountNameLength, "%s", detail::GetSubPath(path));

        if( static_cast<size_t>(commonPathLength) < pathBufferSize - mountNameLength )
        {
            NN_RESULT_SUCCESS;
        }
        else
        {
            return nn::fs::ResultTooLongPath();
        }
    }

}}
