﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_FileSystemForDebug.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fssrv/sf/fssrv_IFileSystem.h>
#include "fs_IntrusiveList.h"
#include "fs_FileAccessor.h"
#include "fs_DirectoryAccessor.h"
#include "fs_MountName.h"

namespace nn { namespace fs {

    namespace fsa {
        enum class QueryId;
        class IFileSystem;
    }

namespace detail {

    class FileAccessor;
    class DirectoryAccessor;
    class FileDataCacheAccessor;

    class FileSystemAccessor : public Newable
    {
        NN_DISALLOW_COPY(FileSystemAccessor);

        friend class FileAccessor;
        friend class DirectoryAccessor;

    public:
        FileSystemAccessor(const char* name, std::unique_ptr<fsa::IFileSystem>&& fileSystem, std::unique_ptr<fsa::ICommonMountNameGenerator>&& mountName = nullptr) NN_NOEXCEPT;
        virtual ~FileSystemAccessor() NN_NOEXCEPT;
        Result CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT;
        Result DeleteFile(const char* path) NN_NOEXCEPT;
        Result CreateDirectory(const char* path) NN_NOEXCEPT;
        Result DeleteDirectory(const char* path) NN_NOEXCEPT;
        Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT;
        Result CleanDirectoryRecursively(const char* path) NN_NOEXCEPT;
        Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT;
        Result RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT;
        Result GetEntryType(DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT;
        Result GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT;
        Result GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT;
        Result OpenFile(std::unique_ptr<FileAccessor>* outValue, const char* path, OpenMode mode) NN_NOEXCEPT;
        Result OpenDirectory(std::unique_ptr<DirectoryAccessor>* outValue, const char* path, OpenDirectoryMode mode) NN_NOEXCEPT;
        Result Commit() NN_NOEXCEPT;
        Result GetFileTimeStampRaw(FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT;
        Result QueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, fsa::QueryId queryId, const char* path) NN_NOEXCEPT;

        const char* GetName() const NN_NOEXCEPT
        {
            return m_Name.string;
        }

        Result GetCommonMountName(char* name, size_t nameSize) const NN_NOEXCEPT;

        IntrusiveListNode m_ListNode;

        void SetAccessLog(bool isEnabled) NN_NOEXCEPT
        {
            m_IsEnabledAccessLog = isEnabled;
        }

        bool IsEnabledAccessLog() const NN_NOEXCEPT
        {
            return m_IsEnabledAccessLog;
        }

        bool IsFileDataCacheAttachable() const NN_NOEXCEPT
        {
            return m_IsFileDataCacheAttachable;
        }

        void SetFileDataCacheAttachable(bool flag) NN_NOEXCEPT
        {
            m_IsFileDataCacheAttachable = flag;
        }

        bool IsPathBasedFileDataCacheAttachable() const NN_NOEXCEPT
        {
            return m_IsPathBasedFileDataCacheAttachable;
        }

        void SetPathBasedFileDataCacheAttachable(bool flag) NN_NOEXCEPT
        {
            m_IsPathBasedFileDataCacheAttachable = flag;
        }

        void SetMultiCommitSupported(bool isMultiCommitSupported) NN_NOEXCEPT
        {
            m_IsMultiCommitSupported = isMultiCommitSupported;
        }

        nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> GetMultiCommitTarget() NN_NOEXCEPT;

        void PurgeFileDataCache(FileDataCacheAccessor* pFileDataCacheAccessor) NN_NOEXCEPT;

        void AttachPathBasedFileDataCache() NN_NOEXCEPT
        {
            if (!m_IsPathBasedFileDataCacheAttachable)
            {
                return;
            }
            m_IsAttachedPathBasedFileDataCache = true;
        }

        void DetachPathBasedFileDataCache() NN_NOEXCEPT
        {
            m_IsAttachedPathBasedFileDataCache = false;
        }

    private:
        void NotifyCloseFile(FileAccessor* file) NN_NOEXCEPT;
        void NotifyCloseDirectory(DirectoryAccessor* directory) NN_NOEXCEPT;

        void DumpUnclosedAccessorList(int fileOpenMode, int directoryOpenMode) NN_NOEXCEPT;

        MountName m_Name;
        std::unique_ptr<fsa::IFileSystem> m_Impl;
        typedef IntrusiveList < FileAccessor, IntrusiveListMemberNodeTraits<FileAccessor, &FileAccessor::m_ListNode>> FileListType;
        typedef IntrusiveList < DirectoryAccessor, IntrusiveListMemberNodeTraits<DirectoryAccessor, &DirectoryAccessor::m_ListNode>> DirectoryListType;
        FileListType m_OpenFileList;
        DirectoryListType m_OpenDirectoryList;
        os::Mutex m_OpenListLock;
        std::unique_ptr<fsa::ICommonMountNameGenerator> m_pMountNameGenerator;
        bool m_IsEnabledAccessLog;
        bool m_IsFileDataCacheAttachable;
        bool m_IsPathBasedFileDataCacheAttachable;
        bool m_IsAttachedPathBasedFileDataCache;
        bool m_IsMultiCommitSupported;
    };

}}}
