﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <mutex>
#include <nn/os.h>
#include <nn/nn_SdkLog.h>

#include "eth_Interface.h"
#include "eth_InterfaceSfObject.h"
#include "device/eth_DeviceMain.h"

namespace nn     {
namespace eth    {
namespace server {
namespace        {

uint32_t      g_InterfaceCount = 0;
nn::os::Mutex g_InterfaceListLock(false);
nn::util::IntrusiveList<Interface, nn::util::IntrusiveListBaseNodeTraits<Interface>> g_InterfaceList;

}

void Interface::Signal(nn::Result result)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ClientListLock);

    for (auto client  = m_ClientList.begin();
              client != m_ClientList.end();
              client++)
    {
        client->Signal(result);
    }
}

void Interface::AttachClient(InterfaceSfObject* pClient)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ClientListLock);
    m_ClientList.push_front(*pClient);
}

void Interface::DetachClient(InterfaceSfObject* pClient)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_ClientListLock);
    m_ClientList.erase(m_ClientList.iterator_to(*pClient));
}

nn::Result ListInterfaces(InterfaceList* pInterfaceListOut)
NN_NOEXCEPT
{
    int count = 0;
    std::memset(pInterfaceListOut, 0x0, sizeof(InterfaceList));

    std::lock_guard<nn::os::Mutex> lock(g_InterfaceListLock);

    for (auto interface  = g_InterfaceList.begin();
              interface != g_InterfaceList.end();
              interface++)
    {
        interface->GetInterfaceInfo(&(pInterfaceListOut->adapterInfo[count]));

        if (++count >= MaxInterfaceCount)
        {
            break;
        }
    }

    pInterfaceListOut->adapterCount = count;

    return ResultSuccess();
}

nn::Result InterfaceCount(uint32_t* pInterfaceCountOut)
NN_NOEXCEPT
{
    *pInterfaceCountOut = nn::eth::device::GetInterfaceCount();
    return ResultSuccess();
}

nn::Result AttachToInterface(Interface** pInterfaceOut, const char* pName, InterfaceSfObject* pClient)
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InterfaceListLock);

    for (auto interface  = g_InterfaceList.begin();
              interface != g_InterfaceList.end();
              interface++)
    {
        if (std::strncmp(interface->DeviceName(), pName, MaxInterfaceNameLength) == 0)
        {
            interface->AttachClient(pClient);
            *pInterfaceOut = &(*interface);
            return ResultSuccess();
        }
    }

    return nn::eth::ResultAdapterNotFound();
}

nn::Result Interface::Initialize()
NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(g_InterfaceListLock);

    g_InterfaceList.push_front(*this);
    g_InterfaceCount++;

    SignalGroup(nn::eth::ResultInterfaceGroupChange());

    return ResultSuccess();
}

nn::Result Interface::Finalize()
NN_NOEXCEPT
{
    g_InterfaceListLock.Lock();
    g_InterfaceList.erase(g_InterfaceList.iterator_to(*this));
    g_InterfaceCount--;
    g_InterfaceListLock.Unlock();

    std::lock_guard<nn::os::Mutex> clientLock(m_ClientListLock);

    for (auto client  = m_ClientList.begin();
              client != m_ClientList.end();
              client++)
    {
        client->Detach();
    }

    m_ClientList.clear();

    SignalGroup(nn::eth::ResultInterfaceGroupChange());

    return ResultSuccess();
}

Interface::Interface()
NN_NOEXCEPT :
    m_ClientListLock(false)
{

}

Interface::~Interface()
NN_NOEXCEPT
{

}

}}}

